package com.j256.ormlite.db;

import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.sql.Blob;
import java.sql.SQLException;
import java.sql.Types;
import java.util.List;

import javax.sql.rowset.serial.SerialBlob;

import com.j256.ormlite.field.FieldConverter;
import com.j256.ormlite.field.FieldType;
import com.j256.ormlite.misc.SqlExceptionUtil;
import com.j256.ormlite.support.Results;

/**
 * Derby database type information used to create the tables, etc.. This is for an embedded Derby databse. For client
 * connections to a remote Derby server, you should use {@link DerbyClientServerDatabaseType}.
 * 
 * @author graywatson
 */
public class DerbyEmbeddedDatabaseType extends BaseDatabaseType implements DatabaseType {

	private final static String DATABASE_URL_PORTION = "derby";
	private final static String DRIVER_CLASS_NAME = "org.apache.derby.jdbc.EmbeddedDriver";
	private final static FieldConverter objectConverter = new ObjectFieldConverter();

	public String getDriverUrlPart() {
		return DATABASE_URL_PORTION;
	}

	public String getDriverClassName() {
		return DRIVER_CLASS_NAME;
	}

	@Override
	public FieldConverter getFieldConverter(FieldType fieldType) {
		// we are only overriding certain types
		switch (fieldType.getJdbcType()) {
			case BOOLEAN :
			case BOOLEAN_OBJ :
				return booleanConverter;
			case SERIALIZABLE :
				return objectConverter;
			default :
				return super.getFieldConverter(fieldType);
		}
	}

	@Override
	protected void appendBooleanType(StringBuilder sb) {
		// I tried "char for bit data" and "char(1)" with no luck
		sb.append("SMALLINT");
	}

	@Override
	protected void appendByteType(StringBuilder sb) {
		sb.append("SMALLINT");
	}

	@Override
	protected void appendObjectType(StringBuilder sb) {
		sb.append("BLOB");
	}

	@Override
	protected void configureGeneratedId(StringBuilder sb, FieldType fieldType, List<String> statementsBefore,
			List<String> additionalArgs, List<String> queriesAfter) {
		sb.append("GENERATED BY DEFAULT AS IDENTITY ");
		configureId(sb, fieldType, statementsBefore, additionalArgs, queriesAfter);
	}

	@Override
	public void appendEscapedEntityName(StringBuilder sb, String word) {
		sb.append('\"').append(word).append('\"');
	}

	@Override
	public boolean isLimitSqlSupported() {
		return false;
	}

	/**
	 * Conversion from the Object Java field to the BLOB Jdbc type because the varbinary needs a size otherwise.
	 */
	private static class ObjectFieldConverter implements FieldConverter {
		public int getJdbcTypeVal() {
			// store it as a short
			return Types.BLOB;
		}
		public Object javaToArg(Object javaObject) throws SQLException {
			ByteArrayOutputStream outStream = new ByteArrayOutputStream();
			try {
				ObjectOutputStream objOutStream = new ObjectOutputStream(outStream);
				objOutStream.writeObject(javaObject);
			} catch (Exception e) {
				throw SqlExceptionUtil.create("Could not write serialized object to output stream", e);
			}
			return new SerialBlob(outStream.toByteArray());
		}
		public Object parseDefaultString(String defaultStr) throws SQLException {
			throw new SQLException("Default values for serializable types are not supported");
		}
		public Object resultToJava(FieldType fieldType, Results results, int columnPos) throws SQLException {
			Blob blob = results.getBlob(columnPos);
			if (blob == null) {
				return null;
			}
			try {
				ObjectInputStream objInStream = new ObjectInputStream(blob.getBinaryStream());
				return objInStream.readObject();
			} catch (Exception e) {
				throw SqlExceptionUtil.create("Could not read serialized object from result blob", e);
			}
		}
		public boolean isStreamType() {
			return true;
		}
	}

	@Override
	public boolean isEntityNamesMustBeUpCase() {
		return true;
	}
}
