package com.j256.ormlite.db;

import java.util.List;

import com.j256.ormlite.field.FieldType;
import com.j256.ormlite.field.JdbcType;

/**
 * HyberSQL database type information used to create the tables, etc..
 * 
 * @author graywatson
 */
public class HsqldbDatabaseType extends BaseDatabaseType implements DatabaseType {

	private final static String DATABASE_URL_PORTION = "hsqldb";
	private final static String DRIVER_CLASS_NAME = "org.hsqldb.jdbcDriver";

	public String getDriverUrlPart() {
		return DATABASE_URL_PORTION;
	}

	public String getDriverClassName() {
		return DRIVER_CLASS_NAME;
	}

	@Override
	protected void appendBooleanType(StringBuilder sb) {
		sb.append("BIT");
	}

	@Override
	protected void appendObjectType(StringBuilder sb) {
		sb.append("BINARY");
	}

	@Override
	protected void configureGeneratedIdSequence(StringBuilder sb, FieldType fieldType, List<String> statementsBefore,
			List<String> additionalArgs, List<String> queriesAfter) {
		// needs to match dropColumnArg()
		StringBuilder seqSb = new StringBuilder();
		seqSb.append("CREATE SEQUENCE ");
		appendEscapedEntityName(seqSb, fieldType.getGeneratedIdSequence());
		if (fieldType.getJdbcType() == JdbcType.LONG) {
			seqSb.append(" AS BIGINT");
		} else {
			// integer is the default
		}
		// with hsqldb (as opposed to all else) the sequences start at 0, grumble
		seqSb.append(" START WITH 1");
		statementsBefore.add(seqSb.toString());
		sb.append("GENERATED BY DEFAULT AS IDENTITY ");
		configureId(sb, fieldType, statementsBefore, additionalArgs, queriesAfter);
	}

	@Override
	public void appendEscapedEntityName(StringBuilder sb, String word) {
		sb.append('\"').append(word).append('\"');
	}

	@Override
	public void dropColumnArg(FieldType fieldType, List<String> statementsBefore, List<String> statementsAfter) {
		if (fieldType.isGeneratedIdSequence()) {
			StringBuilder sb = new StringBuilder();
			sb.append("DROP SEQUENCE ");
			appendEscapedEntityName(sb, fieldType.getGeneratedIdSequence());
			statementsAfter.add(sb.toString());
		}
	}

	@Override
	public boolean isIdSequenceNeeded() {
		return true;
	}

	@Override
	public boolean isVarcharFieldWidthSupported() {
		return false;
	}

	@Override
	public boolean isLimitAfterSelect() {
		return true;
	}

	@Override
	public void appendLimitValue(StringBuilder sb, int limit) {
		// the 0 is the offset, could also use TOP X
		sb.append("LIMIT 0 ").append(limit).append(' ');
	}

	@Override
	public void appendSelectNextValFromSequence(StringBuilder sb, String sequenceName) {
		sb.append("CALL NEXT VALUE FOR ");
		appendEscapedEntityName(sb, sequenceName);
	}

	@Override
	public boolean isEntityNamesMustBeUpCase() {
		return true;
	}
}
