/*
 * Copyright (c) 2015, Jared Rummler
 * Copyright (c) 2015, Liu Dong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the <organization> nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jaredrummler.apkparser.struct;

/**
 * A Chunk is just a piece of memory split into two parts, a header and a body.
 * The exact structure of the header and the body of a given Chunk is determined by its type.
 * <pre>
 * chunk header struct.
 * struct ResChunk_header {
 * uint16_t type;
 * uint16_t headerSize;
 * uint32_t size;
 * }
 * </pre>
 */
public class ChunkHeader {

  // Type identifier for this chunk.  The meaning of this value depends
  // on the containing chunk.
  private int chunkType;

  // Size of the chunk header (in bytes).  Adding this value to
  // the address of the chunk allows you to find its associated data
  // (if any).
  private int headerSize;

  // Total size of this chunk (in bytes).  This is the chunkSize plus
  // the size of any data associated with the chunk.  Adding this value
  // to the chunk allows you to completely skip its contents (including
  // any child chunks).  If this value is the same as chunkSize, there is
  // no data associated with the chunk.
  private long chunkSize;

  public ChunkHeader(int chunkType, int headerSize, long chunkSize) {
    this.chunkType = chunkType;
    this.headerSize = headerSize;
    this.chunkSize = chunkSize;
  }

  public int getBodySize() {
    return (int) (this.chunkSize - this.headerSize);
  }

  public int getChunkType() {
    return chunkType;
  }

  public void setChunkType(int chunkType) {
    this.chunkType = chunkType;
  }

  public int getHeaderSize() {
    return headerSize;
  }

  public void setHeaderSize(int headerSize) {
    this.headerSize = headerSize;
  }

  public long getChunkSize() {
    return chunkSize;
  }

  public void setChunkSize(long chunkSize) {
    this.chunkSize = chunkSize;
  }
}
