#  Core module.
# Holds the core application logic and all interactions with JSREPL.
# Emits events so other modules can hook into.
$ = jQuery

$.extend REPLIT,
  Init: ->
    @jsrepl = new JSREPL {
      input: $.proxy @InputCallback, @
      output: $.proxy @OutputCallback, @
      result: $.proxy @ResultCallback, @
      error: $.proxy @ErrorCallback, @
      progress: $.proxy @OnProgress, @
      outputHtml: $.proxy @OnOutputHtml, @
      tutResult: $.proxy @OnTutResult, @
      aceInjection: $.proxy @OnAceInjection, @
      printJson: $.proxy @OnPrintJson, @
      printWire: $.proxy @OnPrintWire, @
      startNav: $.proxy @OnStartNav, @
      timeout:
        time: 12000,
        callback: =>
          if a = confirm 'The program is taking too long to finish. Do you want to stop it?'
            code = @editor.getSession().getValue()
            @LoadLanguage @current_lang.system_name , =>
              @editor.getSession().setValue code
          return a
    }

    # Init console.
    @jqconsole = @$consoleContainer.jqconsole '', '   ', '.. '
    @$console = @$consoleContainer.find '.jqconsole'

    # Init editor.
    @$editor = @$editorContainer.find '#editor-widget'
    if not @ISMOBILE
      @editor = ace.edit 'editor-widget'
      @editor.setTheme 'ace/theme/textmate'
      @editor.setShowPrintMargin false
      @editor.renderer.setHScrollBarAlwaysVisible false
      ace.config.loadModule 'ace/ext/language_tools', ->
        REPLIT.editor.setOptions
          enableBasicAutocompletion:true,
          enableSnippets:true

        snippetManager = ace.require("ace/snippets").snippetManager
        config = ace.require "ace/config"

        ace.config.loadModule "ace/snippets/javascript", (m) ->
          if m
            snippetManager.files.javascript = m
            m.snippets = snippetManager.parseSnippetFile m.snippetText
            m.snippets = REPLIT.setSnippets(m)
            snippetManager.register(m.snippets, m.scope);

      @$run.click =>
        # TODO(max99x): Expose state properly from jqconsole.
        if @jqconsole.state is 2  # STATE_PROMPT
          @jqconsole.AbortPrompt()
          @Evaluate REPLIT.editor.getSession().getValue()
          history = @jqconsole.GetHistory()
          history.push REPLIT.editor.getSession().getValue()
          @jqconsole.SetHistory history
          REPLIT.aceInConsole REPLIT.editor.getSession().getValue()

      @$switch.click =>
        position = REPLIT.editor.getCursorPosition()
        REPLIT.editor.getSession().insert position, @jqconsole.GetPromptText()
        @jqconsole.ClearPromptText()
        REPLIT.editor.focus()

      @$loadButton.on 'change', (e) =>
        file = e.originalEvent.srcElement.files[0]
        @putInAce file
        input = $('#button-load')
        input.replaceWith(input = input.clone(true))
        $('#exitSave').click()

      @$saveButton.click =>
        content = REPLIT.editor.getSession().getValue()
        if(!content)
          return
        blob = new Blob([content], type:'text/plain')
        date = new Date()
        saveAs blob, 'session(' + date.getTime() + ").txt"
        $('#exitSave').click()

      @$saveOptions.click =>
        regex = new RegExp '^[a-zA-Z1-9]+$'
        accessKey = @$accessKeyInput.val()
        if regex.test accessKey
          @jsrepl.setUserName accessKey
          $('#exitOptions').click()
        else
          $invalidOptionsAlert = $ """
            <div class="alert alert-error alert-icon" role="alert">
              <a class="close" data-dismiss="alert" href="#">×</a>
              <h3>Invalid!</h3>
              <p>The access key you put in '<strong>#{accessKey}</strong>' is invalid. Please put in a single alpha numeric word without any spaces</p>
            </div>
          """
        @$optionsBody.prepend $invalidOptionsAlert

      @$editorContainer.on 'dragover', (e) =>
        @stopEvent e
      @$editorContainer.on 'drop', (e) =>
        @stopEvent e
        @putInAce e.originalEvent.dataTransfer.files[0]

      @editor.commands.addCommand
        name: 'run'
        bindKey:
          win: 'Ctrl-Return'
          mac: 'Command-Return'
          sebder: 'editor'
        exec: =>
          @$run.click()
          # Allow async eval to happen then reclaim focus to editor.
          setTimeout (=> @editor.focus()), 0

      @editor.commands.addCommand
        name: 'save'
        bindKey:
          win: 'Ctrl-S'
          mac: 'Command-S'
          sebder: 'editor'
        exec: =>
          @$saveButton.click()

    @current_lang = null
    @current_lang_name = null
    @inited = true

  #Save / Load Logic
  stopEvent: (e)->
    e.preventDefault()
    e.stopPropagation()
  #puts content in the ace editor
  putInAce: (file) ->
    reader = new FileReader()
    reader.onload = (file)->
      text = file.target.result
      REPLIT.editor.getSession().setValue text
    reader.readAsText(file)

  # Load a given language by name.
  LoadLanguage: (lang_name, callback=$.noop) ->
    @$this.trigger 'language_loading', [lang_name]
    @current_lang = @jsrepl.getLangConfig lang_name.toLowerCase()
    # Hold the name for saving and such.
    @current_lang.system_name = lang_name

    #Load Ace mode.
    if not @ISMOBILE

      EditSession = ace.require("ace/edit_session").EditSession
      UndoManager = ace.require("ace/undomanager").UndoManager

      session = new EditSession ''
      session.setUndoManager new UndoManager
      ace_mode = @Languages[lang_name.toLowerCase()].ace_mode
      if ace_mode?
        # jQuery deferred object for getting ace mode.
        ace_mode_ajax = $.getScript ace_mode.script, =>
          mode = ace.require(ace_mode.module).Mode
          session.setMode new mode
          session.setUseWrapMode true
          @editor.setSession session
          callback()
      else
        # No ace mode found create a resolved deferred.
        ace_mode_ajax = jQuery.Deferred().resolve()
        textMode = ace.require("ace/mode/text").Mode
        session.setMode new textMode
        @editor.setSession session

    # Empty out the history and prompt.
    @jqconsole.Reset()
    # Register character matchings in jqconsole for the current language.
    for [open, close], index in @current_lang.matchings
      @jqconsole.RegisterMatching open, close, 'matching-' + index

    # Register shortcuts.
    @jqconsole.RegisterShortcut 'Z', =>
      @jqconsole.AbortPrompt()
      @StartPrompt()
    @jqconsole.RegisterShortcut 'G', =>
      @OpenPage 'examples'
    @jqconsole.RegisterShortcut 'H', =>
      @OpenPage 'help'
    @jqconsole.RegisterShortcut 'S', =>
      @$saveButton.click()
    @jqconsole.RegisterShortcut 'A', =>
      @jqconsole.MoveToStart();
    @jqconsole.RegisterShortcut 'E', =>
      @jqconsole.MoveToEnd();
    @jqconsole.RegisterShortcut 'K', =>
      @jqconsole.Clear();
    # Load the language engine from jsREPL.
    @jsrepl.loadLanguage lang_name.toLowerCase(), =>
      # Continue only when the ace mode retrieval is done.
      $.when(ace_mode_ajax).then =>
        @StartPrompt()
        @jqconsole.Write @Languages[lang_name.toLowerCase()].header + '\n', "", ""
        @$this.trigger 'language_loaded', [lang_name]
        callback()

  # Receives the result of a command evaluation.
  #   @arg result: The user-readable string form of the result of an evaluation.
  ResultCallback: (result) ->
    if result
      if result == '[Object]'
        result = ''
      result = result + '\n' if result[-1] isnt '\n'
      @jqconsole.Write '=> ' + result, 'result'
    @StartPrompt()
    @$this.trigger 'result', [result]
  # Receives an error message resulting from a command evaluation.
  #   @arg error: A message describing the error.
  ErrorCallback: (error) ->
    if typeof error == 'object'
      error = error.message
    error = error + '\n' if error[-1] isnt '\n'
    @jqconsole.Write String(error), 'error'
    @StartPrompt()
    @$this.trigger 'error', [error]
  # Receives any output from a language engine. Acts as a low-level output
  # stream or port.
  #   @arg output: The string to output. May contain control characters.
  #   @arg cls: An optional class for styling the output.
  OutputCallback: (output, cls) ->
    if output
      @jqconsole.Write output, cls, ""
      @$this.trigger 'output', [output]
      return undefined
  # Receives a request for a string input from a language engine. Passes back
  # the user's response asynchronously.
  #   @arg callback: The function called with the string containing the user's
  #     response.
  InputCallback: (callback) ->
    @jqconsole.Input (result) =>
      try
        callback result
        @$this.trigger 'input', [result]
      catch e
        @ErrorCallback e
    @$this.trigger 'input_request', [callback]
    return undefined

  #Injects text into the ACE editor
  OnAceInjection: (content) ->
    if content
      @editor.setValue content
      return undefined
  OnOutputHtml: (content) ->
    if content
      content += "</br>"
      @jqconsole.Write content, "", ""
      @$this.trigger 'output', [content]
      return undefined
  OnTutResult: (content) ->
    if(content)
      @$this.trigger 'output', [content]
      return undefined
    @$this.trigger 'output', [content]
  Evaluate: (command) ->
    if command
      @jsrepl.eval command
      @$this.trigger 'eval', [command]
    else
      @StartPrompt()
  OnPrintJson: (content) ->
    $html = jsonPrettyPrint content , false;
    @jqconsole.Write $html[0], "", ""
    @$this.trigger 'output', [html]

  OnPrintWire: (content) ->
    $html = jsonPrettyPrint content , true;
    @jqconsole.Write $html[0], "", ""
    @$this.trigger 'output', [html]

  OnStartNav: (numberOfLessons) ->
    REPLIT.openNav()
    REPLIT.numOfSteps = numberOfLessons

  # Shows a command prompt in the console and waits for input.
  StartPrompt: ->
    @jqconsole.Prompt true, $.proxy(@Evaluate, @), @jsrepl.checkLineEnd, true

$ ->
  REPLIT.Init()
