/**
 * Copyright (c) 2013-2015, xembly.org
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met: 1) Redistributions of source code must retain the above
 * copyright notice, this list of conditions and the following
 * disclaimer. 2) Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following
 * disclaimer in the documentation and/or other materials provided
 * with the distribution. 3) Neither the name of the xembly.org nor
 * the names of its contributors may be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.xembly;

import java.util.Deque;
import java.util.LinkedList;
import java.util.NoSuchElementException;
import lombok.EqualsAndHashCode;

/**
 * Stack of DOM cursors.
 *
 * <p>The class is mutable and thread-safe.
 *
 * @author Yegor Bugayenko (yegor@teamed.io)
 * @version $Id: 29ed70c2b306855c94f51345d783cb84275bc66c $
 * @since 0.16
 */
@EqualsAndHashCode(of = "cursors")
final class DomStack implements Directive.Stack {

    /**
     * Queue of cursors.
     */
    private final transient Deque<Directive.Cursor> cursors =
        new LinkedList<Directive.Cursor>();

    @Override
    public void push(final Directive.Cursor cursor) {
        synchronized (this.cursors) {
            this.cursors.push(cursor);
        }
    }

    @Override
    public Directive.Cursor pop() throws ImpossibleModificationException {
        try {
            synchronized (this.cursors) {
                return this.cursors.pop();
            }
        } catch (final NoSuchElementException ex) {
            throw new ImpossibleModificationException(
                "stack is empty, can't POP", ex
            );
        }
    }
}
