/**
 * Copyright (c) 2013-2022, jcabi.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met: 1) Redistributions of source code must retain the above
 * copyright notice, this list of conditions and the following
 * disclaimer. 2) Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following
 * disclaimer in the documentation and/or other materials provided
 * with the distribution. 3) Neither the name of the jcabi.com nor
 * the names of its contributors may be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.jcabi.github.mock;

import com.google.common.collect.ImmutableList;
import com.jcabi.aspects.Immutable;
import com.jcabi.aspects.Loggable;
import com.jcabi.github.CommitsComparison;
import com.jcabi.github.Coordinates;
import com.jcabi.github.FileChange;
import com.jcabi.github.Repo;
import javax.json.Json;
import javax.json.JsonObject;
import lombok.ToString;

/**
 * Mock commits' comparison of a Github repository.
 * @author Andrej Istomin (andrej.istomin.ikeen@gmail.com)
 * @version $Id: 60c4c9c050e3af044aeb92cbd0e745b3534001fd $
 */
@Immutable
@Loggable(Loggable.DEBUG)
@ToString
final class MkCommitsComparison implements CommitsComparison {
    /**
     * File change JSON object.
     */
    private static final JsonObject FILE_JSON = Json.createObjectBuilder()
        .add("sha", "bbcd538c8e72b8c175046e27cc8f907076331401")
        .add("filename", "test-file")
        // @checkstyle MultipleStringLiterals (1 lines)
        .add("status", "modified")
        .build();

    /**
     * Storage.
     */
    private final transient MkStorage storage;

    /**
     * Login of the user logged in.
     */
    private final transient String self;

    /**
     * Repo coordinates.
     */
    private final transient Coordinates coords;

    /**
     * Public ctor.
     * @param stg Storage
     * @param login User to login
     * @param repo Repository coordinates
     */
    MkCommitsComparison(
        final MkStorage stg,
        final String login,
        final Coordinates repo
    ) {
        this.storage = stg;
        this.self = login;
        this.coords = repo;
    }

    @Override
    public Repo repo() {
        return new MkRepo(this.storage, this.self, this.coords);
    }

    @Override
    public JsonObject json() {
        return Json.createObjectBuilder()
            // @checkstyle MultipleStringLiterals (3 lines)
            .add("status", "test-status")
            .add("ahead_by", 1)
            .add("behind_by", 2)
            .add(
                "author",
                Json.createObjectBuilder()
                    // @checkstyle MultipleStringLiterals (3 lines)
                    .add("login", "test")
                    .build()
            )
            .add(
                "files",
                Json.createArrayBuilder()
                    .add(MkCommitsComparison.FILE_JSON)
                    .build()
            )
            .add("commits", Json.createArrayBuilder().build())
            .build();
    }

    @Override
    public Iterable<FileChange> files() {
        return ImmutableList.<FileChange>of(
            new MkFileChange(MkCommitsComparison.FILE_JSON)
        );
    }
}
