/**
 * Copyright (c) 2012-2017, jcabi.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met: 1) Redistributions of source code must retain the above
 * copyright notice, this list of conditions and the following
 * disclaimer. 2) Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following
 * disclaimer in the documentation and/or other materials provided
 * with the distribution. 3) Neither the name of the jcabi.com nor
 * the names of its contributors may be used to endorse or promote
 * products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.jcabi.xml;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import javax.xml.transform.Source;
import javax.xml.transform.TransformerException;
import javax.xml.transform.stream.StreamSource;
import lombok.EqualsAndHashCode;

/**
 * Sources in file system.
 *
 * If you have some resources in files, just configure your
 * XSL with this "sources":
 *
 * <pre> XSL xsl = new XSLDocument(input).with(
 *   new FileSources("/tmp/my-resources")
 * );</pre>
 *
 * @author Yegor Bugayenko (yegor256@gmail.com)
 * @version $Id: 4651c16137ecca6d9728792e69fda7a9799e7c4b $
 * @since 0.18
 */
@EqualsAndHashCode(of = "path")
public final class FileSources implements Sources {

    /**
     * Directory.
     */
    private final transient File path;

    /**
     * Public ctor.
     */
    public FileSources() {
        this("");
    }

    /**
     * Public ctor.
     * @param dir Directory
     */
    public FileSources(final String dir) {
        this(new File(dir));
    }

    /**
     * Public ctor.
     * @param dir Directory
     */
    public FileSources(final File dir) {
        this.path = dir;
    }

    @Override
    public Source resolve(final String href, final String base)
        throws TransformerException {
        File file = new File(this.path, href);
        if (!file.exists()) {
            if (base == null) {
                file = new File(href);
            } else {
                file = new File(new File(base), href);
            }
            if (!file.exists()) {
                throw new TransformerException(
                    String.format(
                        "file \"%s\" not found in \"%s\" and in base \"%s\"",
                        href, this.path, base
                    )
                );
            }
        }
        try {
            return new StreamSource(new FileInputStream(file));
        } catch (final FileNotFoundException ex) {
            throw new TransformerException(ex);
        }
    }
}
