/*
 * Decompiled with CFR 0.152.
 */
package js.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.net.URL;
import java.net.URLConnection;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import js.io.ReaderInputStream;
import js.lang.BugError;
import js.lang.Predicate;
import js.log.Log;
import js.log.LogFactory;
import js.util.Classes;
import js.util.Params;
import js.util.Strings;

public class Files {
    private static Log log = LogFactory.getLog(Files.class);
    public static final String CURRENT_DIR = ".";
    public static final String LINE_SEPARATOR = System.getProperty("line.separator");
    private static final int BUFFER_SIZE = 4096;
    private static final String PARTIAL_FILE_SUFFIX = ".part";
    private static final String SEPARATOR_CHAR_REX = File.separatorChar == '\\' ? "\\\\" : "/";
    private static final Pattern DOT_PATTERN = Pattern.compile("\\.");
    private static final String URL_PATH_SEPARATOR = "/";
    public static final char EXTENSION_SEPARATOR = '.';
    private static final char UNIX_SEPARATOR = '/';
    private static final char WINDOWS_SEPARATOR = '\\';
    private static final String TMP_FILE_PREFIX = "jslib";
    private static final String TMP_FILE_EXTENSION = ".tmp";
    private static final List<String> IMAGE_FILE_EXTENSIONS = new ArrayList<String>();
    private static final Map<FileNotify, Predicate> INOTIFY_PREDICATES;

    protected Files() {
    }

    public static BufferedReader createBufferedReader(InputStream stream) {
        try {
            return new BufferedReader(new InputStreamReader(stream, "UTF-8"));
        }
        catch (UnsupportedEncodingException unused) {
            throw new BugError("Unsupported UTF-8 ecoding.", new Object[0]);
        }
    }

    public static BufferedWriter createBufferedWriter(OutputStream stream) {
        try {
            return new BufferedWriter(new OutputStreamWriter(stream, "UTF-8"));
        }
        catch (UnsupportedEncodingException unused) {
            throw new BugError("Unsupported UTF-8 ecoding.", new Object[0]);
        }
    }

    public static File copy(URL url) throws IOException, IllegalArgumentException {
        Params.notNull(url, "Source URL", new Object[0]);
        URLConnection connection = url.openConnection();
        connection.setDoInput(true);
        return Files.copy(connection.getInputStream());
    }

    public static File copy(InputStream stream) throws IOException {
        File file = Files.createTempFile();
        Files.copy(stream, (OutputStream)new FileOutputStream(file));
        return file;
    }

    public static File copy(URL url, String encoding) throws IOException, IllegalArgumentException {
        Params.notNull(url, "Source URL", new Object[0]);
        URLConnection connection = url.openConnection();
        connection.setDoInput(true);
        return Files.copy(connection.getInputStream(), encoding);
    }

    public static File copy(InputStream stream, String encoding) throws IOException {
        File file = Files.createTempFile();
        Files.copy(new InputStreamReader(stream, encoding), new OutputStreamWriter((OutputStream)new FileOutputStream(file), encoding));
        return file;
    }

    public static File copy(Reader reader) throws IOException {
        File file = Files.createTempFile();
        Files.copy(reader, new FileWriter(file));
        return file;
    }

    public static long copy(File source, File target) throws FileNotFoundException, IOException {
        return Files.copy((InputStream)new FileInputStream(source), (OutputStream)new FileOutputStream(target));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static int copy(Reader reader, Writer writer) throws IOException {
        Params.notNull(reader, "Reader", new Object[0]);
        Params.notNull(writer, "Writer", new Object[0]);
        if (!(reader instanceof BufferedReader)) {
            reader = new BufferedReader(reader);
        }
        if (!(writer instanceof BufferedWriter)) {
            writer = new BufferedWriter(writer);
        }
        int charsCount = 0;
        try {
            int readChars;
            char[] buffer = new char[4096];
            while ((readChars = reader.read(buffer)) != -1) {
                charsCount += readChars;
                writer.write(buffer, 0, readChars);
            }
        }
        finally {
            Files.close(reader);
            Files.close(writer);
        }
        return charsCount;
    }

    public static long copy(URL url, File baseDir, String fileName) throws FileNotFoundException, IOException, IllegalArgumentException {
        Params.notNull(url, "Source URL", new Object[0]);
        URLConnection connection = url.openConnection();
        connection.setDoInput(true);
        return Files.copy(connection.getInputStream(), (OutputStream)new FileOutputStream(new File(baseDir, fileName)));
    }

    public static long copy(InputStream inputStream, File baseDir, String fileName) throws FileNotFoundException, IOException {
        return Files.copy(inputStream, (OutputStream)new FileOutputStream(new File(baseDir, fileName)));
    }

    public static long copy(InputStream inputStream, File file) throws FileNotFoundException, IOException {
        return Files.copy(inputStream, (OutputStream)new FileOutputStream(file));
    }

    public static long copy(URL url, File file) throws FileNotFoundException, IOException {
        return Files.copy(url, (OutputStream)new FileOutputStream(file));
    }

    public static long copy(URL url, OutputStream outputStream) throws IOException, IllegalArgumentException {
        Params.notNull(url, "Source URL", new Object[0]);
        URLConnection connection = url.openConnection();
        connection.setDoInput(true);
        return Files.copy(connection.getInputStream(), outputStream);
    }

    public static long copy(File file, OutputStream outputStream) throws IOException {
        Params.notNull(file, "Input file", new Object[0]);
        return Files.copy((InputStream)new FileInputStream(file), outputStream);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static long copy(InputStream inputStream, OutputStream outputStream) throws IOException, IllegalArgumentException {
        Params.notNull(inputStream, "Input stream", new Object[0]);
        Params.notNull(outputStream, "Output stream", new Object[0]);
        Params.isFalse(inputStream instanceof ZipInputStream, "Input stream is ZIP.", new Object[0]);
        Params.isFalse(outputStream instanceof ZipOutputStream, "Output stream is ZIP.", new Object[0]);
        if (!(inputStream instanceof BufferedInputStream)) {
            inputStream = new BufferedInputStream(inputStream);
        }
        if (!(outputStream instanceof BufferedOutputStream)) {
            outputStream = new BufferedOutputStream(outputStream);
        }
        long bytes = 0L;
        try {
            int length;
            byte[] buffer = new byte[4096];
            while ((length = inputStream.read(buffer)) != -1) {
                bytes += (long)length;
                outputStream.write(buffer, 0, length);
            }
        }
        finally {
            Files.close(inputStream);
            Files.close(outputStream);
        }
        return bytes;
    }

    public static void download(URL url, File file) throws IOException {
        File partialFile = new File(file.getAbsolutePath() + PARTIAL_FILE_SUFFIX);
        try {
            Files.copy(url, partialFile);
            Files.renameTo(partialFile, file);
        }
        finally {
            partialFile.delete();
        }
    }

    public static void dump(InputStream inputStream) throws IOException {
        if (!(inputStream instanceof BufferedInputStream)) {
            inputStream = new BufferedInputStream(inputStream);
        }
        try {
            int length;
            byte[] buffer = new byte[4096];
            while ((length = inputStream.read(buffer)) != -1) {
                System.out.write(buffer, 0, length);
            }
        }
        finally {
            inputStream.close();
        }
    }

    public static void dump(Reader reader) throws IOException {
        Files.copy((InputStream)new ReaderInputStream(reader), (OutputStream)System.out);
    }

    public static void close(Closeable closeable) {
        if (closeable == null) {
            return;
        }
        try {
            closeable.close();
        }
        catch (IOException e) {
            log.error(e);
        }
        catch (IncompatibleClassChangeError error) {
            log.debug("Closeable interface not implemented. Invoke close method directly on instance |%s|.", closeable.getClass());
            try {
                Classes.invoke((Object)closeable, "close", new Object[0]);
            }
            catch (Throwable throwable) {
                log.error(throwable);
            }
        }
    }

    public static void renameTo(File source, File destination) throws IOException {
        Files.mkdirs(destination);
        destination.delete();
        if (!source.renameTo(destination)) {
            throw new IOException(String.format("Fail to rename file |%s| to |%s|.", source, destination));
        }
    }

    public static boolean move(String sourcePath, String targetPath) throws IllegalArgumentException {
        Params.notNull(sourcePath, "Source path", new Object[0]);
        Params.notNull(targetPath, "Target path", new Object[0]);
        File sourceFile = new File(sourcePath);
        if (!sourceFile.exists()) {
            log.error("Missing file to move |%s|.", sourcePath);
            return false;
        }
        File targetFile = new File(targetPath);
        if (targetFile.exists()) {
            log.debug("Target file exists |%s|. File moving aborted.", targetPath);
            return false;
        }
        return sourceFile.renameTo(targetFile);
    }

    public static String getRelativePath(File baseDir, File file, boolean ... forceURLPath) {
        int i;
        Params.notNull(baseDir, "Base directory", new Object[0]);
        Params.notNull(file, "File", new Object[0]);
        Params.notNullOrEmpty(baseDir, "Base directory");
        Params.notNullOrEmpty(file, "File");
        List<String> baseDirPathComponents = Files.getPathComponents(baseDir.getAbsoluteFile());
        if (baseDirPathComponents.size() > 0) {
            int lastIndex = baseDirPathComponents.size() - 1;
            if (baseDirPathComponents.get(lastIndex).equals(CURRENT_DIR)) {
                baseDirPathComponents.remove(lastIndex);
            } else if (baseDirPathComponents.get(lastIndex).equals("..")) {
                if (baseDirPathComponents.size() < 2) {
                    throw new BugError("Invalid base directory for relative path. It ends with '..' but has no parent directory.", new Object[0]);
                }
                baseDirPathComponents.remove(baseDirPathComponents.size() - 1);
                baseDirPathComponents.remove(baseDirPathComponents.size() - 1);
            }
        }
        List<String> filePathComponenets = Files.getPathComponents(file.getAbsoluteFile());
        ArrayList<String> relativePath = new ArrayList<String>();
        for (i = 0; i < baseDirPathComponents.size() && i != filePathComponenets.size() && baseDirPathComponents.get(i).equals(filePathComponenets.get(i)); ++i) {
        }
        for (int j = i; j < baseDirPathComponents.size(); ++j) {
            relativePath.add("..");
        }
        while (i < filePathComponenets.size()) {
            relativePath.add(filePathComponenets.get(i));
            ++i;
        }
        return Strings.join(relativePath, (char)(forceURLPath.length > 0 && forceURLPath[0] ? 47 : File.separatorChar));
    }

    public static List<String> getPathComponents(File file) {
        if (file == null) {
            return Collections.emptyList();
        }
        ArrayList<String> pathComponents = new ArrayList<String>();
        do {
            pathComponents.add(0, file.getName());
        } while ((file = file.getParentFile()) != null);
        return pathComponents;
    }

    public static String removeExtension(String path) throws IllegalArgumentException {
        Params.notNull(path, "Path", new Object[0]);
        int extensionIndex = path.lastIndexOf(46);
        return extensionIndex != -1 ? path.substring(0, extensionIndex) : path;
    }

    public static File removeExtension(File file) throws IllegalArgumentException {
        Params.notNull(file, "File", new Object[0]);
        return new File(Files.removeExtension(file.getPath()));
    }

    public static String replaceExtension(String path, String newExtension) throws IllegalArgumentException {
        int extensionDotIndex;
        Params.notNull(path, "Path", new Object[0]);
        Params.notNull(newExtension, "New extension", new Object[0]);
        if (newExtension.charAt(0) == '.') {
            newExtension = newExtension.substring(1);
        }
        if ((extensionDotIndex = path.lastIndexOf(46) + 1) == 0) {
            extensionDotIndex = path.length();
        }
        StringBuilder sb = new StringBuilder(path.length());
        sb.append(path.substring(0, extensionDotIndex));
        sb.append(newExtension);
        return sb.toString();
    }

    public static File replaceExtension(File file, String newExtension) throws IllegalArgumentException {
        Params.notNull(file, "File", new Object[0]);
        return new File(Files.replaceExtension(file.getPath(), newExtension));
    }

    public static String path2dot(String path) {
        return path != null ? path.replaceAll(SEPARATOR_CHAR_REX, CURRENT_DIR) : null;
    }

    public static String path2unix(File file) {
        return file != null ? Files.path2unix(file.getPath()) : null;
    }

    public static String path2unix(String path) {
        if (path == null) {
            return null;
        }
        if (path.endsWith(URL_PATH_SEPARATOR) || path.endsWith("\\")) {
            path = path.substring(0, path.length() - 1);
        }
        return path.replaceAll("(^[a-zA-Z]\\:\\\\?)?\\\\", URL_PATH_SEPARATOR);
    }

    public static String dot2path(String qualifiedName) {
        return qualifiedName != null ? DOT_PATTERN.matcher(qualifiedName).replaceAll(Matcher.quoteReplacement(File.separator)) : null;
    }

    public static String dot2urlpath(String qualifiedName) {
        return qualifiedName != null ? DOT_PATTERN.matcher(qualifiedName).replaceAll(Matcher.quoteReplacement(URL_PATH_SEPARATOR)) : null;
    }

    public static String dot2path(String qualifiedName, String fileExtension) {
        if (qualifiedName == null) {
            return null;
        }
        StringBuilder path = new StringBuilder();
        path.append(Files.dot2path(qualifiedName));
        if (fileExtension != null) {
            if (fileExtension.charAt(0) != '.') {
                path.append('.');
            }
            path.append(fileExtension);
        }
        return path.toString();
    }

    public static String getExtension(URL url) {
        return url != null ? Files.getExtension(url.getPath()) : null;
    }

    public static String getExtension(File file) {
        return file != null ? Files.getExtension(file.getAbsolutePath()) : null;
    }

    public static String getExtension(String path) {
        int lastWindowsPos;
        if (path == null) {
            return null;
        }
        int extensionPos = path.lastIndexOf(46);
        int lastUnixPos = path.lastIndexOf(47);
        int lastSeparatorPos = Math.max(lastUnixPos, lastWindowsPos = path.lastIndexOf(92));
        int i = lastSeparatorPos > extensionPos ? -1 : extensionPos;
        return i == -1 ? "" : path.substring(i + 1).toLowerCase();
    }

    public static String basename(File file) {
        if (file == null) {
            return null;
        }
        String fileName = file.getName();
        int i = fileName.lastIndexOf(46);
        return i != -1 ? fileName.substring(0, i) : fileName;
    }

    public static String basename(String path) {
        return path != null ? Files.basename(new File(path)) : null;
    }

    private static File createTempFile() throws IOException {
        File file = File.createTempFile(TMP_FILE_PREFIX, TMP_FILE_EXTENSION);
        file.deleteOnExit();
        return file;
    }

    public static boolean isImage(File file) {
        return file != null ? IMAGE_FILE_EXTENSIONS.contains(Files.getExtension(file)) : false;
    }

    public static boolean isImage(String extension) {
        return extension != null ? IMAGE_FILE_EXTENSIONS.contains(extension.toLowerCase()) : false;
    }

    public static byte[] getFileDigest(File file) throws FileNotFoundException, IOException {
        return Files.getFileDigest(new FileInputStream(file));
    }

    public static byte[] getFileDigest(InputStream inputStream) throws IOException {
        if (!(inputStream instanceof BufferedInputStream)) {
            inputStream = new BufferedInputStream(inputStream);
        }
        MessageDigest messageDigest = null;
        try {
            int bytesRead;
            byte[] buffer = new byte[1024];
            messageDigest = MessageDigest.getInstance("MD5");
            while ((bytesRead = inputStream.read(buffer)) > 0) {
                messageDigest.update(buffer, 0, bytesRead);
            }
        }
        catch (NoSuchAlgorithmException e) {
            throw new BugError("JVM with missing MD5 algorithm for message digest.", new Object[0]);
        }
        finally {
            inputStream.close();
        }
        return messageDigest.digest();
    }

    public static File mkdirs(File file) throws IOException, IllegalArgumentException {
        Params.notNull(file, "File", new Object[0]);
        File parentFile = file.getParentFile();
        if (parentFile != null && !parentFile.exists() && !file.getParentFile().mkdirs()) {
            throw new IOException(String.format("Fail to create target file |%s| directories.", file.getAbsolutePath()));
        }
        return file;
    }

    public static void delete(File file) throws IOException {
        Params.notNull(file, "File", new Object[0]);
        if (file.exists() && !file.delete()) {
            throw new IOException(Strings.format("Fail to delete file |%s|.", file));
        }
    }

    public static File removeFilesHierarchy(File baseDir) throws IllegalArgumentException, IOException {
        Params.notNull(baseDir, "Base directory", new Object[0]);
        Params.isDirectory(baseDir, "Base directory");
        log.debug("Remove files hierarchy with base directory |%s|.", baseDir);
        Files.removeDirectory(baseDir);
        return baseDir;
    }

    private static void removeDirectory(File directory) throws IOException {
        for (File file : directory.listFiles()) {
            if (file.isDirectory()) {
                Files.removeDirectory(file);
            }
            log.trace("Delete %s |%s|.", file.isDirectory() ? "empty directory" : "file", file);
            if (file.delete()) continue;
            throw new IOException(String.format("Fail to delete %s |%s|.", file.isDirectory() ? "empty directory" : "file", file));
        }
    }

    public static boolean inotify(File file, FileNotify notify, int timeout) throws IllegalArgumentException {
        Params.notNull(file, "File", new Object[0]);
        Predicate predicate = INOTIFY_PREDICATES.get((Object)notify);
        if (predicate == null) {
            throw new BugError("Unsupported file notification |%s|. Missing predicate.", new Object[]{notify});
        }
        long timestamp = System.currentTimeMillis() + (long)timeout;
        while (!predicate.test(file)) {
            if (timestamp <= System.currentTimeMillis()) {
                return false;
            }
            try {
                Thread.sleep(500L);
            }
            catch (InterruptedException unused) {
                Thread.currentThread().interrupt();
            }
        }
        return true;
    }

    public static String normalizePath(String path) {
        if (File.separatorChar == '/') {
            return path.replace('\\', '/');
        }
        return path.replace('/', '\\');
    }

    static {
        IMAGE_FILE_EXTENSIONS.add("png");
        IMAGE_FILE_EXTENSIONS.add("gif");
        IMAGE_FILE_EXTENSIONS.add("jpg");
        IMAGE_FILE_EXTENSIONS.add("jpeg");
        IMAGE_FILE_EXTENSIONS.add("tiff");
        IMAGE_FILE_EXTENSIONS.add("bmp");
        INOTIFY_PREDICATES = new HashMap<FileNotify, Predicate>();
        INOTIFY_PREDICATES.put(FileNotify.CREATE, new Predicate(){

            @Override
            public boolean test(Object value) {
                assert (value instanceof File);
                return ((File)value).exists();
            }
        });
        INOTIFY_PREDICATES.put(FileNotify.DELETE, new Predicate(){

            @Override
            public boolean test(Object value) {
                assert (value instanceof File);
                return !((File)value).exists();
            }
        });
    }

    public static enum FileNotify {
        NONE,
        DELETE,
        CREATE;

    }
}

