/*
 * Decompiled with CFR 0.152.
 */
package com.kdgregory.logging.aws.cloudwatch;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.logs.AWSLogs;
import com.amazonaws.services.logs.model.CreateLogGroupRequest;
import com.amazonaws.services.logs.model.CreateLogStreamRequest;
import com.amazonaws.services.logs.model.DataAlreadyAcceptedException;
import com.amazonaws.services.logs.model.DescribeLogGroupsRequest;
import com.amazonaws.services.logs.model.DescribeLogGroupsResult;
import com.amazonaws.services.logs.model.DescribeLogStreamsRequest;
import com.amazonaws.services.logs.model.DescribeLogStreamsResult;
import com.amazonaws.services.logs.model.InputLogEvent;
import com.amazonaws.services.logs.model.InvalidSequenceTokenException;
import com.amazonaws.services.logs.model.LogGroup;
import com.amazonaws.services.logs.model.LogStream;
import com.amazonaws.services.logs.model.OperationAbortedException;
import com.amazonaws.services.logs.model.PutLogEventsRequest;
import com.amazonaws.services.logs.model.PutLogEventsResult;
import com.amazonaws.services.logs.model.PutRetentionPolicyRequest;
import com.amazonaws.services.logs.model.ResourceAlreadyExistsException;
import com.amazonaws.services.logs.model.ResourceNotFoundException;
import com.kdgregory.logging.aws.cloudwatch.CloudWatchWriterConfig;
import com.kdgregory.logging.aws.cloudwatch.CloudWatchWriterStatistics;
import com.kdgregory.logging.aws.internal.AbstractLogWriter;
import com.kdgregory.logging.aws.internal.Utils;
import com.kdgregory.logging.common.LogMessage;
import com.kdgregory.logging.common.factories.ClientFactory;
import com.kdgregory.logging.common.util.InternalLogger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;

public class CloudWatchLogWriter
extends AbstractLogWriter<CloudWatchWriterConfig, CloudWatchWriterStatistics, AWSLogs> {
    private static final int INITIAL_RETRY_DELAY = 100;
    private static final int DESCRIBE_RETRY_TIMEOUT = 30000;
    private static final int SEQNUM_RETRY_TIMEOUT = 3000;
    private String sequenceToken;

    public CloudWatchLogWriter(CloudWatchWriterConfig config, CloudWatchWriterStatistics stats, InternalLogger logger, ClientFactory<AWSLogs> clientFactory) {
        super(config, stats, logger, clientFactory);
        ((CloudWatchWriterStatistics)this.stats).setActualLogGroupName(config.logGroupName);
        ((CloudWatchWriterStatistics)this.stats).setActualLogStreamName(config.logStreamName);
    }

    @Override
    public boolean isMessageTooLarge(LogMessage message) {
        return message.size() + 26 > 0x100000;
    }

    @Override
    protected boolean ensureDestinationAvailable() {
        if (!Pattern.matches("[A-Za-z0-9_/.-]{1,512}", ((CloudWatchWriterConfig)this.config).logGroupName)) {
            this.reportError("invalid log group name: " + ((CloudWatchWriterConfig)this.config).logGroupName, null);
            return false;
        }
        if (!Pattern.matches("[^:*]{1,512}", ((CloudWatchWriterConfig)this.config).logStreamName)) {
            this.reportError("invalid log stream name: " + ((CloudWatchWriterConfig)this.config).logStreamName, null);
            return false;
        }
        try {
            LogGroup logGroup = this.findLogGroup();
            if (logGroup == null) {
                this.logger.debug("creating CloudWatch log group: " + ((CloudWatchWriterConfig)this.config).logGroupName);
                this.createLogGroup();
            } else {
                this.logger.debug("using existing CloudWatch log group: " + ((CloudWatchWriterConfig)this.config).logGroupName);
            }
            LogStream logStream = this.findLogStream();
            if (logStream == null) {
                this.logger.debug("creating CloudWatch log stream: " + ((CloudWatchWriterConfig)this.config).logStreamName);
                this.createLogStream();
            } else {
                this.logger.debug("using existing CloudWatch log stream: " + ((CloudWatchWriterConfig)this.config).logStreamName);
            }
            return true;
        }
        catch (Exception ex) {
            this.reportError("unable to configure log group/stream", ex);
            return false;
        }
    }

    @Override
    protected List<LogMessage> sendBatch(List<LogMessage> currentBatch) {
        Collections.sort(currentBatch);
        return this.attemptToSend(currentBatch);
    }

    @Override
    protected int effectiveSize(LogMessage message) {
        return message.size() + 26;
    }

    @Override
    protected boolean withinServiceLimits(int batchBytes, int numMessages) {
        return batchBytes <= 0x100000 && numMessages <= 10000;
    }

    @Override
    protected void stopAWSClient() {
        ((AWSLogs)this.client).shutdown();
    }

    private List<LogMessage> attemptToSend(List<LogMessage> batch) {
        if (batch.isEmpty()) {
            return batch;
        }
        PutLogEventsRequest request = new PutLogEventsRequest().withLogGroupName(((CloudWatchWriterConfig)this.config).logGroupName).withLogStreamName(((CloudWatchWriterConfig)this.config).logStreamName).withLogEvents(this.constructLogEvents(batch));
        long retryDelay = 100L;
        long timeoutAt = System.currentTimeMillis() + 3000L;
        while (System.currentTimeMillis() < timeoutAt) {
            try {
                request.setSequenceToken(this.getSequenceToken());
                PutLogEventsResult response = ((AWSLogs)this.client).putLogEvents(request);
                this.sequenceToken = response.getNextSequenceToken();
                return Collections.emptyList();
            }
            catch (InvalidSequenceTokenException ex) {
                this.sequenceToken = null;
                ((CloudWatchWriterStatistics)this.stats).updateWriterRaceRetries();
            }
            catch (DataAlreadyAcceptedException ex) {
                this.reportError("received DataAlreadyAcceptedException, dropping batch", (Exception)((Object)ex));
                return Collections.emptyList();
            }
            catch (ResourceNotFoundException ex) {
                this.reportError("log stream missing: " + ((CloudWatchWriterConfig)this.config).logStreamName, null);
                this.ensureDestinationAvailable();
                return batch;
            }
            catch (Exception ex) {
                this.reportError("failed to send batch", ex);
                return batch;
            }
            Utils.sleepQuietly(retryDelay);
            retryDelay *= 2L;
        }
        this.reportError("failed to send due to repeated InvalidSequenceTokenExceptions", null);
        ((CloudWatchWriterStatistics)this.stats).updateUnrecoveredWriterRaceRetries();
        return batch;
    }

    private List<InputLogEvent> constructLogEvents(List<LogMessage> batch) {
        ArrayList<InputLogEvent> result = new ArrayList<InputLogEvent>(batch.size());
        for (LogMessage msg : batch) {
            InputLogEvent event = new InputLogEvent().withTimestamp(Long.valueOf(msg.getTimestamp())).withMessage(msg.getMessage());
            result.add(event);
        }
        return result;
    }

    private String getSequenceToken() {
        if (((CloudWatchWriterConfig)this.config).dedicatedWriter && this.sequenceToken != null) {
            return this.sequenceToken;
        }
        LogStream stream = this.findLogStream();
        if (stream == null) {
            throw new ResourceNotFoundException("stream appears to have been deleted");
        }
        return stream.getUploadSequenceToken();
    }

    private LogGroup findLogGroup() {
        DescribeLogGroupsResult result;
        DescribeLogGroupsRequest request = new DescribeLogGroupsRequest().withLogGroupNamePrefix(((CloudWatchWriterConfig)this.config).logGroupName);
        do {
            result = ((AWSLogs)this.client).describeLogGroups(request);
            for (LogGroup group : result.getLogGroups()) {
                if (!group.getLogGroupName().equals(((CloudWatchWriterConfig)this.config).logGroupName)) continue;
                return group;
            }
            request.setNextToken(result.getNextToken());
        } while (result.getNextToken() != null);
        return null;
    }

    private void createLogGroup() {
        while (true) {
            try {
                CreateLogGroupRequest request = new CreateLogGroupRequest().withLogGroupName(((CloudWatchWriterConfig)this.config).logGroupName);
                ((AWSLogs)this.client).createLogGroup(request);
                for (int ii = 0; ii < 300; ++ii) {
                    if (this.findLogGroup() != null) {
                        this.optSetLogGroupRetentionPolicy();
                        return;
                    }
                    Utils.sleepQuietly(100L);
                }
                throw new RuntimeException("unable to create log group after 30 seconds; aborting");
            }
            catch (ResourceAlreadyExistsException ex) {
                return;
            }
            catch (OperationAbortedException ex) {
                Utils.sleepQuietly(250L);
                continue;
            }
            break;
        }
    }

    private void optSetLogGroupRetentionPolicy() {
        if (((CloudWatchWriterConfig)this.config).retentionPeriod == null) {
            return;
        }
        this.logger.debug("setting retention policy on " + ((CloudWatchWriterConfig)this.config).logGroupName + " to " + ((CloudWatchWriterConfig)this.config).retentionPeriod + " days");
        try {
            ((AWSLogs)this.client).putRetentionPolicy(new PutRetentionPolicyRequest(((CloudWatchWriterConfig)this.config).logGroupName, ((CloudWatchWriterConfig)this.config).retentionPeriod));
        }
        catch (Exception ex) {
            this.reportError("failed to set retention policy on log group " + ((CloudWatchWriterConfig)this.config).logGroupName, ex);
        }
    }

    private LogStream findLogStream() {
        DescribeLogStreamsRequest request = new DescribeLogStreamsRequest().withLogGroupName(((CloudWatchWriterConfig)this.config).logGroupName).withLogStreamNamePrefix(((CloudWatchWriterConfig)this.config).logStreamName);
        try {
            DescribeLogStreamsResult result;
            do {
                result = this.describeStreamsWithRetry(request);
                for (LogStream stream : result.getLogStreams()) {
                    if (!stream.getLogStreamName().equals(((CloudWatchWriterConfig)this.config).logStreamName)) continue;
                    return stream;
                }
                request.setNextToken(result.getNextToken());
            } while (result.getNextToken() != null);
        }
        catch (Exception ex) {
            this.reportError("unable to describe log stream", ex);
        }
        return null;
    }

    private DescribeLogStreamsResult describeStreamsWithRetry(DescribeLogStreamsRequest request) {
        long retryDelay = 100L;
        long timeoutAt = System.currentTimeMillis() + 30000L;
        while (System.currentTimeMillis() < timeoutAt) {
            try {
                return ((AWSLogs)this.client).describeLogStreams(request);
            }
            catch (AmazonServiceException ex) {
                if (!ex.getMessage().contains("ThrottlingException")) {
                    throw ex;
                }
                Utils.sleepQuietly(retryDelay);
                retryDelay *= 2L;
            }
        }
        throw new RuntimeException("DescribeLogStreams has been throttled for 30000 ms");
    }

    private void createLogStream() {
        try {
            CreateLogStreamRequest request = new CreateLogStreamRequest().withLogGroupName(((CloudWatchWriterConfig)this.config).logGroupName).withLogStreamName(((CloudWatchWriterConfig)this.config).logStreamName);
            ((AWSLogs)this.client).createLogStream(request);
            for (int ii = 0; ii < 300; ++ii) {
                if (this.findLogStream() != null) {
                    return;
                }
                Utils.sleepQuietly(100L);
            }
            throw new RuntimeException("unable to create log strean after 30 seconds; aborting");
        }
        catch (ResourceAlreadyExistsException ex) {
            return;
        }
    }
}

