// Copyright (c) Keith D Gregory
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.kdgregory.logging.aws.internal.retrievers;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;


/**
 *  A helper class for performing retrievals using reflection. This exists to
 *  avoid making hard references to optional libraries.
 *  <p>
 *  Each instance is intended to perform a single retrieval operation on an AWS
 *  client. A typical usage has the following steps:
 *  <ol>
 *  <li> Create an instance with the classnames of the client, request, and response objects.
 *  <li> Call {@link #instantiate} for the client and request objects.
 *  <li> Call {@link #setRequestValue} to prepare the request.
 *  <li> Call {@link #invokeRequest} to peform the request, saving the result.
 *  <li> Call {@link #getResponseValue} to retrieve any values from the result.
 *  <li> Call {@link #shutdown} to shut down the client. Typically in a finally
 *       block.
 *  </ol>
 *  Any exceptions thrown during method invocations are retained, and short-circuit
 *  any subsequent operations. It is therefore safe to perform a chain of operations,
 *  as long as you can accept a null value at the end of the chain.
 *  <p>
 *  All variables involved in this process must be defined as <code>Object</code>,
 *  to avoid hard references to the library.
 *  <p>
 *  In addition to the request-response methods, there are utility methods to load
 *  a class, instantiate that class, and invoke arbitrary 0- or 1-arity static or
 *  instance methods. These are preferable to similar functions in <code>Utils</code>
 *  if you want to implement a sequence operations (due to exception handling).
 */
public class AbstractReflectionBasedRetriever
{
    // these are public because otherwise I'd just need to add accessors for testing
    public Throwable exception;
    public Class<?> builderKlass;
    public Class<?> clientKlass;
    public Class<?> requestKlass;
    public Class<?> responseKlass;


    /**
     *  Constructs an instance that will attempt to invoke the default builder rather than
     *  directly instantiate the client.
     */
    public AbstractReflectionBasedRetriever(String builderClassName, String clientClassName, String requestClassName, String responseClassName)
    {
        builderKlass = loadClass(builderClassName);
        clientKlass = loadClass(clientClassName);
        requestKlass = loadClass(requestClassName);
        responseKlass = loadClass(responseClassName);
    }


    /**
     *  Constructs an instance that will use a direct client constructor.
     */
    public AbstractReflectionBasedRetriever(String clientClassName, String requestClassName, String responseClassName)
    {
        this(null, clientClassName, requestClassName, responseClassName);
    }


    /**
     *  Constructs an instance for static method invocation. This just loads the "client" class.
     */
    public AbstractReflectionBasedRetriever(String className)
    {
        this(null, className, null, null);
    }


    /**
     *  Constructs an instance for when you just want to use the utility methods.
     */
    public AbstractReflectionBasedRetriever()
    {
        // nothing happening here
    }


    /**
     *  Exception-safe class load. Returns null if unable to load the class, and tracks
     *  the exception.
     */
    public Class<?> loadClass(String className)
    {
        if (className == null)
            return null;

        try
        {
            return Class.forName(className);
        }
        catch (Throwable ex)
        {
            exception = ex;
            return null;
        }
    }


    /**
     *  Instantiates the provided class. Normally this will be one of the classes
     *  loaded during construction, or via {@link #loadClass}.
     */
    public Object instantiate(Class<?> klass)
    {
        if ((exception != null) || (klass == null))
            return null;

        try
        {
            return klass.newInstance();
        }
        catch (Throwable ex)
        {
            exception = ex;
            return null;
        }
    }


    /**
     *  Attempts to invoke the builder's "default client" method, returning null
     *  if there's an error or the builder class is not available.
     */
    public Object invokeBuilder()
    {
        if ((exception != null) || (builderKlass == null) || (clientKlass == null))
            return null;

        return clientKlass.cast(invokeMethod(builderKlass, null, "defaultClient", null, null));
    }


    /**
     *  Base method invoker. This can be used for functions with arity 0 or 1:
     *  for former, pass null <code>paramKlass</code>.
     *  <p>
     *  Returns null if there's an outstanding exception or the provided object is
     *  null (this is a short-circuit), as well as when an exception occurs during
     *  invocation. For the latter, unwraps <code>InvocationTargetException</code>.
     */
    public Object invokeMethod(Class<?> objKlass, Object obj, String methodName, Class<?> paramKlass, Object value)
    {
        if ((exception != null) || (objKlass == null))
            return null;

        try
        {
            if (paramKlass != null)
            {
                Method method = objKlass.getMethod(methodName, paramKlass);
                return method.invoke(obj, value);
            }
            else
            {
                Method method = objKlass.getMethod(methodName);
                return method.invoke(obj);
            }
        }
        catch (InvocationTargetException ex)
        {
            exception = ex.getCause();
            return null;
        }
        catch (Throwable ex)
        {
            exception = ex;
            return null;
        }
    }


    /**
     *  Static method invoke. Can also be used for 0- or 1-arity functions, by passing
     *  a null parameter class.
     */
    public Object invokeStatic(Class<?> objKlass, String methodName, Class<?> paramKlass, Object value)
    {
        return invokeMethod(objKlass, null, methodName, paramKlass, value);
    }


    /**
     *  Convenience method that invokes a setter on the provided object, which is
     *  assumed to be an instance of the constructed request class.
     */
    public void setRequestValue(Object request, String methodName, Class<?> valueKlass, Object value)
    {
        invokeMethod(requestKlass, request, methodName, valueKlass, value);
    }


    /**
     *  Convenience method that invokes a single-argument client method. This uses
     *  the client, request, and response classes passed to the constructor.
     */
    public Object invokeRequest(Object client, String methodName, Object value)
    {
        return invokeMethod(clientKlass, client, methodName, requestKlass, value);
    }


    /**
     *  Convenience method to invoke an accessor method on an object that's assumed
     *  to be the constructed response class.
     */
    public <T> T getResponseValue(Object response, String methodName, Class<T> resultKlass)
    {
        if (resultKlass == null)
            return null;

        return resultKlass.cast(invokeMethod(responseKlass, response, methodName, null, null));
    }


    /**
     *  Invokes the <code>shutdown()</code> method on the provided client, which is assumed
     *  to be an instance of the constructed client class. This short-circuits if the provided
     *  client object is null, and ignores any exceptions. It should be called in a finally
     *  block for any operation that creates an AWS client.
     */
    public void shutdown(Object client)
    {
        // note: if we have a client we want to shut it down, even if an exception has happened
        if (client == null)
            return;

        try
        {
            Method method = clientKlass.getMethod("shutdown");
            method.invoke(client);
        }
        catch (Throwable ex)
        {
            // ignored: at this point we don't care about exceptions because we've got a value
        }
    }
}