/*
 * Decompiled with CFR 0.152.
 */
package com.kdgregory.logging.aws.cloudwatch;

import com.kdgregory.logging.aws.cloudwatch.CloudWatchWriterConfig;
import com.kdgregory.logging.aws.cloudwatch.CloudWatchWriterStatistics;
import com.kdgregory.logging.aws.facade.CloudWatchFacade;
import com.kdgregory.logging.aws.facade.CloudWatchFacadeException;
import com.kdgregory.logging.aws.internal.AbstractLogWriter;
import com.kdgregory.logging.common.LogMessage;
import com.kdgregory.logging.common.util.InternalLogger;
import com.kdgregory.logging.common.util.RetryManager2;
import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;

public class CloudWatchLogWriter
extends AbstractLogWriter<CloudWatchWriterConfig, CloudWatchWriterStatistics> {
    private static final String SEQUENCE_TOKEN_FLAG_VALUE = "";
    private CloudWatchFacade facade;
    protected RetryManager2 describeRetry = new RetryManager2("describe", Duration.ofMillis(50L), true, true);
    protected RetryManager2 createRetry = new RetryManager2("create", Duration.ofMillis(200L), true, true);
    protected Duration sendTimeout = Duration.ofMillis(2000L);
    protected RetryManager2 sendRetry = new RetryManager2("send", Duration.ofMillis(200L), true, false);
    private String sequenceToken = "";

    public CloudWatchLogWriter(CloudWatchWriterConfig config, CloudWatchWriterStatistics stats, InternalLogger logger, CloudWatchFacade facade) {
        super(config, stats, logger);
        this.facade = facade;
        ((CloudWatchWriterStatistics)this.stats).setActualLogGroupName(config.getLogGroupName());
        ((CloudWatchWriterStatistics)this.stats).setActualLogStreamName(config.getLogStreamName());
    }

    @Override
    public int maxMessageSize() {
        return 262118;
    }

    @Override
    protected boolean ensureDestinationAvailable() {
        List<String> configErrors = ((CloudWatchWriterConfig)this.config).validate();
        if (!configErrors.isEmpty()) {
            for (String error : configErrors) {
                this.reportError(error, null);
            }
            return false;
        }
        Instant timeoutAt = Instant.now().plusMillis(((CloudWatchWriterConfig)this.config).getInitializationTimeout());
        try {
            this.logger.debug("checking for existence of CloudWatch log group: " + ((CloudWatchWriterConfig)this.config).getLogGroupName());
            if (this.facade.findLogGroup() == null) {
                this.createLogGroup(timeoutAt);
            } else {
                this.logger.debug("using existing CloudWatch log group: " + ((CloudWatchWriterConfig)this.config).getLogGroupName());
            }
            this.logger.debug("checking for existence of CloudWatch log stream: " + ((CloudWatchWriterConfig)this.config).getLogStreamName());
            if (this.facade.findLogStream() == null) {
                this.createLogStream(timeoutAt);
            } else {
                this.logger.debug("using existing CloudWatch log stream: " + ((CloudWatchWriterConfig)this.config).getLogStreamName());
            }
            return true;
        }
        catch (Throwable ex) {
            this.reportError("exception during initialization", ex);
            return false;
        }
    }

    @Override
    protected List<LogMessage> sendBatch(List<LogMessage> batch) {
        ((CloudWatchWriterStatistics)this.stats).setLastBatchSize(batch.size());
        if (((CloudWatchWriterConfig)this.config).getEnableBatchLogging()) {
            this.logger.debug("about to write batch of " + batch.size() + " message(s)");
        }
        if (batch.isEmpty()) {
            return batch;
        }
        Collections.sort(batch);
        Instant timeoutAt = Instant.now().plus(this.sendTimeout);
        List result = this.sendRetry.invoke(timeoutAt, () -> {
            try {
                this.sequenceToken = this.facade.putEvents(this.retrieveSequenceToken(timeoutAt), batch);
                if (((CloudWatchWriterConfig)this.config).getEnableBatchLogging()) {
                    this.logger.debug("wrote batch of " + batch.size() + " message(s)");
                }
                return Collections.emptyList();
            }
            catch (CloudWatchFacadeException ex) {
                switch (ex.getReason()) {
                    case THROTTLING: {
                        ((CloudWatchWriterStatistics)this.stats).incrementThrottledWrites();
                        return null;
                    }
                    case INVALID_SEQUENCE_TOKEN: {
                        this.sequenceToken = SEQUENCE_TOKEN_FLAG_VALUE;
                        ((CloudWatchWriterStatistics)this.stats).updateWriterRaceRetries();
                        return null;
                    }
                    case ALREADY_PROCESSED: {
                        this.logger.warn("received DataAlreadyAcceptedException, dropping batch");
                        return Collections.emptyList();
                    }
                    case MISSING_LOG_GROUP: 
                    case MISSING_LOG_STREAM: {
                        this.reportError(ex.getMessage(), ex);
                        this.ensureDestinationAvailable();
                        return batch;
                    }
                }
                this.reportError("failed to send: " + ex.getMessage(), ex.getCause());
                return batch;
            }
            catch (Exception ex) {
                this.logger.error("unexpected exception in sendBatch()", ex);
                return batch;
            }
        });
        if (result != null) {
            return result;
        }
        if (this.sequenceToken.equals(SEQUENCE_TOKEN_FLAG_VALUE)) {
            this.logger.warn("batch failed: unrecovered sequence token race");
            ((CloudWatchWriterStatistics)this.stats).updateUnrecoveredWriterRaceRetries();
        } else {
            this.logger.warn("batch failed: repeated throttling");
        }
        return batch;
    }

    @Override
    protected int effectiveSize(LogMessage message) {
        return message.size() + 26;
    }

    @Override
    protected boolean withinServiceLimits(int batchBytes, int numMessages) {
        return batchBytes <= 0x100000 && numMessages <= 10000;
    }

    @Override
    protected void stopAWSClient() {
        try {
            this.facade.shutdown();
        }
        catch (CloudWatchFacadeException ex) {
            this.reportError("exception shutting down CloudWatch client", ex);
        }
    }

    private void createLogGroup(Instant timeoutAt) {
        this.logger.debug("creating CloudWatch log group: " + ((CloudWatchWriterConfig)this.config).getLogGroupName());
        this.createRetry.invoke(timeoutAt, () -> {
            this.facade.createLogGroup();
            return Boolean.TRUE;
        }, (Consumer<RuntimeException>)new DefaultExceptionHandler());
        String logGroupArn = this.describeRetry.invoke(timeoutAt, () -> this.facade.findLogGroup());
        if (logGroupArn == null) {
            throw new RuntimeException("timed out while waiting for CloudWatch log group");
        }
        try {
            if (((CloudWatchWriterConfig)this.config).getRetentionPeriod() != null) {
                this.logger.debug("setting retention period to: " + ((CloudWatchWriterConfig)this.config).getRetentionPeriod());
                this.facade.setLogGroupRetention();
            }
        }
        catch (CloudWatchFacadeException ex) {
            this.logger.error("exception setting retention policy", ex);
        }
    }

    private void createLogStream(Instant timeoutAt) {
        this.logger.debug("creating CloudWatch log stream: " + ((CloudWatchWriterConfig)this.config).getLogStreamName());
        this.createRetry.invoke(timeoutAt, () -> {
            this.facade.createLogStream();
            return Boolean.TRUE;
        }, (Consumer<RuntimeException>)new DefaultExceptionHandler());
        this.describeRetry.invoke(timeoutAt, () -> this.facade.findLogStream());
        this.sequenceToken = null;
    }

    private String retrieveSequenceToken(Instant timeoutAt) {
        if (!((CloudWatchWriterConfig)this.config).getDedicatedWriter() || SEQUENCE_TOKEN_FLAG_VALUE.equals(this.sequenceToken)) {
            this.sequenceToken = this.facade.retrieveSequenceToken();
        }
        return this.sequenceToken;
    }

    private static class DefaultExceptionHandler
    implements Consumer<RuntimeException> {
        private DefaultExceptionHandler() {
        }

        @Override
        public void accept(RuntimeException ex) {
            if (ex instanceof CloudWatchFacadeException && ((CloudWatchFacadeException)ex).isRetryable()) {
                return;
            }
            throw ex;
        }
    }
}

