/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.koloboke.collect.map;

import com.koloboke.collect.*;
import com.koloboke.compile.KolobokeMap;
import java.util.function.Consumer;

import javax.annotation.Nonnull;

import java.util.Map;



/**
 * An immutable factory of {@code DoubleDoubleMap}s.
 *
 * 
 * 
 * @param <F> the concrete factory type which extends this interface
 * @see DoubleDoubleMap
 * @see KolobokeMap @KolobokeMap
 */
public interface DoubleDoubleMapFactory<F extends DoubleDoubleMapFactory<F>>
        extends ContainerFactory<F> {

    

    

    

    

    

    
    

    
    




    /**
     * Returns the value to which {@linkplain DoubleDoubleMap#defaultValue() default value} of the maps
     * constructed by this factory is set. Default value is {@code
     * 0.0}.
     *
     * @return the default value of the maps constructed by this factory
     */
    double getDefaultValue();

    /**
     * Returns a copy of this factory, with exception that it constructs maps with
     * {@linkplain DoubleDoubleMap#defaultValue() default value} set to the given {@code double} value.
     *
     * @param defaultValue the new default {@code double} value
     * @return a copy of this factory, which constructs maps with the given {@code defaultValue}
     */
    @Nonnull
    F withDefaultValue(double defaultValue);

    

    /**
     * Constructs a new empty mutable map of the {@linkplain #getDefaultExpectedSize()
     * default expected size}.
     *
    
     
     * @return a new empty mutable map
     */
    @Nonnull
     DoubleDoubleMap newMutableMap();

    /**
     * Constructs a new empty mutable map of the given expected size.
     *
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new empty mutable map of the given expected size
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(int expectedSize);

    

    


    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2, int expectedSize);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3, int expectedSize);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4, int expectedSize);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4,
            @Nonnull Map<Double, Double> map5, int expectedSize);



    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull
            Consumer<com.koloboke.function.DoubleDoubleConsumer> entriesSupplier
            , int expectedSize);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(
            @Nonnull double[] keys, @Nonnull double[] values, int expectedSize);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(
            @Nonnull Double[] keys, @Nonnull Double[] values, int expectedSize);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Double> values, int expectedSize);
    

    

    /**
     * Constructs a new mutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new mutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull Map<Double, Double> map);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4,
            @Nonnull Map<Double, Double> map5);



    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * 
    
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull
            Consumer<com.koloboke.function.DoubleDoubleConsumer> entriesSupplier
            );

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(
            @Nonnull double[] keys, @Nonnull double[] values);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(
            @Nonnull Double[] keys, @Nonnull Double[] values);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleDoubleMap newMutableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Double> values);

    /**
     * Constructs a new mutable map of the single specified mapping.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new mutable map of the single specified mapping
     */
    @Nonnull
     DoubleDoubleMap newMutableMapOf(double k1, double v1);

    /**
     * Constructs a new mutable map of the two specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new mutable map of the two specified mappings
     */
    @Nonnull
     DoubleDoubleMap newMutableMapOf(double k1, double v1,
            double k2, double v2);

    /**
     * Constructs a new mutable map of the three specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new mutable map of the three specified mappings
     */
    @Nonnull
     DoubleDoubleMap newMutableMapOf(double k1, double v1,
            double k2, double v2, double k3, double v3);

    /**
     * Constructs a new mutable map of the four specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new mutable map of the four specified mappings
     */
    @Nonnull
     DoubleDoubleMap newMutableMapOf(double k1, double v1,
            double k2, double v2, double k3, double v3,
            double k4, double v4);

    /**
     * Constructs a new mutable map of the five specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new mutable map of the five specified mappings
     */
    @Nonnull
     DoubleDoubleMap newMutableMapOf(double k1, double v1,
            double k2, double v2, double k3, double v3,
            double k4, double v4, double k5, double v5);
    /**
     * Constructs a new empty updatable map of the {@linkplain #getDefaultExpectedSize()
     * default expected size}.
     *
    
     
     * @return a new empty updatable map
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap();

    /**
     * Constructs a new empty updatable map of the given expected size.
     *
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new empty updatable map of the given expected size
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(int expectedSize);

    

    


    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2, int expectedSize);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3, int expectedSize);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4, int expectedSize);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4,
            @Nonnull Map<Double, Double> map5, int expectedSize);



    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull
            Consumer<com.koloboke.function.DoubleDoubleConsumer> entriesSupplier
            , int expectedSize);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(
            @Nonnull double[] keys, @Nonnull double[] values, int expectedSize);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(
            @Nonnull Double[] keys, @Nonnull Double[] values, int expectedSize);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Double> values, int expectedSize);
    

    

    /**
     * Constructs a new updatable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new updatable map with the same mappings as the specified {@code map}
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull Map<Double, Double> map);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4,
            @Nonnull Map<Double, Double> map5);



    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * 
    
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull
            Consumer<com.koloboke.function.DoubleDoubleConsumer> entriesSupplier
            );

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(
            @Nonnull double[] keys, @Nonnull double[] values);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(
            @Nonnull Double[] keys, @Nonnull Double[] values);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Double> values);

    /**
     * Constructs a new updatable map of the single specified mapping.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new updatable map of the single specified mapping
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMapOf(double k1, double v1);

    /**
     * Constructs a new updatable map of the two specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new updatable map of the two specified mappings
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMapOf(double k1, double v1,
            double k2, double v2);

    /**
     * Constructs a new updatable map of the three specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new updatable map of the three specified mappings
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMapOf(double k1, double v1,
            double k2, double v2, double k3, double v3);

    /**
     * Constructs a new updatable map of the four specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new updatable map of the four specified mappings
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMapOf(double k1, double v1,
            double k2, double v2, double k3, double v3,
            double k4, double v4);

    /**
     * Constructs a new updatable map of the five specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new updatable map of the five specified mappings
     */
    @Nonnull
     DoubleDoubleMap newUpdatableMapOf(double k1, double v1,
            double k2, double v2, double k3, double v3,
            double k4, double v4, double k5, double v5);

    

    


    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2, int expectedSize);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3, int expectedSize);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4, int expectedSize);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4,
            @Nonnull Map<Double, Double> map5, int expectedSize);



    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull
            Consumer<com.koloboke.function.DoubleDoubleConsumer> entriesSupplier
            , int expectedSize);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(
            @Nonnull double[] keys, @Nonnull double[] values, int expectedSize);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(
            @Nonnull Double[] keys, @Nonnull Double[] values, int expectedSize);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Double> values, int expectedSize);
    

    

    /**
     * Constructs a new immutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new immutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull Map<Double, Double> map);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull Map<Double, Double> map1,
            @Nonnull Map<Double, Double> map2,
            @Nonnull Map<Double, Double> map3,
            @Nonnull Map<Double, Double> map4,
            @Nonnull Map<Double, Double> map5);



    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * 
    
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull
            Consumer<com.koloboke.function.DoubleDoubleConsumer> entriesSupplier
            );

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(
            @Nonnull double[] keys, @Nonnull double[] values);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(
            @Nonnull Double[] keys, @Nonnull Double[] values);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleDoubleMap newImmutableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Double> values);

    /**
     * Constructs a new immutable map of the single specified mapping.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new immutable map of the single specified mapping
     */
    @Nonnull
     DoubleDoubleMap newImmutableMapOf(double k1, double v1);

    /**
     * Constructs a new immutable map of the two specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new immutable map of the two specified mappings
     */
    @Nonnull
     DoubleDoubleMap newImmutableMapOf(double k1, double v1,
            double k2, double v2);

    /**
     * Constructs a new immutable map of the three specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new immutable map of the three specified mappings
     */
    @Nonnull
     DoubleDoubleMap newImmutableMapOf(double k1, double v1,
            double k2, double v2, double k3, double v3);

    /**
     * Constructs a new immutable map of the four specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new immutable map of the four specified mappings
     */
    @Nonnull
     DoubleDoubleMap newImmutableMapOf(double k1, double v1,
            double k2, double v2, double k3, double v3,
            double k4, double v4);

    /**
     * Constructs a new immutable map of the five specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new immutable map of the five specified mappings
     */
    @Nonnull
     DoubleDoubleMap newImmutableMapOf(double k1, double v1,
            double k2, double v2, double k3, double v3,
            double k4, double v4, double k5, double v5);
}
