/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.koloboke.collect.map;

import com.koloboke.collect.Container;
import com.koloboke.collect.Equivalence;
import com.koloboke.compile.KolobokeMap;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.BiConsumer;
import java.util.function.BiPredicate;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Function;
import java.util.function.BiFunction;
import com.koloboke.collect.ObjCollection;
import com.koloboke.collect.set.ObjSet;
import com.koloboke.collect.set.ObjSet;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import java.util.Map;


/**
 * 
 * The library's extension of the classic {@link Map} interface.
 * 
 *
 * @see ObjObjMapFactory
 * @see KolobokeMap @KolobokeMap
 */
public interface ObjObjMap<K, V> extends Map<K, V>, Container {

    /**
     * Returns the equivalence strategy for keys in this map. All methods in the {@link Map}
     * interface which defined in terms of {@link Object#equals(Object)} equality of key objects
     * (almost all methods, actually), are supposed to use this equivalence instead.
     *
     * @return the equivalence strategy for keys in this map
     * @see com.koloboke.collect.map.hash.HashObjObjMapFactory#withKeyEquivalence
     */
    @Nonnull
    Equivalence<K> keyEquivalence();

    /**
     * Returns the equivalence strategy for values in this map. All methods in the {@link Map}
     * interface which defined in terms of {@link Object#equals(Object)} equality of value objects,
     * for example, {@link #containsValue(Object)} and {@link #remove(Object, Object)},
     * are supposed to use this equivalence instead.
     *
     * @return the equivalence strategy for values in this map
     * @see ObjObjMapFactory#withValueEquivalence
     */
    @Nonnull
    Equivalence<V> valueEquivalence();




    









    /**
     * Checks the given {@code predicate} on each entry in this map until all entries
     * have been processed or the predicate returns {@code false} for some entry,
     * or throws an {@code Exception}. Exceptions thrown by the predicate are relayed to the caller.
     *
     * <p>The entries will be processed in the same order as the entry set iterator unless that
     * order is unspecified in which case implementations may use an order which differs from
     * the entry set iterator.
     *
     * <p>If the map is empty, this method returns {@code true} immediately.
     *
     * @return {@code true} if the predicate returned {@code true} for all entries of the map,
     *         {@code false} if it returned {@code false} for the entry
     * @param predicate the predicate to be checked for each entry
     * @see <a href="{@docRoot}/overview-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    boolean forEachWhile(@Nonnull BiPredicate<? super K, ? super V> predicate);

    /**
     * Returns a new cursor over the entries of this map. It's order is always correspond to the
     * entry set iterator order.
     *
     * <p>Basic cursor usage idiom is: <pre>{@code
     * for (ObjObjCursor<K, V> cur = map.cursor(); cur.moveNext();) {
     *     // Work with cur.key() and cur.value()
     *     // Call cur.remove() to remove the current entry
     * }}</pre>
     *
     * @return a new cursor over the entries of this map
     * @see <a href="{@docRoot}/overview-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    @Nonnull
    ObjObjCursor<K, V> cursor();


    @Override
    @Nonnull
    ObjSet<K> keySet();

    @Override
    @Nonnull
    ObjCollection<V> values();

    @Override
    @Nonnull
    ObjSet<Entry<K, V>> entrySet();

    

    

    

    

    


    

    




    

    

    

    























    





    /**
     * Removes all of the entries of this collection that satisfy the given predicate.
     * Errors or runtime exceptions thrown during iteration or by the predicate are relayed
     * to the caller.
     *
     * @param filter a predicate which returns {@code true} for elements to be removed
     * @return {@code true} if any elements were removed
     * @throws NullPointerException if the specified filter is null
     * @throws UnsupportedOperationException if elements cannot be removed from this collection.
     *         Implementations may throw this exception if a matching element cannot be removed
     *         or if, in general, removal is not supported.
     * @see <a href="{@docRoot}/overview-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    boolean removeIf(@Nonnull BiPredicate<? super K, ? super V> filter);
}
