/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.koloboke.collect.map.hash;

import com.koloboke.collect.map.ObjShortMap;
import com.koloboke.compile.KolobokeMap;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.ServiceLoader;


/**
 * This class consists only of static factory methods to construct {@code HashObjShortMap}s, and
 * the default {@link HashObjShortMapFactory} static provider ({@link #getDefaultFactory()}).
 *
 * @see HashObjShortMap
 * @see KolobokeMap @KolobokeMap
 */
public final class HashObjShortMaps {
    
    private static class DefaultFactoryHolder {
        private static final HashObjShortMapFactory defaultFactory =
                ServiceLoader.load(HashObjShortMapFactory.class).iterator().next();
    }

    /**
     * Returns the default {@link HashObjShortMapFactory} implementation, to which
     * all static methods in this class delegate.
     *
      * @param <K> the most general key type of the maps that could
                               be constructed by the returned factory 
     
     * @return the default {@link HashObjShortMapFactory} implementation
     * @throws RuntimeException if no implementations
     *         of {@link HashObjShortMapFactory} are provided
     */
    @Nonnull
    public static <K> HashObjShortMapFactory<K> getDefaultFactory() {
        return (HashObjShortMapFactory<K>) DefaultFactoryHolder.defaultFactory;
    }

    

    

    

    


    /**
     * Constructs a new empty mutable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap() newMutableMap()}.
     *
      * @param <K> the key type of the returned map 
     
     * @return a new empty mutable map
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap() {
        return getDefaultFactory().<K>newMutableMap();
    }

    /**
     * Constructs a new empty mutable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(int) newMutableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new empty mutable map of the given expected size
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(int expectedSize) {
        return getDefaultFactory().<K>newMutableMap(expectedSize);
    }

    
    
    
    

    /**
     * Constructs a new mutable map with the same mappings as the specified {@code map}.
     *
     * 
     * <p>If the specified map is an instance of {@code ObjShortMap} and has the same
     * {@linkplain ObjShortMap#keyEquivalence() key equivalence} with this factory (and thus
     * the constructed map), the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(
     * Map, int) newMutableMap(map, expectedSize)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Map<? extends K, Short> map, int expectedSize) {
        return getDefaultFactory().<K>newMutableMap(map, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Map,
     * Map, int) newMutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2, int expectedSize) {
        return getDefaultFactory().<K>newMutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Map, Map,
     * Map, int) newMutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3, int expectedSize) {
        return getDefaultFactory().<K>newMutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Map, Map,
     * Map, Map, int) newMutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4, int expectedSize) {
        return getDefaultFactory().<K>newMutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map, int) newMutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4,
            @Nonnull Map<? extends K, Short> map5, int expectedSize) {
        return getDefaultFactory().<K>newMutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(
     * Consumer, int) newMutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(@Nonnull
            Consumer<com.koloboke.function.ObjShortConsumer<K>> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().<K>newMutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Object[], short[], int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull K[] keys, @Nonnull short[] values, int expectedSize) {
        return getDefaultFactory().<K>newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Object[],
     * Short[], int) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull K[] keys, @Nonnull Short[] values, int expectedSize) {
        return getDefaultFactory().<K>newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Iterable, Iterable, int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Short> values
            , int expectedSize) {
        return getDefaultFactory().<K>newMutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new mutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(
     * Map) newMutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Map<? extends K, Short> map) {
        return getDefaultFactory().<K>newMutableMap(map);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Map,
     * Map) newMutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2) {
        return getDefaultFactory().<K>newMutableMap(map1, map2);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Map, Map,
     * Map) newMutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3) {
        return getDefaultFactory().<K>newMutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Map, Map,
     * Map, Map) newMutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4) {
        return getDefaultFactory().<K>newMutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map) newMutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4,
            @Nonnull Map<? extends K, Short> map5) {
        return getDefaultFactory().<K>newMutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(
     * Consumer) newMutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(@Nonnull
            Consumer<com.koloboke.function.ObjShortConsumer<K>> entriesSupplier
            ) {
        return getDefaultFactory().<K>newMutableMap(entriesSupplier);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Object[], short[]
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull K[] keys, @Nonnull short[] values) {
        return getDefaultFactory().<K>newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Object[],
     * Short[]) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull K[] keys, @Nonnull Short[] values) {
        return getDefaultFactory().<K>newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMap(Iterable, Iterable
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Short> values
            ) {
        return getDefaultFactory().<K>newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMapOf(Object, short
     * ) newMutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map of the single specified mapping
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMapOf(
            K k1, short v1) {
        return getDefaultFactory().<K>newMutableMapOf(k1, v1);
    }

    /**
     * Constructs a new mutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMapOf(Object, short,
     * Object, short) newMutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map of the two specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMapOf(
            K k1, short v1, K k2, short v2) {
        return getDefaultFactory().<K>newMutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new mutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMapOf(Object, short,
     * Object, short, Object, short) newMutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map of the three specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMapOf(
            K k1, short v1, K k2, short v2,
            K k3, short v3) {
        return getDefaultFactory().<K>newMutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new mutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMapOf(Object, short,
     * Object, short, Object, short, Object, short
     * ) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map of the four specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMapOf(
            K k1, short v1, K k2, short v2,
            K k3, short v3, K k4, short v4) {
        return getDefaultFactory().<K>newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new mutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newMutableMapOf(Object, short,
     * Object, short, Object, short, Object, short,
     * Object, short) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new mutable map of the five specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newMutableMapOf(
            K k1, short v1, K k2, short v2,
            K k3, short v3, K k4, short v4,
            K k5, short v5) {
        return getDefaultFactory().<K>newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    /**
     * Constructs a new empty updatable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap() newUpdatableMap()}.
     *
      * @param <K> the key type of the returned map 
     
     * @return a new empty updatable map
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap() {
        return getDefaultFactory().<K>newUpdatableMap();
    }

    /**
     * Constructs a new empty updatable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(int) newUpdatableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new empty updatable map of the given expected size
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(int expectedSize) {
        return getDefaultFactory().<K>newUpdatableMap(expectedSize);
    }

    
    
    
    

    /**
     * Constructs a new updatable map with the same mappings as the specified {@code map}.
     *
     * 
     * <p>If the specified map is an instance of {@code ObjShortMap} and has the same
     * {@linkplain ObjShortMap#keyEquivalence() key equivalence} with this factory (and thus
     * the constructed map), the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(
     * Map, int) newUpdatableMap(map, expectedSize)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Short> map, int expectedSize) {
        return getDefaultFactory().<K>newUpdatableMap(map, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Map,
     * Map, int) newUpdatableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2, int expectedSize) {
        return getDefaultFactory().<K>newUpdatableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Map, Map,
     * Map, int) newUpdatableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3, int expectedSize) {
        return getDefaultFactory().<K>newUpdatableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, int) newUpdatableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4, int expectedSize) {
        return getDefaultFactory().<K>newUpdatableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map, int) newUpdatableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4,
            @Nonnull Map<? extends K, Short> map5, int expectedSize) {
        return getDefaultFactory().<K>newUpdatableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(
     * Consumer, int) newUpdatableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(@Nonnull
            Consumer<com.koloboke.function.ObjShortConsumer<K>> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().<K>newUpdatableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Object[], short[], int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull K[] keys, @Nonnull short[] values, int expectedSize) {
        return getDefaultFactory().<K>newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Object[],
     * Short[], int) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull K[] keys, @Nonnull Short[] values, int expectedSize) {
        return getDefaultFactory().<K>newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Iterable, Iterable, int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Short> values
            , int expectedSize) {
        return getDefaultFactory().<K>newUpdatableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new updatable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(
     * Map) newUpdatableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Short> map) {
        return getDefaultFactory().<K>newUpdatableMap(map);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Map,
     * Map) newUpdatableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2) {
        return getDefaultFactory().<K>newUpdatableMap(map1, map2);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Map, Map,
     * Map) newUpdatableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3) {
        return getDefaultFactory().<K>newUpdatableMap(map1, map2, map3);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Map, Map,
     * Map, Map) newUpdatableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4) {
        return getDefaultFactory().<K>newUpdatableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map) newUpdatableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4,
            @Nonnull Map<? extends K, Short> map5) {
        return getDefaultFactory().<K>newUpdatableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(
     * Consumer) newUpdatableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(@Nonnull
            Consumer<com.koloboke.function.ObjShortConsumer<K>> entriesSupplier
            ) {
        return getDefaultFactory().<K>newUpdatableMap(entriesSupplier);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Object[], short[]
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull K[] keys, @Nonnull short[] values) {
        return getDefaultFactory().<K>newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Object[],
     * Short[]) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull K[] keys, @Nonnull Short[] values) {
        return getDefaultFactory().<K>newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMap(Iterable, Iterable
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Short> values
            ) {
        return getDefaultFactory().<K>newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMapOf(Object, short
     * ) newUpdatableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map of the single specified mapping
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMapOf(
            K k1, short v1) {
        return getDefaultFactory().<K>newUpdatableMapOf(k1, v1);
    }

    /**
     * Constructs a new updatable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMapOf(Object, short,
     * Object, short) newUpdatableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map of the two specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMapOf(
            K k1, short v1, K k2, short v2) {
        return getDefaultFactory().<K>newUpdatableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new updatable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMapOf(Object, short,
     * Object, short, Object, short) newUpdatableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map of the three specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMapOf(
            K k1, short v1, K k2, short v2,
            K k3, short v3) {
        return getDefaultFactory().<K>newUpdatableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new updatable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMapOf(Object, short,
     * Object, short, Object, short, Object, short
     * ) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map of the four specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMapOf(
            K k1, short v1, K k2, short v2,
            K k3, short v3, K k4, short v4) {
        return getDefaultFactory().<K>newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new updatable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newUpdatableMapOf(Object, short,
     * Object, short, Object, short, Object, short,
     * Object, short) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new updatable map of the five specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newUpdatableMapOf(
            K k1, short v1, K k2, short v2,
            K k3, short v3, K k4, short v4,
            K k5, short v5) {
        return getDefaultFactory().<K>newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }


    
    
    
    

    /**
     * Constructs a new immutable map with the same mappings as the specified {@code map}.
     *
     * 
     * <p>If the specified map is an instance of {@code ObjShortMap} and has the same
     * {@linkplain ObjShortMap#keyEquivalence() key equivalence} with this factory (and thus
     * the constructed map), the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(
     * Map, int) newImmutableMap(map, expectedSize)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Short> map, int expectedSize) {
        return getDefaultFactory().<K>newImmutableMap(map, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Map,
     * Map, int) newImmutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2, int expectedSize) {
        return getDefaultFactory().<K>newImmutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Map, Map,
     * Map, int) newImmutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3, int expectedSize) {
        return getDefaultFactory().<K>newImmutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Map, Map,
     * Map, Map, int) newImmutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4, int expectedSize) {
        return getDefaultFactory().<K>newImmutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map, int) newImmutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4,
            @Nonnull Map<? extends K, Short> map5, int expectedSize) {
        return getDefaultFactory().<K>newImmutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(
     * Consumer, int) newImmutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(@Nonnull
            Consumer<com.koloboke.function.ObjShortConsumer<K>> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().<K>newImmutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Object[], short[], int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull K[] keys, @Nonnull short[] values, int expectedSize) {
        return getDefaultFactory().<K>newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Object[],
     * Short[], int) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull K[] keys, @Nonnull Short[] values, int expectedSize) {
        return getDefaultFactory().<K>newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Iterable, Iterable, int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Short> values
            , int expectedSize) {
        return getDefaultFactory().<K>newImmutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new immutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(
     * Map) newImmutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Short> map) {
        return getDefaultFactory().<K>newImmutableMap(map);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Map,
     * Map) newImmutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2) {
        return getDefaultFactory().<K>newImmutableMap(map1, map2);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Map, Map,
     * Map) newImmutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3) {
        return getDefaultFactory().<K>newImmutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Map, Map,
     * Map, Map) newImmutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4) {
        return getDefaultFactory().<K>newImmutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map) newImmutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Map<? extends K, Short> map1,
            @Nonnull Map<? extends K, Short> map2,
            @Nonnull Map<? extends K, Short> map3,
            @Nonnull Map<? extends K, Short> map4,
            @Nonnull Map<? extends K, Short> map5) {
        return getDefaultFactory().<K>newImmutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(
     * Consumer) newImmutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(@Nonnull
            Consumer<com.koloboke.function.ObjShortConsumer<K>> entriesSupplier
            ) {
        return getDefaultFactory().<K>newImmutableMap(entriesSupplier);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Object[], short[]
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull K[] keys, @Nonnull short[] values) {
        return getDefaultFactory().<K>newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Object[],
     * Short[]) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull K[] keys, @Nonnull Short[] values) {
        return getDefaultFactory().<K>newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMap(Iterable, Iterable
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMap(
            @Nonnull Iterable<? extends K> keys, @Nonnull Iterable<Short> values
            ) {
        return getDefaultFactory().<K>newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMapOf(Object, short
     * ) newImmutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map of the single specified mapping
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMapOf(
            K k1, short v1) {
        return getDefaultFactory().<K>newImmutableMapOf(k1, v1);
    }

    /**
     * Constructs a new immutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMapOf(Object, short,
     * Object, short) newImmutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map of the two specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMapOf(
            K k1, short v1, K k2, short v2) {
        return getDefaultFactory().<K>newImmutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new immutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMapOf(Object, short,
     * Object, short, Object, short) newImmutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map of the three specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMapOf(
            K k1, short v1, K k2, short v2,
            K k3, short v3) {
        return getDefaultFactory().<K>newImmutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new immutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMapOf(Object, short,
     * Object, short, Object, short, Object, short
     * ) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map of the four specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMapOf(
            K k1, short v1, K k2, short v2,
            K k3, short v3, K k4, short v4) {
        return getDefaultFactory().<K>newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new immutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashObjShortMapFactory#newImmutableMapOf(Object, short,
     * Object, short, Object, short, Object, short,
     * Object, short) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
     * @param <K> the key type of the returned map 
     
     * @return a new immutable map of the five specified mappings
     */
    @Nonnull
    public static <K> HashObjShortMap<K> newImmutableMapOf(
            K k1, short v1, K k2, short v2,
            K k3, short v3, K k4, short v4,
            K k5, short v5) {
        return getDefaultFactory().<K>newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    private HashObjShortMaps() {}
}
