package com.ksyun.kmr.hadoop.fs.ks3;


import com.ksyun.kmr.hadoop.fs.ks3.committer.CommitInfoFileCommitter;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.EngineShutter;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.conveyor.CopyDirAction;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.conveyor.DestroyAction;
import com.ksyun.kmr.hadoop.fs.ks3.requestbuilder.ListDir;
import com.ksyun.kmr.hadoop.fs.ks3.requestbuilder.ListFileStatus;
import com.ksyun.kmr.hadoop.fs.ks3.requestbuilder.ListLocatedFileStatus;
import com.ksyun.ks3.dto.ObjectMetadata;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.*;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.StopWatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.NoSuchPaddingException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Date;
import java.util.EnumSet;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Ks3FileSystem: Hadoop File System Ks3 Implementation
 * KS3 URI Format ks3://[AccessKey]:[AccessSecret]@[Bucket]/[objectKey]
 * <p>
 * About folder structure:
 * using object key a/b.txt to represent a file within folder
 * There will not having fake object to represent folder if the folder is not empty;
 * but if a folder is empty, then will have a empty object to represent the folder, i.e. "a/c/"
 */
public class Ks3FileSystem extends FileSystem {
    private static final Logger LOG = LoggerFactory.getLogger(Ks3FileSystem.class);

    public URI uri;
    public Path workingDir;
    private Ks3FileSystemStore store;
    public long ks3BlockSize;
    public boolean ks3CheckSub;

    public Ks3FileSystemStore getStore() {
        return this.store;
    }

    /**
     * Called after a new FileSystem instance is constructed.
     *
     * @param name a uri whose authority section names the host, port, etc.
     *             for this FileSystem
     * @param conf the configuration
     */
    @Override
    public void initialize(URI name, Configuration conf) throws IOException {
        super.initialize(name, conf);
        this.uri = URI.create(name.getScheme() + "://" + name.getAuthority());
        this.workingDir = new Path("/user", System.getProperty("user.name")).makeQualified(this.uri, null);
        this.store = new Ks3FileSystemStore();
        this.store.initialize(uri, workingDir, statistics, conf);
        this.ks3BlockSize = conf.getLong(Constants.KS3_BLOCK_SIZE, Constants.DEFAULT_KS3_BLOCK_SIZE);
        this.ks3CheckSub = conf.getBoolean(Constants.KS3_CHECK_SUBDIR_IN_GETFILESTATUS, Constants.DEFAULT_KS3_CHECK_SUBDIR_IN_GETFILESTATUS);
        setConf(conf);
    }

    @Override
    public String getScheme() {
        return Constants.FS_KS3;
    }

    @Override
    public URI getUri() {
        return this.uri;
    }

    public Ks3FileSystem() {
        super();
    }


    @Override
    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        final FileStatus fileStatus = getFileStatus(f);
        if (fileStatus.isDirectory()) {
            throw new FileNotFoundException("Can't open " + f + " because it is a directory");
        }

        Ks3InputStream ks3InputStream = new Ks3InputStream(pathToKey(f), fileStatus.getLen(), this.store, this.statistics);
        return new FSDataInputStream(ks3InputStream);
    }

    @Override
    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite,
                                     int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        Path targetPath = toQ(f);

        if (!overwrite && exists(targetPath)) {
            throw new FileAlreadyExistsException(targetPath + " already exists");
        }

        Path targetParentPath = targetPath.getParent();
        if (targetParentPath != null){
            mkdirs(targetParentPath);
        }

        OutputStream ks3OutputStream;
        String keySeed = this.getConf().get(Constants.KS3_CLIENT_ENCRYPT_KEY_SEED, null);
        int keySize = this.getConf().getInt(Constants.KS3_CLIENT_ENCRYPT_KEY_SIZE,
                Constants.DEFAULT_KS3_CLIENT_ENCRYPT_KEY_SIZE);
        if (targetPath.toString().contains(CommitInfoFileCommitter.PENDING_DIR_NAME)) {
            String finalKey = pathToKey(getDirectOutputPath(targetPath));
            String infoKey = pathToKey(targetPath);
            ks3OutputStream = new Ks3OutputStream(store, finalKey, infoKey);
        } else {
            String finalKey = pathToKey(targetPath);
            ks3OutputStream = new Ks3OutputStream(store, finalKey);
        }
        if (null != keySeed) {
            try {
                ks3OutputStream = Utils.getCipherOutputStream(ks3OutputStream, keySeed, keySize);
            } catch (NoSuchPaddingException | NoSuchAlgorithmException | InvalidKeyException e) {
                throw new IOException("Cipher exception: Can not get CipherOutputStream.", e);
            }
        }
        return new FSDataOutputStream(ks3OutputStream, statistics);
    }

    /**
     * {@inheritDoc}
     * @throws FileNotFoundException if the parent directory is not present -or
     * is not a directory.
     */
    @Override
    public FSDataOutputStream createNonRecursive(Path path,
                                                 FsPermission permission,
                                                 EnumSet<CreateFlag> flags,
                                                 int bufferSize,
                                                 short replication,
                                                 long blockSize,
                                                 Progressable progress) throws IOException {
        Path parent = path.getParent();
        if (parent != null) {
            // expect this to raise an exception if there is no parent
            if (!getFileStatus(parent).isDirectory()) {
                throw new FileAlreadyExistsException("Not a directory: " + parent);
            }
        }
        return create(path, permission, flags.contains(CreateFlag.OVERWRITE),
          bufferSize, replication, blockSize, progress);
    }

    @Override
    public FSDataOutputStream append(Path f, int bufferSize,
                                     Progressable progress) throws IOException {
        throw new IOException("KS3 File System not supporting Append operation");
    }

    /**
     * Renames Path src to Path dst
     * <p>
     * Warning: KS3 does not support renames. This method does a copy which can
     * take KS3 some time to execute with large files and directories. Since
     * there is no Progressable passed in, this can time out jobs.
     * Cases    srcType    dstType          Result
     * 1        file/dir    N/A             Copy
     * 2        file/dir    file            Fail
     * 3        file        dir             Fail
     * 4        dir         dir(not empty)  Fail   (whatever src type, is dst is not an empty dir, then will fail)
     * 5        dir         dir(empty)      copy
     *
     * @param src path to be renamed
     * @param dst new path after rename
     * @return true if rename is successful
     * @throws IOException on failure
     */
    @Override
    public boolean rename(Path src, Path dst) throws IOException {
        StopWatch sw = new StopWatch().start();

        try {
            return rename0(src, dst);
        } finally {
            long costTime = sw.now(TimeUnit.MICROSECONDS);
            LOG.info("rename cost " + costTime);
        }
    }

    public boolean rename0(Path src, Path dst) throws IOException {
        String srcKey = pathToKey(src);
        String dstKey = pathToKey(dst);

        if (srcKey.isEmpty() || dstKey.isEmpty()) {
            return false;
        }

        Ks3FileStatus srcStatus;
        try {
            srcStatus = getFileStatus(src);
        } catch (FileNotFoundException e) {
            LOG.error("rename: src not found {}", src);
            return false;
        }

        if (srcKey.equals(dstKey)) {
            return srcStatus.isFile();
        }

        Ks3FileStatus dstStatus = null;
        try {
            dstStatus = getFileStatus(dst);

            // case 2
            if (dstStatus.isFile()) {
                return false;
            }

            // case 4
            if (dstStatus.isDirectory() && isNotEmptyDir(dstStatus)) {
                return false;
            }

            // case 3
            if (srcStatus.isFile() && dstStatus.isDirectory()) {
                return false;
            }
        } catch (FileNotFoundException e) {
            // Parent must exist
            Path parent = dst.getParent();
            if (!pathToKey(parent).isEmpty()) {
                try {
                    Ks3FileStatus dstParentStatus = getFileStatus(parent);
                    if (!dstParentStatus.isDirectory()) {
                        return false;
                    }
                } catch (FileNotFoundException e2) {
                    return false;
                }
            }
        }

        if (srcStatus.isFile()) {
            // only case 1 or 2 can reach here.
            store.copyObject(srcKey, dstKey);
            delete(src, false, srcStatus);
        } else {
            // case 1 or 5
            // This is a directory to directory copy
            if (!dstKey.endsWith("/")) {
                dstKey = dstKey + "/";
            }

            if (!srcKey.endsWith("/")) {
                srcKey = srcKey + "/";
            }

            // Verify dst is not a child of the src directory
            if (dstKey.startsWith(srcKey)) {
                return false;
            }

            ListDir listDir = new ListDir(store, srcKey, true);
            AtomicReference<Exception> exceptionAtomicReference = new AtomicReference<>();
            DestroyAction destroyAction = new DestroyAction(store, exceptionAtomicReference);
            CopyDirAction renameAction = new CopyDirAction(store, srcKey, dstKey, exceptionAtomicReference);

            try {
                renameAction.startEngines();
                destroyAction.startEngines();
                renameAction.sink = (data) -> {
                    String copiedSrcKey = data.getLeft();
                    destroyAction.sendData(copiedSrcKey);
                };

                listDir.genStream(exceptionAtomicReference).forEach(batch -> {
                    renameAction.run(batch);
                });
            } finally {
                EngineShutter.shutdownAll(renameAction, destroyAction);
            }
        }

        return true;
    }

    @Override
    public boolean delete(Path f, boolean recursive) throws IOException {
        Ks3FileStatus status;
        try {
            status = getFileStatus(f);
        } catch (FileNotFoundException e) {
            return false;
        }

        return delete(f, recursive, status);
    }


    public boolean delete(Path f, boolean recursive, Ks3FileStatus status) throws IOException {
        String key = pathToKey(f);

        if (status.isDirectory()) {
            boolean isNotEmptyDirectory = isNotEmptyDir(status);
            if (!recursive && isNotEmptyDirectory) {
                throw new IOException("Path is a folder: " + f +
                        " and it is not an empty directory");
            }

            if (!key.endsWith("/")) {
                key = key + "/";
            }

            if (key.equals("/")) {
                return false;
            }

            if (isNotEmptyDirectory) {
                store.deleteDir(key, false);
            } else {
                store.deleteObject(status.getDirKey());
            }
        } else {
            store.deleteObject(key);
        }

        return true;
    }

    // 必须要支持f是单个文件的情况
    @Override
    public FileStatus[] listStatus(Path f) throws IOException {
        return new ListFileStatus(this, f, false).listStatus();
    }

    public FileStatus[] listStatusWithFilter(Path f, boolean isFlat, PathFilter... filters) throws IOException {
        return new ListFileStatus(this, f, isFlat).listStatus(filters);
    }

    @Override
    public RemoteIterator<LocatedFileStatus> listFiles(final Path f, final boolean recursive) throws IOException {
        return ListLocatedFileStatus.genIterator(this, f, recursive, true);
    }

    @Override
    protected RemoteIterator<LocatedFileStatus> listLocatedStatus(final Path f,
                                                                  final PathFilter filter) throws IOException {
        return ListLocatedFileStatus.genIterator(this, f, false, false, filter);
    }

    @Override
    public void setWorkingDirectory(Path new_dir) {
        this.workingDir = new_dir;
    }

    @Override
    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    @Override
    public boolean mkdirs(Path f, FsPermission permission) throws IOException {
        f = toQ(f);

        try {
            FileStatus fileStatus = getFileStatus(f, false);

            if (fileStatus.isDirectory()) {
                return true;
            } else {
                throw new FileAlreadyExistsException("Path is a file: " + f);
            }
        } catch (FileNotFoundException e) {
            validatePath(f);
            createFakeDirectory(pathToKey(f));
            return true;
        }
    }

    public void validatePath(Path f) throws IOException{
        Path fPart = f.getParent();
        List<String> toCreateDir = new LinkedList<>();

        while (fPart != null && fPart.getParent() != null) {
            try {
                FileStatus fileStatus = getFileStatus(fPart, false);
                if (fileStatus.isFile()) {
                    throw new FileAlreadyExistsException(String.format(
                            "Can't make directory for path '%s' since it is a file.",
                            fPart));
                }

                if (fileStatus.isDirectory()) {
                    break;
                }
            } catch (FileNotFoundException fnfe) {
                toCreateDir.add(pathToKey(fPart));
            }
            fPart = fPart.getParent();
        };

        for (String key : toCreateDir){
            createFakeDirectory(key);
        }
    }

    @Override
    public Ks3FileStatus getFileStatus(Path f) throws IOException {
        return getFileStatus(f, ks3CheckSub);
    }

    // 这个操作是个重量级操作，要避免
    public Ks3FileStatus getFileStatus(Path f, boolean checkSub) throws IOException {
        String key = pathToKey(f);
        Path qualifiedPath = f.makeQualified(uri, workingDir);

        if (key.isEmpty()) {
            //for the root path
            return new Ks3FileStatus(qualifiedPath, key);
        }

        ObjectMetadata meta = store.getMetadata(key);

        if (meta != null) {
            return new Ks3FileStatus(meta.getContentLength(),
                    dateToLong(meta.getLastModified()),
                    qualifiedPath,
                    ks3BlockSize, meta.getETag());
        }

        // Must to add delimiter before using 'listSubPaths' function to get sub path.
        // It's have two key, 'test' and 'test123'. To get sub paths of 'test', if don't
        // add delimiter at end of key, 'test123' will be a sub path of the key 'test'.
        key = key + "/";

        meta = store.getMetadata(key);
        if (meta != null) {
            if (objectRepresentsDirectory(key, meta.getContentLength())) {
                return new Ks3FileStatus(qualifiedPath, key);
            } else {
                throw new RuntimeException("filename with \"/\" suffix, please change it to a normal name");
            }
        }

        if (checkSub) {
            // 因为之前已经确定没有目录文件存在，所以这里可以直接使用store.listAllObjects(key, 1)，而不用担心listAllObjects可能会包含key本身的情况
            if (store.listAllObjects(key, 1).isNotEmpty()){
                return new Ks3FileStatus(qualifiedPath, key, true);
            }
        }

        throw new FileNotFoundException("No such file or directory: " + f);
    }

    @Override
    public long getDefaultBlockSize(Path f) {
        return ks3BlockSize;
    }

    public boolean objectRepresentsDirectory(final String name, final long size) {
        return !name.isEmpty() && name.endsWith("/") && size == 0L;
    }

    public static long dateToLong(final Date date) {
        if (date == null) {
            return 0L;
        }

        return date.getTime();
    }

    public void createFakeDirectory(String objectName) {
        if ("".equals(objectName)){
            return;
        }

        if (!objectName.endsWith("/")) {
            objectName += "/";
        }

        this.store.createEmptyObject(objectName);
    }

    public String pathToKey(Path path) {
        return pathToKey(path, workingDir);
    }

    public static String pathToKey(Path path, Path workingDir) {
        if (!path.isAbsolute()) {
            path = new Path(workingDir, path);
        }

        URI u = path.toUri();

        if (u.getScheme() != null && u.getPath().isEmpty()) {
            return "";
        }

        return u.getPath().substring(1);
    }

    public Path keyToPath(String key) {
        return new Path("/" + key);
    }

    public Path getDirectOutputPath(Path qualifiedPath) {
        String pathName = qualifiedPath.toString();
        if (pathName.contains(CommitInfoFileCommitter.PENDING_DIR_NAME)) {
            String[] splitedPath1 = pathName.split(CommitInfoFileCommitter.PENDING_DIR_NAME);
            String[] splitedPath2 = pathName.split(CommitInfoFileCommitter.TASK_DIR_NAME);
            String[] splitedPath3 = splitedPath2[1].split("/");
            String filename = StringUtils.join(Arrays.copyOfRange(splitedPath3, 2, splitedPath3.length), "/");
            qualifiedPath = new Path(splitedPath1[0], filename);
        }

        return qualifiedPath;
    }

    public Path toQ(Path f){
        return f.makeQualified(uri, workingDir);
    }

    public boolean isNotEmptyDir(Ks3FileStatus ks3FileStatus) throws IOException{
        if (ks3FileStatus.isFile()){
            throw new RuntimeException("is file, not dir");
        } else {
            if (ks3FileStatus.isCheckedIsNotEmpty()){
                return true;
            } else {
                return checkNotEmptyDir(ks3FileStatus.getPath());
            }
        }
    }

    public boolean checkNotEmptyDir(Path path){
        String dirKey = pathToKey(path) + "/";
        ListDir listDir = new ListDir(store, dirKey, true, 2, 2);
        List<String> rs = listDir.listAll().getObjectKeys();
        if (rs.contains(dirKey)){
            return rs.size() > 1;
        } else {
            return rs.size() > 0;
        }
    }

}
