package com.ksyun.kmr.hadoop.fs.ks3;

import com.google.common.util.concurrent.RateLimiter;
import com.ksyun.kmr.hadoop.FileSystemStoreContext;
import com.ksyun.kmr.hadoop.fs.ks3.bean.CopyPartBean;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.MultiActionEngine;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.conveyor.CopyAction;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.conveyor.DestroyAction;
import com.ksyun.kmr.hadoop.fs.ks3.requestbuilder.ListDir;
import com.ksyun.ks3.AutoAbortInputStream;
import com.ksyun.ks3.dto.*;
import com.ksyun.ks3.exception.Ks3ClientException;
import com.ksyun.ks3.exception.Ks3ServiceException;
import com.ksyun.ks3.http.HttpClientConfig;
import com.ksyun.ks3.service.Ks3;
import com.ksyun.ks3.service.Ks3Client;
import com.ksyun.ks3.service.Ks3ClientConfig;
import com.ksyun.ks3.service.common.StorageClass;
import com.ksyun.ks3.service.request.*;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.URI;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Ks3FileSystemStore {
    private static final Logger LOG = LoggerFactory.getLogger(Ks3FileSystemStore.class);
    public Ks3 ks3Client;
    public String bucket;
    private String ks3StorageClass;
    private boolean isSupportKs3Storage;
    private Configuration conf;
    private URI uri;
    private Path workingDir;
    private Ks3ClientConfig ks3config;
    private static SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    private long overallCopyMaxLen;
    private int copyBlockSize;
    private FileSystem.Statistics statistics;
    public FileSystemStoreContext context;
    public int parallel_delete_pool_size;
    public int parallel_delete_thread_size;
    public int parallel_delete_speed_limit;
    public int parallel_copy_pool_size;
    public int parallel_copy_thread_size;
    public int parallel_copy_speed_limit;
    public int parallel_copy_part_pool_size;
    public int parallel_copy_part_thread_size;
    public int parallel_copy_part_speed_limit;
    public int parallel_commit_pool_size;
    public int parallel_commit_thread_size;
    public int parallel_commit_speed_limit;
    public int parallel_unser_pool_size;
    public int parallel_unser_thread_size;
    public int parallel_unser_speed_limit;
    public int greater_than_this_use_parallel_when_destroy;
    public int parallel_upload_part_pool_size;
    public int parallel_upload_part_thread_size;
    public int parallel_upload_part_limit;
    public int blockSize;

    public int getBlockSize() {
        return blockSize;
    }

    public FileSystem.Statistics getStatistics() {
        return statistics;
    }

    public static Pair<String, String> getBucketAndEndpoint(URI uri, String defaultBucket){
        String host = uri.getHost();

        String bucket = host;
        String endpoint = defaultBucket;

        Pattern pattern = Pattern.compile("\\.(kss|ks3)[\\w-]*\\.ksyun(cs)?\\.com$");
        Matcher matcher = pattern.matcher(host);
        if (matcher.find()) {
            endpoint = matcher.group().substring(1);
            bucket = matcher.replaceAll("");
        }

        return Pair.of(bucket, endpoint);
    }

    public void initialize(URI uri, Path workingDir, FileSystem.Statistics statistics, Configuration conf) {
        /**
         * host格式为bucket 或者 bucket.endpoint
         * 如果bucket在同region KS3，则uri中host为bucket
         * 如果bucket在不同region KS3，则uri中host为bucket.endpoint，其中endpoint为对应region的KS3外网域名
         */
        this.uri = uri;
        this.workingDir = workingDir;
        this.statistics = statistics;

//        String defaultPort = conf.getTrimmed(Constants.ENDPOINT, Constants.DEFAULT_ENDPOINT);
        String endpointEnv = System.getenv().get(Constants.ENDPOINT_ENV);
        String defaultPort = conf.getTrimmed(Constants.ENDPOINT, endpointEnv);
        if (defaultPort == null) {
            defaultPort = Constants.DEFAULT_ENDPOINT;
        }
        Pair<String, String> result = getBucketAndEndpoint(uri, defaultPort);
        bucket = result.getKey();
        String endpoint = result.getValue();

        Ks3ClientConfig ks3config = new Ks3ClientConfig();
        ks3config.setEndpoint(endpoint);
        ks3config.setProtocol(Ks3ClientConfig.PROTOCOL.http);
        ks3config.setSignerClass(conf.getTrimmed(Constants.CLIENT_SIGNER, Constants.DEFAULT_CLIENT_SINGER));
        ks3config.setPathStyleAccess(intValToBoolean(conf.getInt(Constants.CLIENT_URL_FORMAT, Constants.DEFAULT_CLIENT_URL_FORMAT)));


        HttpClientConfig hconfig = new HttpClientConfig();
        hconfig.setConnectionTimeOut(conf.getInt(Constants.CONNECTION_TIMEOUT, Constants.DEFAULT_CONNECTION_TIMEOUT));
        hconfig.setSocketTimeOut(conf.getInt(Constants.SOCKET_TIMEOUT, Constants.DEFAULT_SOCKET_TIMEOUT));
        hconfig.setSocketSendBufferSizeHint(conf.getInt(Constants.SOCKET_SEND_BUFFER_SIZE_HINT, Constants.DEFAULT_SOCKET_SEND_BUFFER_SIZE));
        hconfig.setSocketReceiveBufferSizeHint(conf.getInt(Constants.SOCKET_RECEIVE_BUFFER_SIZE_HINT, Constants.DEFAULT_SOCKET_RECEIVE_BUFFER_SIZE));
        hconfig.setMaxRetry(conf.getInt(Constants.MAX_ERROR_RETRIES, Constants.DEFAULT_MAX_ERROR_RETRIES));
        hconfig.setConnectionTTL(conf.getInt(Constants.CONNECTION_TTL, -1));
        hconfig.setMaxConnections(conf.getInt(Constants.MAXIMUM_CONNECTIONS, Constants.DEFAULT_MAXIMUM_CONNECTIONS));
        hconfig.setProxyHost(conf.getTrimmed(Constants.PROXY_HOST));
        hconfig.setProxyPort(conf.getInt(Constants.PROXY_PORT, -1));
        hconfig.setProxyDomain(conf.getTrimmed(Constants.PROXY_DOMAIN));
        hconfig.setProxyUserName(conf.getTrimmed(Constants.PROXY_USERNAME));
        hconfig.setProxyPassWord(conf.getTrimmed(Constants.PROXY_PASSWORD));
        hconfig.setProxyWorkStation(conf.getTrimmed(Constants.PROXY_WORKSTATION));
        hconfig.setPreemptiveBasicProxyAuth(conf.getBoolean(Constants.IS_PREEMPTIVE_BASIC_PROXY_AUTH, Constants.DEFAULT_PREEMPTIVE_BASIC_PROXY_AUTH));
        ks3config.setHttpClientConfig(hconfig);
        this.ks3config = ks3config;

        this.ks3StorageClass = conf.getTrimmed(Constants.STORAGE_CLASS, Constants.DEFAULT_STORAGE_CLASS);
        this.isSupportKs3Storage = conf.getBoolean(Constants.IS_SUPPORT_KS3_STORAGE, Constants.DEFAULT_IS_SUPPORT_KS3_STORAGE);


        this.bucket = bucket;
        this.conf = conf;

        this.overallCopyMaxLen = this.conf.getLong(Constants.KS3_OVERALL_COPY_TOTAL_LIMIT,
                Constants.DEFAULT_KS3_OVERALL_COPY_LIMIT_SIZE);

        copyBlockSize = this.conf.getInt(Constants.KS3_MULTI_PART_COPY_BLOCK_SIZE,
                Constants.DEFAULT_KS3_PART_COPY_BLOCK_SIZE);

        initContext();
        renewKs3Client();
        initParallelConfig();
    }

    public void initParallelConfig(){
        parallel_delete_pool_size = conf.getInt(Constants.PARALLEL_DELETE_POOL_SIZE, Constants.DEFAULT_PARALLEL_DELETE_POOL_SIZE);
        parallel_delete_thread_size = conf.getInt(Constants.PARALLEL_DELETE_THREAD_SIZE, Constants.DEFAULT_PARALLEL_DELETE_THREAD_SIZE);
        parallel_delete_speed_limit = conf.getInt(Constants.PARALLEL_DELETE_SPEED_LIMIT, Constants.DEFAULT_PARALLEL_DELETE_SPEED_LIMIT);

        parallel_copy_pool_size = conf.getInt(Constants.PARALLEL_COPY_POOL_SIZE, Constants.DEFAULT_PARALLEL_COPY_POOL_SIZE);
        parallel_copy_thread_size = conf.getInt(Constants.PARALLEL_COPY_THREAD_SIZE, Constants.DEFAULT_PARALLEL_COPY_THREAD_SIZE);
        parallel_copy_speed_limit = conf.getInt(Constants.PARALLEL_COPY_SPEED_LIMIT, Constants.DEFAULT_PARALLEL_COPY_SPEED_LIMIT);

        parallel_copy_part_pool_size = conf.getInt(Constants.PARALLEL_COPY_PART_POOL_SIZE, Constants.DEFAULT_PARALLEL_COPY_PART_POOL_SIZE);
        parallel_copy_part_thread_size = conf.getInt(Constants.PARALLEL_COPY_PART_THREAD_SIZE, Constants.DEFAULT_PARALLEL_COPY_PART_THREAD_SIZE);
        parallel_copy_part_speed_limit = conf.getInt(Constants.PARALLEL_COPY_PART_SPEED_LIMIT, Constants.DEFAULT_PARALLEL_COPY_PART_SPEED_LIMIT);

        parallel_commit_pool_size = conf.getInt(Constants.PARALLEL_COMMIT_POOL_SIZE, Constants.DEFAULT_PARALLEL_COMMIT_POOL_SIZE);
        parallel_commit_thread_size = conf.getInt(Constants.PARALLEL_COMMIT_THREAD_SIZE, Constants.DEFAULT_PARALLEL_COMMIT_THREAD_SIZE);
        parallel_commit_speed_limit = conf.getInt(Constants.PARALLEL_COMMIT_SPEED_LIMIT, Constants.DEFAULT_PARALLEL_COMMIT_SPEED_LIMIT);

        parallel_unser_pool_size = conf.getInt(Constants.PARALLEL_UNSER_POOL_SIZE, Constants.DEFAULT_PARALLEL_UNSER_POOL_SIZE);
        parallel_unser_thread_size = conf.getInt(Constants.PARALLEL_UNSER_THREAD_SIZE, Constants.DEFAULT_PARALLEL_UNSER_THREAD_SIZE);
        parallel_unser_speed_limit = conf.getInt(Constants.PARALLEL_UNSER_SPEED_LIMIT, Constants.DEFAULT_PARALLEL_UNSER_SPEED_LIMIT);

        greater_than_this_use_parallel_when_destroy = conf.getInt(Constants.GREATER_THAN_THIS_USE_PARALLEL_WHEN_DESTROY, Constants.DEFAULT_GREATER_THAN_THIS_USE_PARALLEL_WHEN_DESTROY);

        parallel_upload_part_pool_size = conf.getInt(Constants.PARALLEL_UPLOAD_PART_POOL_SIZE, Constants.DEFAULT_PARALLEL_UPLOAD_PART_POOL_SIZE);
        parallel_upload_part_thread_size = conf.getInt(Constants.PARALLEL_UPLOAD_PART_THREAD_SIZE, Constants.DEFAULT_PARALLEL_UPLOAD_PART_THREAD_SIZE);
        parallel_upload_part_limit = conf.getInt(Constants.PARALLEL_UPLOAD_PART_SPEED_LIMIT, Constants.DEFAULT_PARALLEL_UPLOAD_PART_SPEED_LIMIT);

        blockSize = conf.getInt(Constants.MULTIPAET_UPLOADS_BLOCK_SIZE, Constants.DEFAULT_MULTIPAET_UPLOADS_BLOCK_SIZE);
    }

    public void initContext(){
        FileSystemStoreContext context = new FileSystemStoreContext();
        context.statistics = this.statistics;
        context.ks3RequestAttemptsMaximum = conf.getInt(Constants.KS3_REQUEST_ATTEMPTS_MAXIMUM, Constants.DEFAULT_KS3_REQUEST_ATTEMPTS_MAXIMUM);
        this.context = context;
    }

    public int getCopyBlockSize() {
        return copyBlockSize;
    }

    public long getOverallCopyMaxLen() {
        return overallCopyMaxLen;
    }

    public void renewKs3Client(){
        Ks3AuthorizationProvider provider = new Ks3AuthorizationProvider(this.uri, this.conf);
        this.ks3Client = new Ks3Client(provider.getCredentials()).withKs3config(this.ks3config);
    }

    public ObjectMetadata getMetadata(String objectKey) {
        return getMetadata(objectKey, null);
    }

    public ObjectMetadata getMetadata(String objectKey, RateLimiter rateLimiter) {
        HeadObjectResult result = new RetryHandler(true, rateLimiter, 1, context).retryProcess(() -> {
            HeadObjectRequest request = new HeadObjectRequest(this.bucket, objectKey);
            return this.ks3Client.headObject(request);
        });

        if (result != null){
            return result.getObjectMetadata();
        }

        return null;
    }

    public ListObjectsResult listAllSubPaths(String objectKey) {
        return listAllSubPaths(objectKey, 0);
    }

    public ListObjectsResult listAllSubPaths(String objectKey, int limit) {
        int pageNum = 1000;
        return listAllSubPaths(objectKey, limit, pageNum);
    }

    // 只列出一级
    public ListObjectsResult listAllSubPaths(String objectKey, int limit, int pageNum) {
        ListDir listDir = new ListDir(this, objectKey, false, limit, pageNum);
        return listDir.listAll();
    }

    public ListObjectsResult listAllObjects(String objectKey) {
        return listAllObjects(objectKey, 0);
    }

    public ListObjectsResult listAllObjects(String objectKey, int limit) {
        int pageNum = 1000;
        return listAllObjects(objectKey, limit, pageNum);
    }

    // 列出全部层级
    public ListObjectsResult listAllObjects(String objectKey, int limit, int pageNum) {
        ListDir listDir = new ListDir(this, objectKey, true, limit, pageNum);
        return listDir.listAll();
    }

    public AutoAbortInputStream getObject(String objectKey, long contentLength, long pos) throws EOFException {
        if (pos < 0) {
            throw new EOFException(FSExceptionMessages.NEGATIVE_SEEK
                    + " " + pos);
        }
        if (contentLength > 0 && pos > contentLength - 1) {
            throw new EOFException(
                    FSExceptionMessages.CANNOT_SEEK_PAST_EOF
                            + " " + pos);
        }

        GetObjectResult result = new RetryHandler(1, context).retryProcess(() -> {
            GetObjectRequest request = new GetObjectRequest(this.bucket, objectKey);
            request.setRange(pos, contentLength - 1);
            return this.ks3Client.getObject(request);
        });

        return result.getObject().getObjectContent();
    }

    public AutoAbortInputStream getObject(String objectKey) {
        return getObject(objectKey, null);
    }

    public AutoAbortInputStream getObject(String objectKey, RateLimiter rateLimiter) {
        GetObjectResult result = new RetryHandler(rateLimiter, 1, context).retryProcess(() -> {
            GetObjectRequest request = new GetObjectRequest(this.bucket, objectKey);
            return this.ks3Client.getObject(request);
        });

        return result.getObject().getObjectContent();
    }

    public void putObject(String objectKey, File file) {
        new RetryHandler(3, context).retryProcess(() -> {
            PutObjectRequest request = new PutObjectRequest(this.bucket, objectKey, file);
            this.ks3Client.putObject(request);
        });
    }

    public void putObject(String objectKey, byte[] bytes) {
        new RetryHandler(3, context).retryProcess(() -> {
            ObjectMetadata meta = new ObjectMetadata();
            meta.setContentLength(bytes.length);
            PutObjectRequest request = new PutObjectRequest(bucket, objectKey, new ByteArrayInputStream(bytes), meta);
            this.ks3Client.putObject(request);
        });
    }

    public void putObject(Ks3BlockBuffer ks3BlockBuffer, int blockLength) {
        new RetryHandler(3, context).retryProcess(() -> {
            ks3BlockBuffer.refreshInputData(blockLength);

            ObjectMetadata meta = new ObjectMetadata();
            meta.setContentLength(blockLength);
            PutObjectRequest request = new PutObjectRequest(bucket, ks3BlockBuffer.getKey(), ks3BlockBuffer.inBuffer, meta);

            this.ks3Client.putObject(request);
        });
    }

    public void deleteDir(String key, boolean skip404){
        ListDir listDir = new ListDir(this, key, true);
        DestroyAction destroyAction = new DestroyAction(this);
        destroyAction.setSkip404(skip404);

        try {
            destroyAction.startEngines();
            listDir.genStream(destroyAction.getExceptionAtomicReference()).forEach(batch -> {
                destroyAction.run(batch);
            });
        } finally {
            destroyAction.shutdown();
        }
    }

    public void deleteObject(String objectKey) {
        deleteObject(objectKey, null);
    }

    public void deleteObject(String objectKey, RateLimiter rateLimiter) {
        new RetryHandler(rateLimiter, 3, context).retryProcess(() -> {
            this.ks3Client.deleteObject(bucket, objectKey);
        });
    }

    public void deleteObjectSkip404(String objectKey) {
        deleteObjectSkip404(objectKey, null);
    }

    public void deleteObjectSkip404(String objectKey, RateLimiter rateLimiter) {
        new RetryHandler(true, rateLimiter, 3, context).retryProcess(() -> {
            this.ks3Client.deleteObject(bucket, objectKey);
        });
    }

    public void deleteObjects(List<String> objectKeys) {
        if (objectKeys.size() > greater_than_this_use_parallel_when_destroy) {
            parallelDeleteObjects(objectKeys);
        } else {
            serialDeleteObjects(objectKeys);
        }
    }

    public void serialDeleteObjects(List<String> objectKeys) {
        for (String objectKey : objectKeys) {
            deleteObject(objectKey);
        }
    }

    public void parallelDeleteObjects(List<String> objectKeys) {
        if (objectKeys.isEmpty()){
            return;
        }

        parallelDeleteObjects((MultiActionEngine engine) -> {
            for (String key : objectKeys) {
                if (!engine.sendData(Collections.singletonMap("key", key))){
                    break;
                }
            }
        });
    }

    public static interface EngineSender {
        public void run(MultiActionEngine engine);
    }

    public void parallelDeleteObjects(EngineSender runnable) {
        parallelDeleteObjects(runnable, parallel_delete_speed_limit);
    }

    public void parallelDeleteObjects(EngineSender runnable, int rateLimit){
        DestroyAction destroyAction = new DestroyAction(this, rateLimit);

        try {
            destroyAction.startEngines();
            runnable.run(destroyAction.source());
        } finally {
            destroyAction.shutdown();
        }
    }

    public void copyObject(String srcObjectKey, String dstObjectKey) {
        ObjectMetadata metaData = getMetadata(srcObjectKey);
        long contentLength = metaData.getContentLength();

        if (contentLength > overallCopyMaxLen) {
            copyObjects(Collections.singletonList(Pair.of(srcObjectKey, dstObjectKey)));
        } else {
            copyObject(metaData, srcObjectKey, dstObjectKey);
        }
    }

    private StorageClass getStorageClass(ObjectMetadata meta) {
        StorageClass storageClass = null;
        if (isSupportKs3Storage) {
            String srcStorageClass = meta.getStorageClass();
            if (srcStorageClass == null) {
                storageClass = StorageClass.Standard;
            } else if (srcStorageClass.equals("STANDARD_IA")) {
                storageClass = StorageClass.StandardInfrequentAccess;
            }
        }
        return storageClass;
    }

    public void copyObject(ObjectMetadata srcMeta, String srcObjectKey, String dstObjectKey) {
        copyObject(srcMeta, srcObjectKey, dstObjectKey, null);
    }

    public void copyObject(ObjectMetadata srcMeta, String srcObjectKey, String dstObjectKey, RateLimiter rateLimiter) {
        new RetryHandler(rateLimiter, 3, context).retryProcess(() -> {
            CopyObjectRequest request = new CopyObjectRequest(bucket, dstObjectKey, bucket, srcObjectKey);
            StorageClass storageClass = this.getStorageClass(srcMeta);
            if (storageClass != null) {
                request.setStorageClass(storageClass);
            }
            this.ks3Client.copyObject(request);
        });
    }

    public void copyPart(RateLimiter rateLimiter, CopyPartBean bean){
        new RetryHandler(rateLimiter, 3, context).retryProcess(() -> {
            CopyPartRequest copyRequest = new CopyPartRequest(bucket, bean.srcKey,
                    bucket, bean.dstKey,
                    bean.partNum, bean.uploadId);
            copyRequest.setBeginRange(bean.beginRange);
            copyRequest.setEndRange(bean.endRange);

            CopyResult copyResult = ks3Client.copyPart(copyRequest);
            PartETag eTag = new PartETag();
            eTag.setPartNumber(bean.partNum);
            eTag.seteTag(copyResult.getETag());
            bean.eTags.add(eTag);
        });
    }

    public void copyObjects(List<Pair<String, String>> keys) {
        CopyAction copyAction = new CopyAction(this);

        try {
            copyAction.startEngines();

            for (Pair<String, String> key : keys){
                boolean sendResult = copyAction.source().sendData(new HashMap<String, Object>(){
                    {
                        put("srcKey", key.getLeft());
                        put("dstKey", key.getRight());
                    }
                });

                if (!sendResult){
                    break;
                }
            }
        } finally {
            copyAction.shutdown();
        }
    }

    public void createEmptyObject(final String objectKey) {
        createEmptyObject(objectKey, null);
    }

    public void createEmptyObject(final String objectKey, RateLimiter rateLimiter) {
        ObjectMetadata om = new ObjectMetadata();
        om.setContentLength(0L);

        new RetryHandler(rateLimiter, 3, context).retryProcess(() -> {
            PutObjectRequest putObjectRequest = new PutObjectRequest(this.bucket, objectKey, new ByteArrayInputStream(new byte[0]), om);
            this.ks3Client.putObject(putObjectRequest);
        });
    }

    public String initMultipartUpload(String objectKey) {
        return initMultipartUpload(objectKey, null);
    }

    public String initMultipartUpload(String objectKey, RateLimiter rateLimiter) {
        InitiateMultipartUploadResult result = new RetryHandler(rateLimiter, 3, context).retryProcess(() -> {
            InitiateMultipartUploadRequest request =
                    new InitiateMultipartUploadRequest(this.bucket, objectKey);

            if (isSupportKs3Storage) {
                if (this.ks3StorageClass.equals("Standard")) {
                    request.setStorageClass(StorageClass.Standard);
                } else if (this.ks3StorageClass.equals("StandardInfrequentAccess")) {
                    request.setStorageClass(StorageClass.StandardInfrequentAccess);
                }
            }
            return this.ks3Client.initiateMultipartUpload(request);
        });

        return result.getUploadId();
    }

    public PartETag uploadPart(Ks3BlockBuffer ks3BlockBuffer, String uploadId, RateLimiter rateLimiter, int blockLength)
      throws Ks3ClientException, Ks3ServiceException {
        return new RetryHandler(rateLimiter, 3, context).retryProcess(() -> {
            ks3BlockBuffer.refreshInputData(blockLength);
            UploadPartRequest uploadPartRequest = new UploadPartRequest(bucket, ks3BlockBuffer.getKey(), uploadId, ks3BlockBuffer.getBlockId(), ks3BlockBuffer.inBuffer, blockLength);
            return ks3Client.uploadPart(uploadPartRequest);
        });
    }

    public PartETag uploadPart(String objectKey, File file, String uploadId, int partId, long partSize) {
        return new RetryHandler(3, context).retryProcess(() -> {
            try {
                InputStream content = new FileInputStream(file);
                UploadPartRequest uploadPartRequest = new UploadPartRequest(this.bucket, objectKey, uploadId, partId, content, partSize);
                return ks3Client.uploadPart(uploadPartRequest);
            } catch (FileNotFoundException e) {
                LOG.warn(e.getMessage());
                throw new RuntimeException(e.getMessage());
            }
        });
    }


    public void completeMultipartUpload(String objectKey, String uploadId, List<PartETag> eTags) {
        completeMultipartUpload(objectKey, uploadId, eTags, null);
    }

    public void completeMultipartUpload(String objectKey, String uploadId, List<PartETag> eTags, RateLimiter rateLimiter) {
        Collections.sort(eTags, new Comparator<PartETag>() {
            @Override
            public int compare(PartETag arg1, PartETag arg2) {
                PartETag part1 = arg1;
                PartETag part2 = arg2;
                return part1.getPartNumber() - part2.getPartNumber();
            }
        });

        new RetryHandler(rateLimiter, 3, context).retryProcess(() -> {
            CompleteMultipartUploadRequest completeMultipartUploadRequest =
                    new CompleteMultipartUploadRequest(this.bucket, objectKey, uploadId, eTags);
            return this.ks3Client.completeMultipartUpload(completeMultipartUploadRequest);
        });
    }

    private boolean intValToBoolean(int val) {
        if (val == 0) {
            return false;
        } else {
            return true;
        }
    }

    public Configuration getConf() {
        return this.conf;
    }

    // 暂时保留
    public void checkIsRequestCompress(Ks3WebServiceRequest request) {
        boolean isON = conf.getBoolean(Constants.KS3_REQUEST_COMPRESS_ON,
                Constants.DEFAULT_KS3_REQUEST_COMPRESS_ON);
        if (!isON) {
            Utils.noUseGzip(request);
        }
    }

    public String pathToKey(Path path) {
        return Ks3FileSystem.pathToKey(path, workingDir);
    }
}