package com.ksyun.kmr.hadoop.fs.ks3;

import com.google.common.base.Throwables;
import com.ksyun.ks3.AutoAbortInputStream;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.http.ConnectionClosedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.lang.Math.max;
import static java.lang.String.format;

/**
 * Ks3InputStream a wrapper for ks3 file input stream
 */
public class Ks3InputStream extends FSInputStream {
    public static final Logger LOG = LoggerFactory.getLogger(Ks3InputStream.class);
    private static final int MAX_SKIP_SIZE = 1048576;
    private InputStream in;
    private long streamPosition;
    private long nextReadPosition;

    private Ks3FileSystemStore store;
    private String key;
    private long contentLength;
    private boolean closed;
    private long initTime;

    public Ks3InputStream(String key, long contentLength, Ks3FileSystemStore store, FileSystem.Statistics stats) {
        this.key = key;
        this.contentLength = contentLength;
        this.store = store;
        this.in = null;
        this.closed = false;
        this.initTime = System.currentTimeMillis();
    }

    @Override
    public void close() {
        closed = true;
        closeStream();
    }

    @Override
    public void seek(long pos) {
        checkState(!closed, "already closed");
        checkArgument(pos >= 0, "position is negative: %s", pos);

        // this allows a seek beyond the end of the stream but the next read will fail
        nextReadPosition = pos;
    }

    @Override
    public long getPos() {
        return nextReadPosition;
    }

    @Override
    public int read() throws IOException {
        // This stream is wrapped with BufferedInputStream, so this method should never be called
        // throw new UnsupportedOperationException();
        checkState(!closed, "already closed");
        int result = -1;

        try {
            seekStream();

            result = in.read();
            if (result >= 0) {
                streamPosition++;
                nextReadPosition++;
            }
        } catch (Exception e) {
            closeStream();
            Throwables.propagateIfInstanceOf(e, IOException.class);
            throw Throwables.propagate(e);
        }
        return result;
    }

    @Override
    public int read(byte[] buffer, int offset, int length)
            throws IOException {
        checkState(!closed, "already closed");
        int bytesRead;
        try {
            seekStream();
            try {
                bytesRead = in.read(buffer, offset, length);
            } catch (Exception e) {
                //如果连接已关闭，则重新打开流并seek至当前offset继续读取数据
                if (e instanceof ConnectionClosedException){
                    LOG.warn("KS3 file {} inputstream closed, reopen and continue read", key);
                    closeStream();
                    openStream();
                    seekStream();
                    bytesRead = in.read(buffer, offset, length);
                } else {
                    closeStream();
                    throw e;
                }
            }

            if (bytesRead != -1) {
                streamPosition += bytesRead;
                nextReadPosition += bytesRead;
            }
            return bytesRead;
        } catch (Exception e) {
            Throwables.propagateIfInstanceOf(e, IOException.class);
            throw Throwables.propagate(e);
        }

    }

    @Override
    public boolean seekToNewSource(long targetPos) {
        return false;
    }

    private void seekStream()
            throws IOException, UnrecoverableS3OperationException {
        if ((in != null) && (nextReadPosition == streamPosition)) {
            // already at specified position
            return;
        }

        if ((in != null) && (nextReadPosition > streamPosition)) {
            // seeking forwards
            long skip = nextReadPosition - streamPosition;
            if (skip <= max(in.available(), MAX_SKIP_SIZE)) {
                // already buffered or seek is small enough
                try {
                    if (in.skip(skip) == skip) {
                        streamPosition = nextReadPosition;
                        return;
                    }
                } catch (IOException ignored) {
                    // will retry by re-opening the stream
                }
            }
        }

        // close the stream and open at desired position
        streamPosition = nextReadPosition;
        closeStream();
        openStream();
    }

    private void openStream()
            throws IOException, UnrecoverableS3OperationException {
        in = openStream(this.key, nextReadPosition);
        streamPosition = nextReadPosition;
    }

    private InputStream openStream(String key, long start)
            throws IOException, UnrecoverableS3OperationException {
        try {
            InputStream in = store.getObject(key, contentLength, start);
            String keySeed = this.store.getConf().get(Constants.KS3_CLIENT_ENCRYPT_KEY_SEED, null);
            int keySize = this.store.getConf().getInt(Constants.KS3_CLIENT_ENCRYPT_KEY_SIZE,
                    Constants.DEFAULT_KS3_CLIENT_ENCRYPT_KEY_SIZE);
            if (null != keySeed) return Utils.getCipherInputStream(in, keySeed, keySize);
            return in;
        } catch (Exception e) {
            Throwables.propagateIfInstanceOf(e, IOException.class);
            Throwables.propagateIfInstanceOf(e, UnrecoverableS3OperationException.class);
            throw Throwables.propagate(e);
        }
    }



    private void closeStream() {
        if (in != null) {
            try {
                if (in instanceof AutoAbortInputStream) {
                    ((AutoAbortInputStream) in).abort();
                } else {
                    in.close();
                }
            } catch (IOException ignored) {

            }
            in = null;
        }
    }

    static class UnrecoverableS3OperationException
            extends Exception {
        public UnrecoverableS3OperationException(Path path, Throwable cause) {
            super(format("%s (Path: %s)", cause, path), cause);
        }
    }
}
