package com.ksyun.kmr.hadoop.fs.ks3;

import com.google.common.util.concurrent.RateLimiter;
import com.ksyun.kmr.hadoop.fs.ks3.bean.Event;
import com.ksyun.kmr.hadoop.fs.ks3.committer.PendingCommit;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.MultiActionEngine;
import com.ksyun.ks3.dto.PartETag;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.io.DataInputBuffer;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

public class Ks3OutputStream extends OutputStream {
    public static final Log LOG = LogFactory.getLog(Ks3OutputStream.class);
    public final byte[] oneByteAry = new byte[1];
    private Ks3FileSystemStore store;
    private String objectKey;
    private String infoKey;
    private int blockId = 1;
    private int blockAvailable;
    private int blockSize;
    private String uploadId = null;
    private Ks3BlockBuffer blockBuffer;
    private MultiActionEngine engine;
    private volatile boolean closed;
    private List<PartETag> eTags = Collections.synchronizedList(new LinkedList<PartETag>());

    public int getBlockId() {
        return blockId;
    }

    public Ks3OutputStream(Ks3FileSystemStore store, String objectKey) {
        this(store, objectKey, null, null);
    }

    public Ks3OutputStream(Ks3FileSystemStore store, String objectKey, String infoKey) {
        this(store, objectKey, infoKey, null);
    }

    public Ks3OutputStream(Ks3FileSystemStore store, String objectKey, String infoKey, Integer blockSize) {
        this.store = store;
        this.objectKey = objectKey;
        this.infoKey = infoKey;

        if (blockSize == null){
            this.blockSize = store.getBlockSize();
        } else {
            this.blockSize = blockSize;
        }

        this.blockBuffer = new Ks3BlockBuffer(this.objectKey, this.blockId, this.blockSize);
        this.blockAvailable = this.blockSize;

        LOG.info("block_size " + this.blockSize);
    }

    public void startEngine(){
        engine = new MultiActionEngine("parallel upload part", store.parallel_upload_part_pool_size, store.parallel_upload_part_thread_size, store.parallel_upload_part_limit, (Event recvData, MultiActionEngine e) -> {
            RateLimiter rateLimiter = e.getRateLimiter();
            Ks3BlockBuffer value = recvData.getValue("block");
            int blockLength = recvData.getValue("blockSize");

            PartETag eTag = store.uploadPart(value, uploadId, rateLimiter, blockLength);
            eTags.add(eTag);

            value.clear();
        });
    }

    @Override
    public synchronized void write(int i) throws IOException {
        oneByteAry[0] = (byte)i;
        write(oneByteAry, 0, 1);
    }

    @Override
    public synchronized void write(byte[] b, int off, int len) throws IOException {
        if (closed) {
            throw new IOException("Stream closed.");
        }
        int left = len;
        int pos = off;

        while (left > 0){
            refreshCurrentPart(false);
            int batchWriteLen = Math.min(left, blockAvailable);
            blockBuffer.outBuffer.write(b, pos, batchWriteLen);

            blockAvailable -= batchWriteLen;
            left -= batchWriteLen;
            pos += batchWriteLen;
        }
    }

    private synchronized void refreshCurrentPart(boolean force) throws IOException {
        if (blockAvailable == 0 || (force && blockAvailable != blockSize)) {
            if (uploadId == null){
                this.uploadId = store.initMultipartUpload(objectKey);
                startEngine();
            }

            Map<String, Object> data = new HashMap<>();
            data.put("block", blockBuffer);
            data.put("blockSize", blockBuffer.outBuffer.getLength());

            engine.sendData(data);
            this.blockAvailable = blockSize;
            blockId++;
            this.blockBuffer = new Ks3BlockBuffer(this.objectKey, this.blockId, this.blockSize);
        }
    }

    public void shutdownEngine() throws IOException{
        refreshCurrentPart(true);

        if (engine != null){
            engine.shutdown();
        }
    }

    @Override
    public synchronized void close() throws IOException {
        if (closed) {
            return;
        }

        try {
            if (infoKey != null){
                shutdownEngine();

                PendingCommit commit = PendingCommit.build(objectKey, uploadId, eTags);
                store.putObject(infoKey + PendingCommit.EXT, commit.toBytes());
            } else {
                if (blockId == 1) {
                    store.putObject(blockBuffer, blockBuffer.outBuffer.getLength());
                    blockBuffer.clear();
                } else {
                    shutdownEngine();

                    store.completeMultipartUpload(objectKey, uploadId, eTags);
                }
            }
        } catch (Exception e) {
            IOException ioe = new IOException("multipart commit fail");
            ioe.initCause(e);
            throw ioe;
        } finally {
            closed = true;
        }
    }
}
