package com.ksyun.kmr.hadoop.fs.ks3;

import com.google.common.util.concurrent.RateLimiter;
import com.ksyun.kmr.hadoop.FileSystemStoreContext;
import com.ksyun.ks3.dto.Ks3Result;
import com.ksyun.ks3.exception.Ks3ServiceException;
import org.apache.hadoop.fs.FileSystem;

public class RetryHandler {
    public RateLimiter rateLimiter = null;
    public boolean skip404 = false;
    public int kind;

    public static interface Processor<T> {
        public T run();
    }

    private FileSystem.Statistics statistics;
    private int ks3RequestAttemptsMaximum;

    public RetryHandler(int kind, FileSystemStoreContext context){
        this.kind = kind;
        if ((context == null) || (context.statistics == null)){
            throw new RuntimeException("ks3 filesystem not init");
        }

        statistics = context.statistics;
        ks3RequestAttemptsMaximum = context.ks3RequestAttemptsMaximum;
    }

    public RetryHandler(boolean skip404, int kind, FileSystemStoreContext context){
        this(kind, context);
        this.skip404 = skip404;
    }

    public RetryHandler(RateLimiter rateLimiter, int kind, FileSystemStoreContext context){
        this(kind, context);
        this.rateLimiter = rateLimiter;
    }

    public RetryHandler(boolean skip404, RateLimiter rateLimiter, int kind, FileSystemStoreContext context){
        this(kind, context);
        this.skip404 = skip404;
        this.rateLimiter = rateLimiter;
    }

    private void updateStats(){
        switch(kind){
            case 1:
                statistics.incrementReadOps(1);
                break;
            case 2:
                statistics.incrementLargeReadOps(1);
                break;
            case 3:
                statistics.incrementWriteOps(1);
                break;
        }
    }

    public <T> T retryProcess(Processor<T> processor) {
        for (int retry = 1; retry <= ks3RequestAttemptsMaximum; retry++) {
            try {
                if (rateLimiter != null){
                    rateLimiter.acquire();
                }

                updateStats();
                return processor.run();
            } catch (Ks3ServiceException e2) {
                if (ExceptionHandler.handleKs3ServiceException(retry, ks3RequestAttemptsMaximum, e2, skip404)){
                    break;
                }
            } catch (RuntimeException e) {
                ExceptionHandler.handleRuntimeException(retry, ks3RequestAttemptsMaximum, e);
            }
        }

        return null;
    }

    public void retryProcess(Runnable runnable) {
        for (int retry = 1; retry <= ks3RequestAttemptsMaximum; retry++) {
            try {
                if (rateLimiter != null){
                    rateLimiter.acquire();
                }

                updateStats();
                runnable.run();
                break;
            } catch (Ks3ServiceException e2) {
                if (ExceptionHandler.handleKs3ServiceException(retry, ks3RequestAttemptsMaximum, e2, skip404)){
                    break;
                }
            } catch (RuntimeException e) {
                ExceptionHandler.handleRuntimeException(retry, ks3RequestAttemptsMaximum, e);
            }
        }
    }
}
