package com.ksyun.kmr.hadoop.fs.ks3;

import com.ksyun.ks3.http.WebServiceRequestConfig;
import com.ksyun.ks3.service.request.Ks3WebServiceRequest;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.util.StopWatch;
import org.slf4j.Logger;

import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

public class Utils {

    /**
     * encrypt output stream
     *
     * @param out
     * @param keySeed
     * @param keySize
     * @return
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     */
    public static OutputStream getCipherOutputStream(OutputStream out, String keySeed, int keySize)
            throws NoSuchPaddingException, NoSuchAlgorithmException, InvalidKeyException {
        Key keySpec = Utils.getKey(keySeed, keySize);
        Cipher cipher = Cipher.getInstance(Constants.DEFAULT_KS3_CLIENT_ENCRYPT_ALGORITHM);
        cipher.init(Cipher.ENCRYPT_MODE, keySpec);
        return new CipherOutputStream(out, cipher);
    }

    /**
     * decrypt input stream
     *
     * @param in
     * @param keySeed
     * @param keySize
     * @return
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     */
    public static InputStream getCipherInputStream(InputStream in, String keySeed, int keySize)
            throws NoSuchPaddingException, NoSuchAlgorithmException, InvalidKeyException {
        Key keySpec = Utils.getKey(keySeed, keySize);
        Cipher cipher = Cipher.getInstance(Constants.DEFAULT_KS3_CLIENT_ENCRYPT_ALGORITHM);
        cipher.init(Cipher.DECRYPT_MODE, keySpec);
        return new CipherInputStream(in, cipher);
    }

    /**
     * get SecretKey via key seed( password )
     * @param keySeed
     * @param keySize
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     */
    public static Key getKey(String keySeed, int keySize) throws NoSuchAlgorithmException,
            InvalidKeyException {
        if (StringUtils.isBlank(keySeed)) throw new InvalidKeyException("invalid encrypt key seed");
        KeyGenerator keygen =
                KeyGenerator.getInstance(Constants.DEFAULT_KS3_CLIENT_ENCRYPT_ALGORITHM);
        SecureRandom random = SecureRandom.getInstance("SHA1PRNG");
        random.setSeed(keySeed.getBytes());
        keygen.init(keySize, random);
        SecretKey original_key = keygen.generateKey();
        byte[] raw = original_key.getEncoded();
        SecretKey key = new SecretKeySpec(raw, Constants.DEFAULT_KS3_CLIENT_ENCRYPT_ALGORITHM);
        return key;
    }

    public static void noUseGzip(Ks3WebServiceRequest request){
        WebServiceRequestConfig config = request.getRequestConfig();
        if(config == null){
            config = new WebServiceRequestConfig();
        }
        Map<String,String> heads = new HashMap<String, String>();
        heads.put("Accept-Encoding", "none");
        config.setExtendHeaders(heads);
        request.setRequestConfig(config);
    }

    public static RuntimeException rethrowRuntimeEx(Exception e, String message){
        RuntimeException re = new RuntimeException(message);
        re.initCause(e);
        throw re;
    }

    public static String getStackFromException(Exception e){
        StringWriter sw = null;
        PrintWriter pw = null;

        try {
            sw = new StringWriter();
            pw = new PrintWriter(sw);
            e.printStackTrace(pw);
            return sw.toString();
        } finally {
            pw.close();
        }
    }

    public static interface RecordRunnable {
        Object run() throws Exception;
    }

    public static <T> T recordCostTime(Logger logger, String message, RecordRunnable runnable) throws Exception {
        StopWatch sw = new StopWatch().start();
        Object result = runnable.run();
        long costTime = sw.now(TimeUnit.MICROSECONDS);
        logger.info(message + " cost " + costTime);
        return (T)result;
    }

    public static void recordCostTime(Logger logger, String message, Runnable runnable) {
        StopWatch sw = new StopWatch().start();
        runnable.run();
        long costTime = sw.now(TimeUnit.MICROSECONDS);
        logger.info(message + " cost " + costTime);
    }
}
