package com.ksyun.kmr.hadoop.fs.ks3.committer;

import com.ksyun.kmr.hadoop.fs.ks3.Constants;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.mapreduce.JobContext;
import org.apache.hadoop.mapreduce.JobStatus;
import org.apache.hadoop.mapreduce.TaskAttemptContext;
import org.apache.hadoop.mapreduce.lib.output.FileOutputCommitter;

import java.io.IOException;

public class DirectFileOutputCommitter extends FileOutputCommitter {
    private static final Log LOG = LogFactory.getLog(DirectFileOutputCommitter.class);
    private boolean directWrite = false;
    private CommitInfoFileCommitter innerCommitter;


    public DirectFileOutputCommitter(Path outputPath, TaskAttemptContext context) throws IOException {
        super(outputPath, context);
        init(outputPath, context);
    }

    public DirectFileOutputCommitter(Path outputPath, JobContext context) throws IOException {
        super(outputPath, context);
        init(outputPath, context);
    }

    public void init(Path outputPath, JobContext context) throws IOException{
        if (outputPath != null) {
            Configuration conf = context.getConfiguration();
            FileSystem fs = outputPath.getFileSystem(conf);

            if (!fs.getClass().getSimpleName().contains("Ks3FileSystem")) {
                this.directWrite = false;
            } else {
                this.directWrite = conf.getBoolean(Constants.KS3_DIRECT_ENABLE, Constants.DEFAULT_KS3_DIRECT_ENABLE);;
            }

            if (this.directWrite){
                this.innerCommitter = new CommitInfoFileCommitter(outputPath, context);
            }

            LOG.info("FileSystem " + fs.getClass() + ", direct write = " + this.directWrite);
        }
    }

    @Override
    public void setupJob(JobContext context) throws IOException {
        if (this.directWrite) {
            innerCommitter.setupJob(context);
        } else {
            super.setupJob(context);
        }
    }

    @Override
    public void commitJob(JobContext context) throws IOException {
        if (this.directWrite) {
            innerCommitter.commitJob(context);
        } else {
            super.commitJob(context);
        }
    }

    @Override
    public void cleanupJob(JobContext context) throws IOException {
        if (this.directWrite) {
            innerCommitter.cleanupJob(context);
        } else {
            super.cleanupJob(context);
        }
    }

    @Override
    public void abortJob(JobContext context, JobStatus.State state) throws IOException {
        if (this.directWrite) {
            innerCommitter.abortJob(context, state);
        } else {
            super.abortJob(context, state);
        }
    }

    @Override
    public void setupTask(TaskAttemptContext context) throws IOException {
        if (this.directWrite) {
            innerCommitter.setupTask(context);
        } else {
            super.setupTask(context);
        }
    }

    @Override
    public void commitTask(TaskAttemptContext context) throws IOException {
        commitTask(context, null);
    }

    @Override
    public void commitTask(TaskAttemptContext context, Path taskAttemptPath) throws IOException {
        if (this.directWrite) {
            innerCommitter.commitTask(context, taskAttemptPath);
        } else {
            super.commitTask(context, taskAttemptPath);
        }
    }

    @Override
    public void abortTask(TaskAttemptContext context) throws IOException {
        this.abortTask(context, null);
    }

    @Override
    public void abortTask(TaskAttemptContext context, Path taskAttemptPath) throws IOException {
        if (this.directWrite) {
            innerCommitter.abortTask(context, taskAttemptPath);
        } else {
            super.abortTask(context, taskAttemptPath);
        }
    }

    @Override
    public void recoverTask(TaskAttemptContext context) throws IOException {
        if (this.directWrite) {
            innerCommitter.recoverTask(context);
        } else {
            super.recoverTask(context);
        }
    }

    @Override
    public boolean needsTaskCommit(TaskAttemptContext context) throws IOException {
        return needsTaskCommit(context, null);
    }

    @Override
    public boolean needsTaskCommit(TaskAttemptContext context, Path taskAttemptPath) throws IOException {
        if (this.directWrite) {
            return innerCommitter.needsTaskCommit(context, taskAttemptPath);
        } else {
            return super.needsTaskCommit(context, taskAttemptPath);
        }
    }

    @Override
    public Path getWorkPath() throws IOException {
        if (this.directWrite) {
            return innerCommitter.getWorkPath();
        } else {
            return super.getWorkPath();
        }
    }

    @Override
    public boolean isRecoverySupported() {
        if (this.directWrite) {
            return innerCommitter.isRecoverySupported();
        } else {
            return super.isRecoverySupported();
        }
    }

    @Override
    public boolean isCommitJobRepeatable(JobContext context) throws IOException {
        if (this.directWrite) {
            return innerCommitter.isCommitJobRepeatable(context);
        } else {
            return super.isCommitJobRepeatable(context);
        }
    }

}
