package com.ksyun.kmr.hadoop.fs.ks3.parallel;

import com.google.common.util.concurrent.RateLimiter;
import com.ksyun.kmr.hadoop.fs.ks3.bean.Event;
import com.ksyun.kmr.hadoop.fs.ks3.Utils;
import com.lmax.disruptor.*;
import com.lmax.disruptor.dsl.Disruptor;
import com.lmax.disruptor.dsl.ProducerType;
import org.apache.commons.lang3.tuple.Pair;

import java.util.Map;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;


public class MultiActionEngine implements EngineShutter {
    private Disruptor<Event> disruptor;
    private String name;
    private int bufferSize;
    private int parallelNum;
    private EventHandler.Processor processor;
    private EventProducer producer;
    private AtomicInteger receivedNum = new AtomicInteger(0);
    private AtomicInteger handledNum = new AtomicInteger(0);
    private boolean beginShutdown = false;
    private RateLimiter rateLimiter;
    public AtomicReference<Exception> exception;
    private ThreadFactoryBuilder threadFactoryBuilder;
    public static Pair<Integer, TimeUnit> defaultTimeoutConf = Pair.of(10, TimeUnit.MINUTES);
    public static Pair<Integer, TimeUnit> timeoutConf = defaultTimeoutConf;

    public static void resetTimeoutConf(){
        timeoutConf = defaultTimeoutConf;
    }

    public MultiActionEngine(String name, int bufferSize, int parallelNum, RateLimiter rateLimiter, AtomicReference<Exception> exception, EventHandler.Processor processor){
        this.name = name;
        this.bufferSize = bufferSize;
        this.parallelNum = parallelNum;
        this.rateLimiter = rateLimiter;
        this.processor = processor;
        this.exception = exception;

        initDisruptor();
    }

    public MultiActionEngine(String name, int bufferSize, int parallelNum, int rateLimit, EventHandler.Processor processor){
        this(name, bufferSize, parallelNum, RateLimiter.create(rateLimit), new AtomicReference<Exception>(), processor);
    }

    public MultiActionEngine(String name, int bufferSize, int parallelNum, int rateLimit, AtomicReference<Exception> exception, EventHandler.Processor processor){
        this(name, bufferSize, parallelNum, RateLimiter.create(rateLimit), exception, processor);
    }

    public RateLimiter getRateLimiter() {
        return rateLimiter;
    }

    public boolean fail(){
        return exception.get() != null;
    }

    public boolean notFail(){
        return exception.get() == null;
    }

    public void increHandledNum(){
        handledNum.incrementAndGet();
    }

    public boolean getBeginShutdown(){
        return beginShutdown;
    }

    private void initDisruptor(){
        Disruptor<Event> disruptor = null;

        try {
            ThreadFactoryBuilder builder = new ThreadFactoryBuilder(name + "-multi-action-engine-%d");
            ThreadFactory threadFactory = builder.setDaemon(true).build();
            this.threadFactoryBuilder = builder;

            /*
            以ks3的实际性能，用最慢的WaitStrategy足够了，而且其他策略占用cpu较高，会影响application master的资源
             */
            disruptor = new Disruptor<Event>(
                    Event::new,
                    bufferSize,
                    threadFactory,
                    ProducerType.MULTI,
                    new BlockingWaitStrategy()
            );

            EventHandler[] eventHandlers = new EventHandler[parallelNum];
            int i = 0;
            while (i < parallelNum) {
                eventHandlers[i] = new EventHandler(this, processor);
                i++;
            }

            disruptor.handleEventsWithWorkerPool(eventHandlers);
            disruptor.start();

            RingBuffer<Event> ringBuffer = disruptor.getRingBuffer();
            this.producer = new EventProducer(ringBuffer);

            this.disruptor = disruptor;
        } catch (RuntimeException e){
            try {
                if (disruptor != null) {
                    disruptor.shutdown(2, TimeUnit.SECONDS);
                }
            } catch (Exception e2) {
                RuntimeException re = new RuntimeException("shutdown disruptor fail when init disruptor fail");
                e2.initCause(e);
                re.initCause(e2);
                throw re;
            }

            throw e;
        }
    }

    public boolean sendData(Map<String, Object> data) {
        receivedNum.incrementAndGet();

        if (notFail()) {
            producer.onData(data);
            return true;
        }

        return false;
    }

    private void initCause(Exception e){
        Exception ce = exception.get();
        if (ce != null){
            e.initCause(ce);
        }
    }

    private void throwException(String message){
        RuntimeException ex = new RuntimeException(message);
        initCause(ex);
        throw ex;
    }

    /*
    shutdown失败的engine可能会造成资源泄露
    工作线程没有停止，正在处理的任务继续进行，可能会造成奇怪的现象
     */
    public void shutdown(){
        // System.out.println(ManagementFactory.getThreadMXBean().getThreadCount() );
        // System.out.println(Thread.activeCount());
        if (!beginShutdown) {
            beginShutdown = true;
            Exception shutdownException = null;
            try {
                if (disruptor != null) {
                    // disruptor shutdown当前的实现很耗资源 需要使用定制版disruptor，disruptor最新版有重大功能改变，已经不适合继续使用下去
                    disruptor.shutdown(timeoutConf.getLeft(), timeoutConf.getRight());
                }
            } catch (Exception e) {
                initCause(e);
                shutdownException = e;

                try {
                    if (disruptor != null) {
                        disruptor.halt();
                    }

                    if (threadFactoryBuilder != null) {
                        for (Thread t : threadFactoryBuilder.getRecords().values()) {
                            t.interrupt();
                        }
                    }

                    Thread.sleep(1000);
                } catch (Exception e2) {
                    e2.initCause(e);
                    shutdownException = e2;
                }
            }

            int recvNum = receivedNum.get();
            int handleNum = handledNum.get();
            String info = " recvnum " + recvNum + " handlenum " + handleNum;

            if (shutdownException != null) {
                Utils.rethrowRuntimeEx(shutdownException, "multi action engine shutdown fail," + info);
            }

            if (fail() || (recvNum != handleNum)) {
                throwException("multi action engine job fail," + info);
            }
        } else {
            throwException("multi action engine shutdown again");
        }
    }
}
