package com.ksyun.kmr.hadoop.fs.ks3.parallel;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

import java.lang.Thread.UncaughtExceptionHandler;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicLong;

/*
copy from guava 2017，并做以下修改：
thread必须有name
创建的新线程会被记录在builder的records中，方便进行interrupt
 */

public final class ThreadFactoryBuilder {
    private String nameFormat = null;
    private Boolean daemon = null;
    private Integer priority = null;
    private UncaughtExceptionHandler uncaughtExceptionHandler = null;
    private ThreadFactory backingThreadFactory = null;
    private ConcurrentHashMap<String, Thread> records = new ConcurrentHashMap<>();

    public ThreadFactoryBuilder(String nameFormat) {
        setNameFormat(nameFormat);
    }

    public Map<String, Thread> getRecords() {
        return records;
    }

    /**
     * @param nameFormat a {@link String#format(String, Object...)}-compatible format String, to which
     *     a unique integer (0, 1, etc.) will be supplied as the single parameter. This integer will
     *     be unique to the built instance of the ThreadFactory and will be assigned sequentially. For
     *     example, {@code "rpc-pool-%d"} will generate thread names like {@code "rpc-pool-0"}, {@code
     *     "rpc-pool-1"}, {@code "rpc-pool-2"}, etc.
     */
    public ThreadFactoryBuilder setNameFormat(String nameFormat) {
        String unused = format(nameFormat, 0); // fail fast if the format is bad or null
        this.nameFormat = nameFormat;
        return this;
    }

    public ThreadFactoryBuilder setDaemon(boolean daemon) {
        this.daemon = daemon;
        return this;
    }

    public ThreadFactoryBuilder setPriority(int priority) {
        // Thread#setPriority() already checks for validity. These error messages
        // are nicer though and will fail-fast.
        checkArgument(
                priority >= Thread.MIN_PRIORITY,
                "Thread priority (%s) must be >= %s",
                priority,
                Thread.MIN_PRIORITY);
        checkArgument(
                priority <= Thread.MAX_PRIORITY,
                "Thread priority (%s) must be <= %s",
                priority,
                Thread.MAX_PRIORITY);
        this.priority = priority;
        return this;
    }

    public ThreadFactoryBuilder setUncaughtExceptionHandler(
            UncaughtExceptionHandler uncaughtExceptionHandler) {
        this.uncaughtExceptionHandler = checkNotNull(uncaughtExceptionHandler);
        return this;
    }

    public ThreadFactoryBuilder setThreadFactory(ThreadFactory backingThreadFactory) {
        this.backingThreadFactory = checkNotNull(backingThreadFactory);
        return this;
    }

    public ThreadFactory build() {
        final String nameFormat = this.nameFormat;
        final Boolean daemon = this.daemon;
        final Integer priority = this.priority;
        final UncaughtExceptionHandler uncaughtExceptionHandler = this.uncaughtExceptionHandler;
        final ThreadFactory backingThreadFactory =
                (this.backingThreadFactory != null)
                        ? this.backingThreadFactory
                        : Executors.defaultThreadFactory();
        final AtomicLong count = (nameFormat != null) ? new AtomicLong(0) : null;
        return new ThreadFactory() {
            @Override
            public Thread newThread(Runnable runnable) {
                Thread thread = backingThreadFactory.newThread(runnable);
                thread.setName(format(nameFormat, count.getAndIncrement()));

                if (daemon != null) {
                    thread.setDaemon(daemon);
                }
                if (priority != null) {
                    thread.setPriority(priority);
                }
                if (uncaughtExceptionHandler != null) {
                    thread.setUncaughtExceptionHandler(uncaughtExceptionHandler);
                }

                records.put(thread.getName(), thread);
                return thread;
            }
        };
    }

    private static String format(String format, Object... args) {
        return String.format(Locale.ROOT, format, args);
    }
}