package com.ksyun.kmr.hadoop.fs.ks3.parallel.conveyor;

import com.google.common.util.concurrent.RateLimiter;
import com.ksyun.kmr.hadoop.fs.ks3.Ks3FileSystemStore;
import com.ksyun.kmr.hadoop.fs.ks3.Utils;
import com.ksyun.kmr.hadoop.fs.ks3.bean.Event;
import com.ksyun.kmr.hadoop.fs.ks3.committer.PendingCommit;
import com.ksyun.kmr.hadoop.fs.ks3.committer.PendingCommitList;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.ActionSource;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.Conveyor;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.EngineShutter;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.MultiActionEngine;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.concurrent.atomic.AtomicReference;

public class CommitAction extends Conveyor implements ActionSource {
    private static final Logger LOG = LoggerFactory.getLogger(CommitAction.class);

    Ks3FileSystemStore store;
    MultiActionEngine engineForCommit;
    MultiActionEngine engine;
    public Sink sink;

    public CommitAction(Ks3FileSystemStore store){
        this(store, new AtomicReference<>());
    }

    public CommitAction(Ks3FileSystemStore store, AtomicReference<Exception> exceptionAtomicReference){
        super(exceptionAtomicReference);
        this.store = store;
    }

    public static interface Sink {
        void callback(String key);
    }

    @Override
    public void startEnginesWithoutCheckStarted() {
        MultiActionEngine engineForCommit = new MultiActionEngine("parallel commit", store.parallel_commit_pool_size, store.parallel_commit_thread_size, store.parallel_commit_speed_limit, exceptionAtomicReference, (Event recvData, MultiActionEngine e) -> {
            RateLimiter rateLimiter = e.getRateLimiter();
            PendingCommit commit = recvData.getValue("data");

            if (commit.getKs3PartETags().size() == 0){
                Utils.recordCostTime(LOG, "createEmpty", () -> {
                    store.createEmptyObject(commit.key, rateLimiter);
                });
            } else {
                Utils.recordCostTime(LOG, "multicommit", () -> {
                    store.completeMultipartUpload(commit.key, commit.uploadId, commit.getKs3PartETags(), rateLimiter);
                });
            }
        });

        MultiActionEngine engine = new MultiActionEngine("parallel unserialize", store.parallel_unser_pool_size, store.parallel_unser_thread_size, store.parallel_unser_speed_limit, exceptionAtomicReference, (Event recvData, MultiActionEngine e) -> {
            RateLimiter rateLimiter = e.getRateLimiter();
            Path commitPath = recvData.getValue("data");
            String key = store.pathToKey(commitPath);
            PendingCommitList commitList = PendingCommitList.load(store, key, rateLimiter);

            for (PendingCommit commit : commitList.data){
                if (!engineForCommit.sendData(Collections.singletonMap("data", commit))){
                    break;
                }
            }

            if (sink != null){
                sink.callback(key);
            }
        });

        this.engineForCommit = engineForCommit;
        this.engine = engine;
    }

    @Override
    public void shutdown() {
        EngineShutter.shutdownAll(engine, engineForCommit);
    }

    public MultiActionEngine source(){
        return engine;
    }
}
