package com.ksyun.kmr.hadoop.fs.ks3.parallel.conveyor;

import com.google.common.util.concurrent.RateLimiter;
import com.ksyun.kmr.hadoop.fs.ks3.Ks3FileSystemStore;
import com.ksyun.kmr.hadoop.fs.ks3.bean.CopyPartBean;
import com.ksyun.kmr.hadoop.fs.ks3.bean.Event;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.ActionSource;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.Conveyor;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.EngineShutter;
import com.ksyun.kmr.hadoop.fs.ks3.parallel.MultiActionEngine;
import com.ksyun.ks3.dto.ObjectMetadata;
import com.ksyun.ks3.dto.PartETag;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

public class CopyAction extends Conveyor implements ActionSource {
    MultiActionEngine mainCopyEngine;
    MultiActionEngine copyPartEngine;
    Ks3FileSystemStore store;
    public Sink sink;

    public CopyAction(Ks3FileSystemStore store){
        this(store, new AtomicReference<>());
    }

    public CopyAction(Ks3FileSystemStore store, AtomicReference<Exception> exceptionAtomicReference){
        super(exceptionAtomicReference);
        this.store = store;
    }

    @Override
    public void startEnginesWithoutCheckStarted() {
        initCopyEngine();
    }

    @Override
    public void shutdown() {
        EngineShutter.shutdownAll(mainCopyEngine, copyPartEngine);
    }

    public MultiActionEngine source(){
        return mainCopyEngine;
    }

    public static interface Sink {
        void callback(Pair<String, String> data);
    }

    public void initCopyEngine(){
        MultiActionEngine engineForPart = new MultiActionEngine("parallel copy part", store.parallel_copy_part_pool_size, store.parallel_copy_part_thread_size, store.parallel_copy_part_speed_limit, exceptionAtomicReference, (Event recvData, MultiActionEngine e) -> {
            RateLimiter rateLimiter = e.getRateLimiter();
            CopyPartBean bean = recvData.getValue("data");

            store.copyPart(rateLimiter, bean);

            if (bean.handledNum.incrementAndGet() == bean.totalNum){
                store.completeMultipartUpload(bean.dstKey, bean.uploadId, bean.eTags, rateLimiter);
                if (sink != null){
                    sink.callback(Pair.of(bean.srcKey, bean.dstKey));
                }
            }
        });


        MultiActionEngine engine = new MultiActionEngine("parallel copy", store.parallel_copy_pool_size, store.parallel_copy_thread_size, store.parallel_copy_speed_limit, exceptionAtomicReference, (Event recvData, MultiActionEngine e) -> {
            RateLimiter rateLimiter = e.getRateLimiter();
            String srcKey = recvData.getValue("srcKey");
            String dstKey = recvData.getValue("dstKey");

            ObjectMetadata metaData = store.getMetadata(srcKey, rateLimiter);
            long contentLength = metaData.getContentLength();

            if (contentLength > store.getOverallCopyMaxLen()) {
                String uploadId = store.initMultipartUpload(dstKey, rateLimiter);
                sendCopyParts(engineForPart, store.getCopyBlockSize(), contentLength, srcKey, dstKey, uploadId);
            } else {
                store.copyObject(metaData, srcKey, dstKey, rateLimiter);
                if (sink != null){
                    sink.callback(Pair.of(srcKey, dstKey));
                }
            }
        });

        copyPartEngine = engineForPart;
        mainCopyEngine = engine;
    }

    public static void sendCopyParts(MultiActionEngine engine, int blockSize, long contentLength, String srcKey, String dstKey, String uploadId){
        List<PartETag> eTags = Collections.synchronizedList(new LinkedList<>());
        List<Triple<Integer, Long, Long>> params = new LinkedList<>();

        long beginRange = 0;
        int partNumber = 1;

        AtomicInteger handledNum = new AtomicInteger(0);

        while (beginRange < contentLength) {
            long endRange = Math.min(beginRange + blockSize, contentLength) - 1;
            params.add(Triple.of(partNumber, beginRange, endRange));
            beginRange = endRange + 1;
            partNumber += 1;
        }

        int totalNum = partNumber - 1;

        for(Triple<Integer, Long, Long> item : params){
            CopyPartBean bean = new CopyPartBean(srcKey, dstKey, uploadId, item.getLeft(), item.getMiddle(), item.getRight(), eTags, handledNum, totalNum);

            if (!engine.sendData(Collections.singletonMap("data", bean))){
                break;
            }
        }
    }
}
