package com.ksyun.kmr.hadoop.fs.ks3.requestbuilder;

import com.ksyun.kmr.hadoop.fs.ks3.Ks3FileSystemStore;
import com.ksyun.kmr.hadoop.fs.ks3.ListObjectsResult;
import com.ksyun.kmr.hadoop.fs.ks3.RetryHandler;
import com.ksyun.ks3.dto.ObjectListing;
import com.ksyun.ks3.service.request.ListObjectsRequest;
import org.apache.hadoop.util.StopWatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Iterator;
import java.util.Spliterator;
import java.util.Spliterators;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/*
ks3会在负载较大的时候，list请求返回的结果条数变少，这会导致list一个目录的请求次数变多，总时间变长
 */

public class ListDir {
    public Ks3FileSystemStore store;
    public int pageNum;
    private boolean isFlat;
    public int limit;
    public String objectKey;
    public int leftNum;
    public ListObjectsRequest request;

    private static final Logger LOG = LoggerFactory.getLogger(ListDir.class);

    public ListDir(Ks3FileSystemStore store, String objectKey, boolean isFlat){
        this(store, objectKey, isFlat, 0, 1000);
    }

    public ListDir(Ks3FileSystemStore store, String objectKey, boolean isFlat, int limit, int pageNum){
        this.store = store;
        this.objectKey = objectKey;
        this.isFlat = isFlat;
        this.limit = limit;
        this.pageNum = pageNum;

        this.leftNum = (limit == 0 ? Integer.MAX_VALUE : limit);
        this.request = genRequest();
    }

    public ListObjectsRequest genRequest(){
        ListObjectsRequest request = new ListObjectsRequest(store.bucket);
        request.setPrefix(objectKey);
        if (!isFlat) {
            request.setDelimiter("/");
        }

        return request;
    }

    public ObjectListing listNextBatchOfResults(ListObjectsRequest request) {
        return new RetryHandler(2, store.context).retryProcess(() -> {
            StopWatch sw = new StopWatch().start();
            ObjectListing result = store.ks3Client.listObjects(request);
            long costTime = sw.now(TimeUnit.MICROSECONDS);
            LOG.info("listNextBatch" + " cost " + costTime);

            return result;
        });
    }

    public ListObjectsResult listAll() {
        ListObjectsResult result = new ListObjectsResult();

        genStream().forEach(item -> {
            result.objectSummaries.addAll(item.objectSummaries);
            result.commonPrefixes.addAll(item.commonPrefixes);
        });

        return result;
    }

    public Stream<ListObjectsResult> genStream(){
        return genStream(new AtomicReference<>());
    }
    
    public Stream<ListObjectsResult> genStream(AtomicReference<Exception> exceptionAtomicReference){
        return StreamSupport.stream(Spliterators.spliteratorUnknownSize(genIterator(exceptionAtomicReference), Spliterator.IMMUTABLE), false);
    }

    protected Iterator<ListObjectsResult> genIterator(AtomicReference<Exception> exceptionAtomicReference){
        return new Iterator<ListObjectsResult>() {
            StopWatch sw = new StopWatch();

            @Override
            public boolean hasNext() {
                boolean result = (exceptionAtomicReference.get() == null) && leftNum > 0;
                if (!result && sw.isRunning()){
                    long costTime = sw.now(TimeUnit.MICROSECONDS);
                    LOG.info("listDirTotal " + objectKey + " cost " + costTime);
                }

                return result;
            }

            @Override
            public ListObjectsResult next() {
                if (!sw.isRunning()){
                    sw.start();
                }

                ListObjectsResult iterResult = new ListObjectsResult();

                int currentPageNum = Math.min(pageNum, leftNum);
                request.setMaxKeys(currentPageNum);
                ObjectListing objects = listNextBatchOfResults(request);
                leftNum -= (objects.getObjectSummaries().size() + objects.getCommonPrefixes().size());

                if (objects.isTruncated()) {
                    if (isFlat) {
                        request.setMarker(objects.getObjectSummaries().get(objects.getObjectSummaries().size() - 1).getKey());
                    } else {
                        request.setMarker(objects.getNextMarker());
                    }
                } else {
                    leftNum = -1;
                }

                iterResult.objectSummaries.addAll(objects.getObjectSummaries());
                iterResult.commonPrefixes.addAll(objects.getCommonPrefixes());

                LOG.info("genStream batch finished");
                return iterResult;
            }
        };
    }
}
