package com.ksyun.kmr.hadoop.fs.ks3.requestbuilder;

import com.ksyun.kmr.hadoop.fs.ks3.Ks3FileStatus;
import com.ksyun.kmr.hadoop.fs.ks3.Ks3FileSystem;
import com.ksyun.kmr.hadoop.fs.ks3.Ks3FileSystemStore;
import com.ksyun.kmr.hadoop.fs.ks3.ListObjectsResult;
import com.ksyun.ks3.dto.Ks3ObjectSummary;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;

import java.io.IOException;
import java.net.URI;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Stream;

public class ListFileStatus {
    public Ks3FileSystem ks3FS;
    public Path f;
    public boolean isFlat;
    public Path qualifiedPath;
    public Ks3FileStatus fileStatus;
    public URI pathUri;
    protected ListDir listDir;
    public Ks3FileSystemStore store;
    public boolean relativeSubPathName;
    protected boolean onlyFiles = false;

    public ListFileStatus(Ks3FileSystem ks3FS, Path f, boolean isFlat, Ks3FileStatus fileStatus, boolean relativeSubPathName){
        this.ks3FS = ks3FS;
        this.store = ks3FS.getStore();
        this.f = f;
        this.isFlat = isFlat;
        this.qualifiedPath = f.makeQualified(ks3FS.uri, ks3FS.workingDir);
        this.fileStatus = fileStatus;
        this.pathUri = this.qualifiedPath.toUri();
        this.relativeSubPathName = relativeSubPathName;

        genListDir();
    }

    public ListFileStatus(Ks3FileSystem ks3FS, Path f, boolean isFlat, Ks3FileStatus fileStatus){
        this(ks3FS, f, isFlat, fileStatus, false);
    }

    public ListFileStatus(Ks3FileSystem ks3FS, Path f, boolean isFlat) throws IOException{
        this(ks3FS, f, isFlat, ks3FS.getFileStatus(f), false);
    }

    public ListFileStatus(Ks3FileSystem ks3FS, Path f, boolean isFlat, boolean relativeSubPathName) throws IOException{
        this(ks3FS, f, isFlat, ks3FS.getFileStatus(f), relativeSubPathName);
    }

    public ListDir getListDir() {
        return listDir;
    }

    public ListFileStatus withOnlyFiles(){
        onlyFiles = true;
        return this;
    }

    public Stream<ListObjectsResult> genStream(AtomicReference<Exception> exceptionAtomicReference){
        return listDir.genStream(exceptionAtomicReference);
    }

    public static String getDirKey(Ks3FileSystem fs, Path f){
        String key = fs.pathToKey(f);

        if (!key.isEmpty()) {
            key = key + "/";
        }

        return key;
    }

    public void genListDir(){
        String key = getDirKey(ks3FS, f);
        listDir = new ListDir(store, key, isFlat);
    }

    /*
    注意： 在这里key和key/无法区分，用户的不当使用可能会引起异常结果
    */
    public Ks3FileStatus[] listStatus(PathFilter... filters) {
        List<Ks3FileStatus> statuses = new LinkedList<>();

        if (fileStatus.isDirectory()) {
            ListObjectsResult listResult = listDir.listAll();
            statuses.addAll(wrapToFileStatus(listResult, filters));
        } else {
            statuses.addAll(filterResult(Collections.singletonList(fileStatus), filters));
        }

        return statuses.toArray(new Ks3FileStatus[0]);
    }

    public List<Ks3FileStatus> wrapToFileStatus(ListObjectsResult listResult){
        List<Ks3FileStatus> statuses = new LinkedList<>();

        for (Ks3ObjectSummary summary : listResult.objectSummaries) {
            String itemKey = summary.getKey();
            Path keyPath = ks3FS.toQ(ks3FS.keyToPath(itemKey));
            String relativePath = pathUri.relativize(keyPath.toUri()).getPath();
            if (keyPath.equals(qualifiedPath)) {
                // placeholder for directory itself
                continue;
            }

            if (relativeSubPathName){
                keyPath = new Path(relativePath);
            }

            if (ks3FS.objectRepresentsDirectory(itemKey, summary.getSize())) {
                if (!onlyFiles){
                    statuses.add(new Ks3FileStatus(keyPath, itemKey));
                }
            } else {
                statuses.add(new Ks3FileStatus(summary.getSize(),
                        ks3FS.dateToLong(summary.getLastModified()), keyPath,
                        ks3FS.ks3BlockSize, summary.getETag()));
            }
        }

        if (!onlyFiles) {
            for (String prefix : listResult.commonPrefixes) {
                Path keyPath = ks3FS.keyToPath(prefix).makeQualified(ks3FS.uri, ks3FS.workingDir);
                String relativePath = pathUri.relativize(keyPath.toUri()).getPath();
                if (keyPath.equals(qualifiedPath)) {
                    continue;
                }

                if (relativeSubPathName) {
                    keyPath = new Path(relativePath);
                }
                statuses.add(new Ks3FileStatus(keyPath, prefix));
            }
        }

        return statuses;
    }

    public List<Ks3FileStatus> wrapToFileStatus(ListObjectsResult listResult, PathFilter... filters){
        return filterResult(wrapToFileStatus(listResult), filters);
    }

    public static List<Ks3FileStatus> filterResult(List<Ks3FileStatus> statuses, PathFilter... filters){
        if (filters.length > 0) {
            LinkedList<Ks3FileStatus> results = new LinkedList<>();

            for (int i = 0; i < statuses.size(); i++) {
                Ks3FileStatus item = statuses.get(i);
                Path itemPath = item.getPath();
                boolean accept = true;

                for (PathFilter filter : filters) {
                    if (!filter.accept(itemPath)) {
                        accept = false;
                        break;
                    }
                }

                if (accept) {
                    results.add(item);
                }
            }

            return results;
        } else {
            return statuses;
        }
    }
}
