package com.ksyun.ks3.dto;

import java.io.IOException;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.ksyun.ks3.exception.Ks3ClientException;

/**
 * <p>
 * 桶镜像配置实体类.
 * </p>
 * 
 * @author huanghaixin
 * @version 1.0 2/23/2021
 */
public class BucketMirror {
    private static final ObjectMapper OBJ_MAPPER = new ObjectMapper();
    static {
        OBJ_MAPPER.setSerializationInclusion(Include.NON_NULL);
        OBJ_MAPPER.setPropertyNamingStrategy(PropertyNamingStrategy.CAMEL_CASE_TO_LOWER_CASE_WITH_UNDERSCORES);
        OBJ_MAPPER.configure(SerializationFeature.WRITE_ENUMS_USING_TO_STRING, true);
        OBJ_MAPPER.configure(DeserializationFeature.READ_ENUMS_USING_TO_STRING, true);
    };

    private Version version;
    private boolean useDefaultRobots;
    private AsyncMirrorRule asyncMirrorRule;
    private List<SyncMirrorRule> syncMirrorRules;
    private String createdTime;
    private String lastModifiedTime;

    public BucketMirror() {

    }

    public BucketMirror(Version version, boolean useDefaultRobots, AsyncMirrorRule asyncMirrorRule,
            List<SyncMirrorRule> syncMirrorRules) {
        this.version = version;
        this.useDefaultRobots = useDefaultRobots;
        this.asyncMirrorRule = asyncMirrorRule;
        this.syncMirrorRules = syncMirrorRules;
    }

    public String toJson() {
        try {
            return OBJ_MAPPER.writeValueAsString(this);
        } catch (JsonProcessingException e) {
            throw new Ks3ClientException("failed to serialize bucketMirror to json text", e);
        }
    }

    public static BucketMirror fromJson(String jsonText) {
        try {
            return OBJ_MAPPER.readValue(jsonText, BucketMirror.class);
        } catch (IOException e) {
            throw new Ks3ClientException("failed to deserialize bucketMirror from json text", e);
        }
    }

    /**
     * @return the version
     */
    public Version getVersion() {
        return version;
    }

    /**
     * @param version the version to set
     */
    public void setVersion(Version version) {
        this.version = version;
    }

    /**
     * @return the useDefaultRobots
     */
    public boolean isUseDefaultRobots() {
        return useDefaultRobots;
    }

    /**
     * @param useDefaultRobots the useDefaultRobots to set
     */
    public void setUseDefaultRobots(boolean useDefaultRobots) {
        this.useDefaultRobots = useDefaultRobots;
    }

    /**
     * @return the asyncMirrorRule
     */
    public AsyncMirrorRule getAsyncMirrorRule() {
        return asyncMirrorRule;
    }

    /**
     * @param asyncMirrorRule the asyncMirrorRule to set
     */
    public void setAsyncMirrorRule(AsyncMirrorRule asyncMirrorRule) {
        this.asyncMirrorRule = asyncMirrorRule;
    }

    /**
     * @return the syncMirrorRules
     */
    public List<SyncMirrorRule> getSyncMirrorRules() {
        return syncMirrorRules;
    }

    /**
     * @param syncMirrorRules the syncMirrorRules to set
     */
    public void setSyncMirrorRules(List<SyncMirrorRule> syncMirrorRules) {
        this.syncMirrorRules = syncMirrorRules;
    }

    /**
     * @return the createdTime
     */
    public String getCreatedTime() {
        return createdTime;
    }

    /**
     * @param createdTime the createdTime to set
     */
    public void setCreatedTime(String createdTime) {
        this.createdTime = createdTime;
    }

    /**
     * @return the lastModifiedTime
     */
    public String getLastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * @param lastModifiedTime the lastModifiedTime to set
     */
    public void setLastModifiedTime(String lastModifiedTime) {
        this.lastModifiedTime = lastModifiedTime;
    }

    public static enum Version {
        V3
    }

    /**
     * <p>
     * 异步回源规则实体类.
     * </p>
     * 
     * @author huanghaixin
     * @version 1.0 2021年3月8日
     */
    public static class AsyncMirrorRule {
        List<String> mirrorUrls;
        SavingSetting savingSetting;

        public AsyncMirrorRule() {
        }

        public AsyncMirrorRule(List<String> mirrorUrls, SavingSetting savingSetting) {
            this.mirrorUrls = mirrorUrls;
            this.savingSetting = savingSetting;
        }

        /**
         * @return the mirrorUrls
         */
        public List<String> getMirrorUrls() {
            return mirrorUrls;
        }

        /**
         * @param mirrorUrls the mirrorUrls to set
         */
        public void setMirrorUrls(List<String> mirrorUrls) {
            this.mirrorUrls = mirrorUrls;
        }

        /**
         * @return the savingSetting
         */
        public SavingSetting getSavingSetting() {
            return savingSetting;
        }

        /**
         * @param savingSetting the savingSetting to set
         */
        public void setSavingSetting(SavingSetting savingSetting) {
            this.savingSetting = savingSetting;
        }

        @Override
        public String toString() {
            return "AsyncMirrorRule [mirrorUrls=" + mirrorUrls + ", savingSetting=" + savingSetting + "]";
        }

    }

    /**
     * <p>
     * 同步回源规则实体类.
     * </p>
     * 
     * @author huanghaixin
     * @version 1.0 2021年3月8日
     */
    public static class SyncMirrorRule {
        MatchCondition matchCondition;
        String mirrorUrl;
        MirrorRequestSetting mirrorRequestSetting;
        SavingSetting savingSetting;

        public SyncMirrorRule() {

        }

        public SyncMirrorRule(MatchCondition matchCondition, String mirrorUrl,
                MirrorRequestSetting mirrorRequestSetting, SavingSetting savingSetting) {
            this.matchCondition = matchCondition;
            this.mirrorUrl = mirrorUrl;
            this.mirrorRequestSetting = mirrorRequestSetting;
            this.savingSetting = savingSetting;
        }

        /**
         * @return the matchCondition
         */
        public MatchCondition getMatchCondition() {
            return matchCondition;
        }

        /**
         * @param matchCondition the matchCondition to set
         */
        public void setMatchCondition(MatchCondition matchCondition) {
            this.matchCondition = matchCondition;
        }

        /**
         * @return the mirrorUrl
         */
        public String getMirrorUrl() {
            return mirrorUrl;
        }

        /**
         * @param mirrorUrl the mirrorUrl to set
         */
        public void setMirrorUrl(String mirrorUrl) {
            this.mirrorUrl = mirrorUrl;
        }

        /**
         * @return the mirrorRequestSetting
         */
        public MirrorRequestSetting getMirrorRequestSetting() {
            return mirrorRequestSetting;
        }

        /**
         * @param mirrorRequestSetting the mirrorRequestSetting to set
         */
        public void setMirrorRequestSetting(MirrorRequestSetting mirrorRequestSetting) {
            this.mirrorRequestSetting = mirrorRequestSetting;
        }

        /**
         * @return the savingSetting
         */
        public SavingSetting getSavingSetting() {
            return savingSetting;
        }

        /**
         * @param savingSetting the savingSetting to set
         */
        public void setSavingSetting(SavingSetting savingSetting) {
            this.savingSetting = savingSetting;
        }

        @Override
        public String toString() {
            return "SyncMirrorRule [matchCondition=" + matchCondition + ", mirrorRequestSetting=" + mirrorRequestSetting
                    + ", mirrorUrl=" + mirrorUrl + ", savingSetting=" + savingSetting + "]";
        }

    }

    /**
     * <p>
     * 进行回源请求时的匹配条件。
     * </p>
     * 
     * @author huanghaixin
     * @version 1.0 2021年3月8日
     */
    public static class MatchCondition {
        List<String> httpCodes;
        List<String> keyPrefixes;

        public MatchCondition() {
        }

        public MatchCondition(List<String> httpCodes, List<String> keyPrefixes) {
            this.httpCodes = httpCodes;
            this.keyPrefixes = keyPrefixes;
        }

        /**
         * @return the httpCodes
         */
        public List<String> getHttpCodes() {
            return httpCodes;
        }

        /**
         * @param httpCodes the httpCodes to set
         */
        public void setHttpCodes(List<String> httpCodes) {
            this.httpCodes = httpCodes;
        }

        /**
         * @return the keyPrefixes
         */
        public List<String> getKeyPrefixes() {
            return keyPrefixes;
        }

        /**
         * @param keyPrefixes the keyPrefixes to set
         */
        public void setKeyPrefixes(List<String> keyPrefixes) {
            this.keyPrefixes = keyPrefixes;
        }

        @Override
        public String toString() {
            return "MatchCondition [httpCodes=" + httpCodes + ", keyPrefixes=" + keyPrefixes + "]";
        }

    }

    /**
     * <p>
     * 执行回源请求时的配置。
     * </p>
     * 
     * @author huanghaixin
     * @version 1.0 2021年3月8日
     */
    public static class MirrorRequestSetting {
        boolean passQueryString;
        boolean follow3xx;
        HeaderSetting headerSetting;

        public MirrorRequestSetting() {
        }

        public MirrorRequestSetting(boolean passQueryString, boolean follow3xx, HeaderSetting headerSetting) {
            this.passQueryString = passQueryString;
            this.follow3xx = follow3xx;
            this.headerSetting = headerSetting;
        }

        /**
         * @return the passQueryString
         */
        public boolean isPassQueryString() {
            return passQueryString;
        }

        /**
         * @param passQueryString the passQueryString to set
         */
        public void setPassQueryString(boolean passQueryString) {
            this.passQueryString = passQueryString;
        }

        /**
         * @return the follow3xx
         */
        public boolean isFollow3xx() {
            return follow3xx;
        }

        /**
         * @param follow3xx the follow3xx to set
         */
        public void setFollow3xx(boolean follow3xx) {
            this.follow3xx = follow3xx;
        }

        /**
         * @return the headerSetting
         */
        public HeaderSetting getHeaderSetting() {
            return headerSetting;
        }

        /**
         * @param headerSetting the headerSetting to set
         */
        public void setHeaderSetting(HeaderSetting headerSetting) {
            this.headerSetting = headerSetting;
        }

        @Override
        public String toString() {
            return "MirrorRequestSetting [follow3xx=" + follow3xx + ", headerSetting=" + headerSetting
                    + ", passQueryString=" + passQueryString + "]";
        }

    }

    /**
     * <p>
     * 执行回源请求时的header配置。
     * </p>
     * 
     * @author huanghaixin
     * @version 1.0 2021年3月8日
     */
    public static class HeaderSetting {
        List<Header> setHeaders;
        List<Header> removeHeaders;
        boolean passAll;
        List<Header> passHeaders;

        public HeaderSetting() {
        }

        public HeaderSetting(List<Header> setHeaders, List<Header> removeHeaders, boolean passAll,
                List<Header> passHeaders) {
            this.setHeaders = setHeaders;
            this.removeHeaders = removeHeaders;
            this.passAll = passAll;
            this.passHeaders = passHeaders;
        }

        /**
         * @return the setHeaders
         */
        public List<Header> getSetHeaders() {
            return setHeaders;
        }

        /**
         * @param setHeaders the setHeaders to set
         */
        public void setSetHeaders(List<Header> setHeaders) {
            this.setHeaders = setHeaders;
        }

        /**
         * @return the removeHeaders
         */
        public List<Header> getRemoveHeaders() {
            return removeHeaders;
        }

        /**
         * @param removeHeaders the removeHeaders to set
         */
        public void setRemoveHeaders(List<Header> removeHeaders) {
            this.removeHeaders = removeHeaders;
        }

        /**
         * @return the passAll
         */
        public boolean isPassAll() {
            return passAll;
        }

        /**
         * @param passAll the passAll to set
         */
        public void setPassAll(boolean passAll) {
            this.passAll = passAll;
        }

        /**
         * @return the passHeaders
         */
        public List<Header> getPassHeaders() {
            return passHeaders;
        }

        /**
         * @param passHeaders the passHeaders to set
         */
        public void setPassHeaders(List<Header> passHeaders) {
            this.passHeaders = passHeaders;
        }

        @Override
        public String toString() {
            return "HeaderSetting [passAll=" + passAll + ", passHeaders=" + passHeaders + ", removeHeaders="
                    + removeHeaders + ", setHeaders=" + setHeaders + "]";
        }

    }

    /**
     * <p>
     * 回源成功后向ks3保存对象时的配置。
     * </p>
     * 
     * @author huanghaixin
     * @version 1.0 2021年3月8日
     */
    public static class SavingSetting {
        CannedAccessControlList acl;

        public SavingSetting() {
        }

        public SavingSetting(CannedAccessControlList acl) {
            this.acl = acl;
        }

        /**
         * @return the acl
         */
        public CannedAccessControlList getAcl() {
            return acl;
        }

        /**
         * @param acl the acl to set
         */
        public void setAcl(CannedAccessControlList acl) {
            this.acl = acl;
        }

        @Override
        public String toString() {
            return "SavingSetting [acl=" + acl + "]";
        }

    }

    /**
     * <p>
     * 回源规则中的Header.
     * </p>
     * 
     * @author huanghaixin
     * @version 1.0 2021年3月8日
     */
    public static class Header {
        String key;
        String value;

        public Header() {

        }

        public Header(String key) {
            this.key = key;
        }

        public Header(String key, String value) {
            this.key = key;
            this.value = value;
        }

        /**
         * @return the key
         */
        public String getKey() {
            return key;
        }

        /**
         * @param key the key to set
         */
        public void setKey(String key) {
            this.key = key;
        }

        /**
         * @return the value
         */
        public String getValue() {
            return value;
        }

        /**
         * @param value the value to set
         */
        public void setValue(String value) {
            this.value = value;
        }

        @Override
        public String toString() {
            return "Header [key=" + key + ", value=" + value + "]";
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */

    @Override
    public String toString() {
        return "BucketMirror [asyncMirrorRule=" + asyncMirrorRule + ", createdTime=" + createdTime
                + ", lastModifiedTime=" + lastModifiedTime + ", syncMirrorRules=" + syncMirrorRules
                + ", useDefaultRobots=" + useDefaultRobots + ", version=" + version + "]";
    }

}
