package com.ksyun.ks3.service.request;

import com.ksyun.ks3.dto.ResponseHeaderOverrides;
import com.ksyun.ks3.dto.SSECustomerKey;
import com.ksyun.ks3.http.HttpHeaders;
import com.ksyun.ks3.http.HttpMethod;
import com.ksyun.ks3.http.Request;
import com.ksyun.ks3.utils.DateUtils;
import com.ksyun.ks3.utils.DateUtils.DATETIME_PROTOCOL;
import com.ksyun.ks3.utils.HttpUtils;
import com.ksyun.ks3.utils.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import static com.ksyun.ks3.exception.client.ClientIllegalArgumentExceptionGenerator.notNull;

/**
 * @author lijunwei[lijunwei@kingsoft.com]  
 * 
 * @date 2014年10月20日 下午7:55:25
 * 
 * @description 获取object
 * <p>支持分块下载，可通过setRange(long,long)实现</p>
 * <p>支持缓存控制，通过matchingETagConstraints、nonmatchingEtagConstraints、unmodifiedSinceConstraint、modifiedSinceConstraint控制</p>
 * <p>支持重写返回的http headers,通过修改overrides实现</p>
 **/
public class GetObjectRequest extends Ks3WebServiceRequest {
	private static final Log log = LogFactory.getLog(GetObjectRequest.class);

	private String bucket;
	private String key;

	/**
	 * 分块下载的范围
	 * 1. 如果range为空、length不是2、起始结束都是负数，则视为未设置range，将请求整个文件
	 * 2. 如果起始结束值均为非负数，则表示请求从起始位置至结束位置的文件内容
	 * 3. 如果起始值为负数，则视为未填写起始位置，文件范围仅由结束值决定。例range=[-1,100]，则等同于range:bytes=-100，表示请求文件的最后100字节
	 * 4. 如果结束值为负数，则视为未填写结束位置，文件范围仅由开始值决定。例range=[100,-1]，则等同于range:bytes=100-，表示请求文件的第101字节至文件末尾
	 */
	private long [] range = null;
	/**
	 * object的etag能匹配到则返回，否则返回结果的ifPreconditionSuccess为false，object为空
	 */
	private List<String> matchingETagConstraints = new ArrayList<String>();
	/**
	 * object的etag不同于其中的任何一个，否则返回结果的ifModified为false,object为空
	 */
	private List<String> nonmatchingEtagConstraints = new ArrayList<String>();
	/**
	 * 在此时间之后没有被修改过，否则返回结果的ifPreconditionSuccess为false，object为空
	 */
	private Date unmodifiedSinceConstraint;
	/**
	 * 在此时间之后被修改过，否则返回结果的ifModified为false,object为空
	 */
	private Date modifiedSinceConstraint;
	/**
	 * 修改返回的response的headers
	 */
	private ResponseHeaderOverrides overrides = new ResponseHeaderOverrides();
	/**
	 * 指定服务端加密使用的算法及key
	 */
	private SSECustomerKey sseCustomerKey;


	/**
	 * 使用标准的range行为
	 * true：标准的range行为，即0-100表示前101个字节，-100表示文件最后100字节，100-表示文件第101字节至文件末尾（第一个字节表示为0）
	 * false：非标准的range行为（不推荐），即0-100表示前101个字节，-100表示文件前101字节（同0-100），100-表示文件第101字节至文件末尾（第一个字节表示为0）
	 */
	private boolean standardRangeBehavior = true;

	private boolean multiThread;
	/**
	 * 
	 * @param bucketname
	 * @param key
	 */
	public GetObjectRequest(String bucketname,String key)
	{
		this.bucket = bucketname;
		this.key = key;
	}

	@Override
	public void validateParams() throws IllegalArgumentException {
		if(StringUtils.isBlank(this.bucket))
			throw notNull("bucketname");
		if(StringUtils.isBlank(this.key))
			throw notNull("objectkey");
	}
	
	public String getBucket() {
		return bucket;
	}

	public void setBucket(String bucket) {
		this.bucket = bucket;
	}

	public String getKey() {
		return key;
	}

	public void setKey(String key) {
		this.key = key;
	}

	public void setRange(long[] range) {
		this.range = range;
	}

	public void setMultiThread(boolean multiThread) {
		this.multiThread = multiThread;
	}

	public boolean getMultiThread() {
		return this.multiThread;
	}

	public long [] getRange() {
		return range;
	}
	public void setRange(long start,long end) {
		this.range = new long[]{start,end};
	}
	/**
	 * object的etag能匹配到则返回，否则返回结果的ifPreconditionSuccess为false，object为空
	 */
	public List<String> getMatchingETagConstraints() {
		return matchingETagConstraints;
	}
	/**
	 * object的etag能匹配到则返回，否则返回结果的ifPreconditionSuccess为false，object为空
	 */
	public void setMatchingETagConstraints(List<String> matchingETagConstraints) {
		this.matchingETagConstraints = matchingETagConstraints;
	}
	/**
	 * object的etag不同于其中的任何一个，否则返回结果的ifModified为false,object为空
	 */
	public List<String> getNonmatchingEtagConstraints() {
		return nonmatchingEtagConstraints;
	}
	/**
	 * object的etag不同于其中的任何一个，否则返回结果的ifModified为false,object为空
	 */
	public void setNonmatchingEtagConstraints(
			List<String> nonmatchingEtagConstraints) {
		this.nonmatchingEtagConstraints = nonmatchingEtagConstraints;
	}
	/**
	 * 在此时间之后没有被修改过，否则返回结果的ifPreconditionSuccess为false，object为空
	 */
	public Date getUnmodifiedSinceConstraint() {
		return unmodifiedSinceConstraint;
	}
	/**
	 * 在此时间之后没有被修改过，否则返回结果的ifPreconditionSuccess为false，object为空
	 */
	public void setUnmodifiedSinceConstraint(Date unmodifiedSinceConstraint) {
		this.unmodifiedSinceConstraint = unmodifiedSinceConstraint;
	}
	/**
	 * 在此时间之后被修改过，否则返回结果的ifModified为false,object为空
	 */
	public Date getModifiedSinceConstraint() {
		return modifiedSinceConstraint;
	}
	/**
	 * 在此时间之后被修改过，否则返回结果的ifModified为false,object为空
	 */
	public void setModifiedSinceConstraint(Date modifiedSinceConstraint) {
		this.modifiedSinceConstraint = modifiedSinceConstraint;
	}
	public ResponseHeaderOverrides getOverrides() {
		return overrides;
	}
	/**
	 * 修改返回的response的headers
	 */
	public void setOverrides(ResponseHeaderOverrides overrides) {
		this.overrides = overrides;
	}
	public SSECustomerKey getSseCustomerKey() {
		return sseCustomerKey;
	}
	public void setSseCustomerKey(SSECustomerKey sseCustomerKey) {
		this.sseCustomerKey = sseCustomerKey;
	}

	public boolean isStandardRangeBehavior() {
		return standardRangeBehavior;
	}

	public void setStandardRangeBehavior(boolean standardRangeBehavior) {
		this.standardRangeBehavior = standardRangeBehavior;
	}

	@Override
	public void buildRequest(Request request) {
		request.setMethod(HttpMethod.GET);
		request.setBucket(bucket);
		request.setKey(key);

		String rangeValue = getRangeHeaderValue();
		if (rangeValue != null) {
			request.addHeader(HttpHeaders.Range, rangeValue);

			if (isStandardRangeBehavior()) {
				request.addHeader(HttpHeaders.RangeBehavior, "standard");
			}
		}

		if(matchingETagConstraints.size()>0)
			request.addHeader(HttpHeaders.IfMatch, StringUtils.join(matchingETagConstraints, ","));
		if(nonmatchingEtagConstraints.size()>0)
			request.addHeader(HttpHeaders.IfNoneMatch, StringUtils.join(nonmatchingEtagConstraints, ","));
		if(this.unmodifiedSinceConstraint !=null)
			request.addHeader(HttpHeaders.IfUnmodifiedSince, DateUtils.convertDate2Str(this.unmodifiedSinceConstraint, DATETIME_PROTOCOL.RFC1123).toString());
		if(this.modifiedSinceConstraint !=null)
			request.addHeader(HttpHeaders.IfModifiedSince, DateUtils.convertDate2Str(this.modifiedSinceConstraint, DATETIME_PROTOCOL.RFC1123).toString());
		request.getQueryParams().putAll(this.getOverrides().getOverrides());
		//添加服务端加密相关
		request.getHeaders().putAll(HttpUtils.convertSSECustomerKey2Headers(sseCustomerKey));
	}

	private String getRangeHeaderValue() {
		if (range == null) {
			return null;
		}

		if (range.length != 2) {
			log.warn("Invalid range value: {}, ignore it and request for entire object");
			return null;
		}

		long start = range[0];
		long end = range[1];
		if (start < 0 && end < 0 || (end >= 0 && start > end)) {
			log.warn("Invalid range value: {}, ignore it and request for entire object");
			return null;
		}

		if (start < 0) {
			return "bytes=-" + end;
		}

		if (end < 0) {
			return "bytes=" + start + "-";
		}

		return "bytes=" + start + "-" + end;
	}
	
}
