package com.ksyun.ks3.service.request;

import com.ksyun.ks3.dto.*;
import com.ksyun.ks3.http.HttpHeaders;
import com.ksyun.ks3.http.HttpMethod;
import com.ksyun.ks3.http.Request;
import com.ksyun.ks3.utils.Md5Utils;
import com.ksyun.ks3.utils.StringUtils;
import com.ksyun.ks3.utils.XmlWriter;

import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import static com.ksyun.ks3.exception.client.ClientIllegalArgumentExceptionGenerator.*;

/**
 * 设置object的tagging
 */
public class PutObjectTaggingRequest extends Ks3WebServiceRequest{
	private String bucket;
	private String key;

    private ObjectTagging tagging;
	private Pattern TAG_PATTERN = Pattern.compile("^[\\w\\-+=.:/][\\w\\-+=.:/\\s]*(?<!\\s)$");
    public String getBucket() {
		return bucket;
	}

	public void setBucket(String bucket) {
		this.bucket = bucket;
	}

	public String getKey() {
		return key;
	}

	public void setKey(String key) {
		this.key = key;
	}

    public ObjectTagging getObjectTagging() {
		return tagging;
	}

	public void setObjectTagging(ObjectTagging tagging) {
		this.tagging = tagging;
	}

    @Override
	public void validateParams() throws IllegalArgumentException {
    	if(StringUtils.isBlank(this.bucket)){
    		throw notNull("bucketname");
    	}
    	if(StringUtils.isBlank(this.key)){
    		throw notNull("objectkey");
    	}
    	if(this.tagging == null){
			throw notNull("objectTagging");
		}

		if (tagging.getTagSet() == null || tagging.getTagSet().size() == 0) {
			throw between("tagset", "0", "1", "10");
		}

		if (tagging.getTagSet().size() > 10) {
			throw between("tagset", "" + tagging.getTagSet().size(), "1", "10");
		}

		List<String> tagKeys = new ArrayList<String>();
		for (ObjectTag tag : tagging.getTagSet()) {
			if (!TAG_PATTERN.matcher(tag.getKey()).matches()) {
				throw notCorrect("key", tag.getKey(), "invalid key format");
			}

			if (tag.getValue() != null && !TAG_PATTERN.matcher(tag.getValue()).matches()) {
				throw notCorrect("value", tag.getValue(), "invalid value format");
			}

			if (tag.getKey().getBytes(StandardCharsets.UTF_8).length > 128) {
				throw notCorrect("key", tag.getKey(), "invalid length");
			}

			if (tag.getValue() != null && tag.getValue().getBytes(StandardCharsets.UTF_8).length > 256) {
				throw notCorrect("value", tag.getValue(), "invalid length");
			}

			if (!tagKeys.contains(tag.getKey())) {
				tagKeys.add(tag.getKey());
			} else {
				throw notCorrect("key", tag.getKey(), "duplicated tagging key");
			}
		}

    }
    public PutObjectTaggingRequest(String bucketName, String objectName) {
    	this.bucket = bucketName;
    	this.key = objectName;
    }
    public PutObjectTaggingRequest(String bucketName, String objectName, ObjectTagging objectTagging){
    	this(bucketName,objectName);
        this.setObjectTagging(objectTagging);
    }

	@Override
	public void buildRequest(Request request) {
		request.setMethod(HttpMethod.PUT);
		request.setBucket(bucket);
		request.setKey(key);
		request.addQueryParam("tagging", "");
		XmlWriter writer = new XmlWriter();
		writer.start("Tagging");
		writer.start("TagSet");
		List<ObjectTag> tags = tagging.getTagSet();
		for(ObjectTag tag : tags) {
			writer.start("Tag");
			writer.start("Key").value(tag.getKey()).end();
			if (tag.getValue() != null) {
				writer.start("Value").value(tag.getValue()).end();
			}
			writer.end();
		}
		writer.end();
		writer.end();
		String xml = writer.toString();
		request.addHeader(HttpHeaders.ContentMD5, Md5Utils.md5AsBase64(xml.getBytes()));
		request.setContent(new ByteArrayInputStream(xml.getBytes()));
	}
}
