
package com.ksyun.ks3.signer;


import com.ksyun.ks3.annotation.SdkProtectedApi;
import com.ksyun.ks3.annotation.SdkTestInternalApi;

import java.util.Date;

/**
 * Clock interface to prevent static coupling to {@link System#currentTimeMillis()}.
 */
@SdkProtectedApi
public interface SdkClock {

    /**
     * Standard implementation that calls out to {@link System#currentTimeMillis()}. Used in production code.
     */
    SdkClock STANDARD = new SdkClock() {
        @Override
        public long currentTimeMillis() {
            return System.currentTimeMillis();
        }
    };

    long currentTimeMillis();

    /**
     * Mock implementation used in tests.
     */
    final class MockClock implements SdkClock {
        private final long mockedTime;

        public MockClock(Date mockedTime) {
            this(mockedTime.getTime());
        }

        public MockClock(long mockedTime) {
            this.mockedTime = mockedTime;
        }

        @Override
        public long currentTimeMillis() {
            return mockedTime;
        }
    }

    final class Instance {

        private static SdkClock clock = STANDARD;

        public static SdkClock get() {
            return clock;
        }

        /**
         * Should only be used by tests to mock the clock.
         *
         * @param newClock New clock to use.
         */
        @SdkTestInternalApi
        public static void set(SdkClock newClock) {
            clock = newClock;
        }

        /**
         * Reset the clock to {@link #STANDARD}. Should only be used by SDK tests.
         */
        @SdkTestInternalApi
        public static void reset() {
            clock = STANDARD;
        }

    }
}
