
package com.ksyun.ks3.signer.internal;

import com.ksyun.ks3.http.Request;
import com.ksyun.ks3.signer.SdkClock;
import com.ksyun.ks3.utils.KSS4SignerUtils;

import java.util.Date;

/**
 * Parameters that are used for computing a Kss 4 signature for a request.
 */
public final class KSSSignerRequestParams {

    /**
     * The request for which the signature needs to be computed.
     */
    private final Request request;

    /**
     * The datetime in milliseconds for which the signature needs to be
     * computed.
     */
    private final long signingDateTimeMilli;

    /**
     * The scope of the signature.
     */
    private final String scope;

    /**
     * The AWS region to be used for computing the signature.
     */
    private final String regionName;

    /**
     * The name of the KSS service.
     */
    private final String serviceName;

    /**
     * UTC formatted version of the signing time stamp.
     */
    private final String formattedSigningDateTime;

    /**
     * UTC Formatted Signing date with time stamp stripped
     */
    private final String formattedSigningDate;

    /**
     * The signing algorithm to be used for computing the signature.
     */
    private final String signingAlgorithm;


    public KSSSignerRequestParams(Request request,
                                  Date signingDateOverride, String regionNameOverride,
                                  String serviceName, String signingAlgorithm) {
        if (request == null) {
            throw new IllegalArgumentException("Request cannot be null");
        }
        if (signingAlgorithm == null) {
            throw new IllegalArgumentException(
                    "Signing Algorithm cannot be null");
        }
        this.request = request;
        this.signingDateTimeMilli = signingDateOverride != null ? signingDateOverride
                .getTime() : getSigningDate(request);
        this.formattedSigningDate = KSS4SignerUtils
                .formatDateStamp(signingDateTimeMilli);
        this.serviceName = serviceName;

        this.regionName = regionNameOverride;

        this.scope = generateScope(request, formattedSigningDate, this.serviceName,
                regionName);
        this.formattedSigningDateTime = KSS4SignerUtils
                .formatTimestamp(signingDateTimeMilli);
        this.signingAlgorithm = signingAlgorithm;
    }



    /**
     * Returns the signing date from the request.
     */
    private final long getSigningDate(Request request) {
        return SdkClock.Instance.get().currentTimeMillis() - request.getExpires().getTime();
//        return SdkClock.Instance.get().currentTimeMillis() - request.getTimeOffset() * 1000L;
    }

    /**
     * Returns the scope to be used for the signing.
     */
    private String generateScope(Request request, String dateStamp,
                                 String serviceName, String regionName) {
        final StringBuilder scopeBuilder = new StringBuilder();
        return scopeBuilder.append(dateStamp).append("/").append(regionName)
                .append("/").append(serviceName).append("/")
                .append("kss4_request").toString();
    }

    /**
     * Returns the request for which the signing needs to be done.
     */
    public Request getRequest() {
        return request;
    }

    /**
     * Returns the scope of the signing.
     */
    public String getScope() {
        return scope;
    }

    /**
     * Returns the formatted date and time of the signing date in UTC zone.
     */
    public String getFormattedSigningDateTime() {
        return formattedSigningDateTime;
    }

    /**
     * Returns the signing date time in millis for which the signature needs to
     * be computed.
     */
    public long getSigningDateTimeMilli() {
        return signingDateTimeMilli;
    }

    /**
     * Returns the KSS region name to be used while computing the signature.
     */
    public String getRegionName() {
        return regionName;
    }

    /**
     * Returns the KSS Service name to be used while computing the signature.
     */
    public String getServiceName() {
        return serviceName;
    }

    /**
     * Returns the formatted date in UTC zone of the signing date.
     */
    public String getFormattedSigningDate() {
        return formattedSigningDate;
    }

    /**
     * Returns the signing algorithm used for computing the signature.
     */
    public String getSigningAlgorithm() {
        return signingAlgorithm;
    }
}
