package com.ksyun.ks3.utils;

import com.ksyun.ks3.Crc64CheckedInputStream;
import com.ksyun.ks3.InputSubStream;
import com.ksyun.ks3.dto.PartETag;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.nio.file.Files;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.zip.CheckedInputStream;

public class CRC64Utils {
    private static final Log log = LogFactory.getLog(CRC64Utils.class);

    private static final int SIXTEEN_K = 1 << 14;

    private CRC64Utils() {

    }

    public static String getCrc64Ecma(File file) throws IOException {
        try (CheckedInputStream checkedInputstream = new Crc64CheckedInputStream(Files.newInputStream(file.toPath()))) {
            byte[] buffer = new byte[SIXTEEN_K];
            int bytesRead = 0;
            while (bytesRead != -1) {
                bytesRead = checkedInputstream.read(buffer, 0, buffer.length);
            }
            return toUnsignedString(checkedInputstream.getChecksum().getValue());
        }
    }

    public static String getCrc64Ecma(File file, long startIndex, long endIndex) throws IOException {
        try (InputStream in = Files.newInputStream(file.toPath());
             InputSubStream inputSubStream = new InputSubStream(in, startIndex, endIndex - startIndex + 1, true);
             CheckedInputStream checkedInputstream = new Crc64CheckedInputStream(inputSubStream);) {

            byte[] buffer = new byte[SIXTEEN_K];
            int bytesRead = 0;
            while (bytesRead != -1) {
                bytesRead = checkedInputstream.read(buffer, 0, buffer.length);
            }
            return toUnsignedString(checkedInputstream.getChecksum().getValue());
        }
    }

    public static String getCrc64Ecma(List<PartETag> partETags) {
        if (partETags == null || partETags.isEmpty()) {
            return null;
        }
        Collections.sort(partETags, new Comparator<PartETag>() {
            @Override
            public int compare(PartETag o1, PartETag o2) {
                return o1.getPartNumber() - o2.getPartNumber();
            }
        });

        Long firstCrc64Ecma = parseServerCrc64(partETags.get(0).getCrc64Ecma());
        if (firstCrc64Ecma == null) {
            return null;
        }
        CRC64 crc64 = new CRC64(firstCrc64Ecma);
        for (int i = 1; i < partETags.size(); i++) {
            PartETag partETag = partETags.get(i);
            Long crc64Ecma = parseServerCrc64(partETag.getCrc64Ecma());
            if (crc64Ecma == null) {
                return null;
            }
            crc64 = CRC64.combine(crc64, new CRC64(crc64Ecma), partETag.getPartSize());
        }
        return toUnsignedString(crc64.getValue());
    }

    public static Long parseServerCrc64(String value) {
        if (StringUtils.isBlank(value) || !StringUtils.isNumeric(value)) {
            return null;
        }

        return new BigInteger(value).longValue();
    }

    public static String toUnsignedString(long value) {
        return toUnsignedString(value, 10);
    }

    public static String toUnsignedString(long value, int radix) {
        BigInteger unsignedValue = BigInteger.valueOf(value).and(new BigInteger("FFFFFFFFFFFFFFFF", 16));
        return unsignedValue.toString(radix);
    }
}
