package com.ksyun.ks3.dto;

import com.ksyun.ks3.exception.client.ClientIllegalArgumentException;
import com.ksyun.ks3.utils.URLEncodeUtils;

import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import static com.ksyun.ks3.exception.client.ClientIllegalArgumentExceptionGenerator.between;
import static com.ksyun.ks3.exception.client.ClientIllegalArgumentExceptionGenerator.notCorrect;

public class ObjectTagging implements Serializable {
    private static final Pattern TAG_PATTERN = Pattern.compile("^[\\w\\-+=.:/][\\w\\-+=.:/\\s]*(?<!\\s)$");

    protected List<ObjectTag> tagSet;

    public ObjectTagging() {}

    public ObjectTagging(List<ObjectTag> tagSet) {
        this.tagSet = tagSet;
    }

    public List<ObjectTag> getTagSet() {
        return tagSet;
    }

    public void setTagSet(List<ObjectTag> tagSet) {
        this.tagSet = tagSet;
    }

    @Override
    public String toString() {
        return "ObjectTagging{" +
                "tagSet=" + tagSet +
                '}';
    }

    public void validate() throws ClientIllegalArgumentException {
        if (getTagSet().size() > 10) {
            throw between("tagSet", "" + getTagSet().size(), "1", "10");
        }

        List<String> tagKeys = new ArrayList<String>();
        for (ObjectTag tag : getTagSet()) {
            String key = tag.getKey();
            String value = tag.getValue();

            if (!TAG_PATTERN.matcher(key).matches()) {
                throw notCorrect("key", key, "invalid key format");
            }

            if (value != null && !"".equals(value) && !TAG_PATTERN.matcher(value).matches()) {
                throw notCorrect("value", value, "invalid value format");
            }

            if (key.getBytes(StandardCharsets.UTF_8).length > 128) {
                throw notCorrect("key", key, "invalid length");
            }

            if (value != null && value.getBytes(StandardCharsets.UTF_8).length > 256) {
                throw notCorrect("value", value, "invalid length");
            }

            if (!tagKeys.contains(key)) {
                tagKeys.add(key);
            } else {
                throw notCorrect("key", key, "duplicated tagging key");
            }
        }
    }

    public String toHeaderValue() {
        StringBuilder builder = new StringBuilder();
        List<ObjectTag> tagList = getTagSet();
        for (int i = 0; i < tagList.size(); i++) {
            if (i > 0) {
                builder.append("&");
            }
            ObjectTag tag = tagList.get(i);
            String encodedKey = encode(tag.getKey());
            String encodedValue = encode(tag.getValue());
            builder.append(encodedKey).append("=").append(encodedValue);
        }
        return builder.toString();
    }

    private String encode(String str) {
        if (str == null) {
            return "";
        }
        // 空格会被encode为+，需替换为%20
        return URLEncodeUtils.encode(str).replace("+", "%20");
    }
}


