package com.ksyun.ks3.exception;

import com.ksyun.ks3.config.Constants;
import com.ksyun.ks3.utils.Jackson;
import com.ksyun.ks3.utils.StringUtils;
import com.ksyun.ks3.utils.XmlReader;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpRequestBase;
import org.w3c.dom.Document;

import java.io.InputStream;
import java.util.Map;

/**
 * @author lijunwei[lijunwei@kingsoft.com]  
 * 
 * @date 2014年10月14日 下午5:55:37
 * 
 * @description 当抛出该异常时表示请求正常执行，但是Ks3服务器抛出异常，详见<a
 *              href="http://ks3.ksyun.com/doc/api/index.html"
 *              >http://ks3.ksyun.com/doc/api/index.html</a>最下面
 **/
public class Ks3ServiceException extends Ks3ClientException {
	/**
	 * 
	 */
	private static final long serialVersionUID = 5225806336951827450L;
	private static final Log log = LogFactory.getLog(Ks3ServiceException.class);

	/** 错误码 */
	private String errorCode;
	/** 状态码 */
	private int statusCode;
	/** 期望的状态码 */
	private String expectedStatusCode;
	/** 错误 */
	private String errorMessage;
	/** 用户请求的资源 */
	private String resource;
	private String requestId;

	private static final String DEFAULT_VALUE = "unknown";

	public Ks3ServiceException() {
		super("");
	}
	public Ks3ServiceException(HttpRequestBase request,HttpResponse response, String expected){
		super("");
		this.expectedStatusCode = expected;
		this.statusCode = response.getStatusLine().getStatusCode();
		String content = "";
		try {
			HttpEntity entity = response.getEntity();
			if (entity == null) {
				return;
			}
			InputStream  in =  entity.getContent();
			content = StringUtils.inputStream2String(in);
			// 是合法的xml 或者 html
			if (content.startsWith("<?xml") || content.startsWith("<!")) {
				log.debug(content);
				Document document = new XmlReader(content).getDocument();
				errorMessage = getByTagName(document, "Message");
				errorCode = getByTagName(document, "Code");
				resource = getByTagName(document, "Resource");
				requestId = getByTagName(document, "RequestId");
				String condition = getByTagName(document, "Condition", "");
				if (!StringUtils.isBlank(condition)) {
					this.errorMessage += " Condition: " + condition;
				}
			} else { // 是json
				Map<String, String> map = Jackson.fromJsonString(content, Map.class);
				errorCode = map.containsKey("errorCode") ? map.get("errorCode") : "400";
				errorMessage = map.containsKey("tipInfo") ? map.get("tipInfo") : "Bad Request";
			}
		} catch (Exception e) {
			log.error("Parse response content error, statusCode: " + this.statusCode + ", content: " + content, e);
		} finally {
			if (request != null) {
				request.abort();
			}
			try {
				HttpEntity entity = response.getEntity();
				if (entity != null && entity.getContent() != null) {
					entity.getContent().close();
				}
			} catch (Exception e) {
				log.error("", e);
			}
		}
	}

	private String getByTagName(Document document, String tagName) {
		return getByTagName(document, tagName, DEFAULT_VALUE);
	}

	private String getByTagName(Document document, String tagName, String defaultValue) {
		try {
			return document.getElementsByTagName(tagName).item(0).getTextContent();
		} catch (Exception e) {
			// do nothing
		}
		return defaultValue;
	}

	public Ks3ServiceException(HttpResponse response, String expected) {
		this(null,response,expected);
	}

	@Override
	public String toString() {
		return this.getClass().getName() + ": " + getMessage();
	}

	public String getErrorCode() {
		return errorCode;
	}

	/**
	 * @deprecated use getStatusCode() instead
	 * @return status code
	 */
	@Deprecated
	public int getStatueCode() {
		return getStatusCode();
	}

	public int getStatusCode() {
		return statusCode;
	}

	public void setStatusCode(int statusCode) {
		this.statusCode = statusCode;
	}

	public String getErrorMessage() {
		return errorMessage;
	}

	public String getResource() {
		return this.resource;
	}

	public String getRequestId() {
		return requestId;
	}

	public void setRequestId(String requestId) {
		this.requestId = requestId;
	}

	/**
	 * @deprecated use getExpectedStatusCode() instead
	 * @return expected status code
	 */
	@Deprecated
	public String getExpectedStatueCode() {
		return getExpectedStatusCode();
	}

	/**
	 * @deprecated use setExpectedStatusCode() instead
	 * @param expectedStatusCode expected status code
	 */
	@Deprecated
	public void setExpectedStatueCode(String expectedStatusCode) {
		setExpectedStatusCode(expectedStatusCode);
	}

	public String getExpectedStatusCode() {
		return expectedStatusCode;
	}

	public void setExpectedStatusCode(String expectedStatusCode) {
		this.expectedStatusCode = expectedStatusCode;
	}

	public void setErrorCode(String errorCode) {
		this.errorCode = errorCode;
	}

	@Deprecated
	public void setStatueCode(int statusCode) {
		setStatusCode(statusCode);
	}

	public void setErrorMessage(String message) {
		this.errorMessage = message;
	}

	public void setResource(String resource) {
		this.resource = resource;
	}

	// 将当前异常转化为com.ksyun.ks3.exception.serviceside.*下的异常
	public <X extends Ks3ServiceException> RuntimeException convert(String reqid) {
		if(!StringUtils.isBlank(reqid))
			this.setRequestId(reqid);
		if (StringUtils.isBlank(this.getErrorCode())
				|| "unknow".equals(this.getErrorCode()) || "unknown".equalsIgnoreCase(this.getErrorCode())) {
			if(this.statusCode == 400)
				this.setErrorCode("InvalidArgument");
			else if (this.statusCode == 403)
				this.setErrorCode("AccessDenied");
			else if(this.statusCode == 404)
				this.setErrorCode("NotFound");
			else if(this.statusCode == 405)
				this.setErrorCode("MethodNotAllowed");
		}
		String error  = this.getErrorCode();
		if(!StringUtils.isBlank(error))
		    error = error.substring(0,1).toUpperCase()+error.substring(1);
		String classString = Constants.KS3_PACAKAGE + ".exception.serviceside."
				+ error + "Exception";
		try {
			@SuppressWarnings("unchecked")
			X e = (X) Class.forName(classString).newInstance();
			e.setErrorMessage(this.getErrorMessage());
			e.setErrorCode(this.getErrorCode());
			e.setExpectedStatusCode(this.getExpectedStatusCode());
			e.setRequestId(this.getRequestId());
			e.setResource(this.getResource());
			e.setStatusCode(this.getStatusCode());
			e.setStackTrace(this.getStackTrace());
			return e;
		} catch (Throwable e) {
			return this;
		}

	}

	@Override
	public String getMessage() {
		return "[RequestId:" + this.requestId
				+ ", Resource:" + resource
				+ ", Status code:" + this.statusCode
				+ ", Expected status code:" + this.expectedStatusCode
				+ ", Error code:" + this.errorCode
				+ ", Error message:" + this.errorMessage
				+ "]";
	}
}
