package com.ksyun.ks3.service.request;

import com.ksyun.ks3.http.HttpHeaders;
import com.ksyun.ks3.http.HttpMethod;
import com.ksyun.ks3.http.Request;
import com.ksyun.ks3.utils.Md5Utils;
import com.ksyun.ks3.utils.StringUtils;
import com.ksyun.ks3.utils.XmlWriter;

import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;

import static com.ksyun.ks3.exception.client.ClientIllegalArgumentExceptionGenerator.notNull;

public class RestoreObjectRequest extends Ks3WebServiceRequest {
	
	private String bucket;
	
	private String key;

	/**
	 * Object的解冻天数，取值范围：归档存储为 1~7，深度冷归档存储为1~365，单位为天
	 */
	private int expirationInDays = 1;

	private RestoreTier tier;

	public RestoreObjectRequest(String bucketName, String objectKey) {
		this.bucket = bucketName;
		this.key = objectKey;
	}

	public RestoreObjectRequest(String bucketName, String objectKey, int expirationInDays) {
		this.bucket = bucketName;
		this.key = objectKey;
		this.expirationInDays = expirationInDays;
	}

	@Override
	public void validateParams() throws IllegalArgumentException {
		if (StringUtils.isBlank(this.bucket)) {
			throw notNull("bucketname");
		}
		if (StringUtils.isBlank(this.key)) {
			throw notNull("objectkey");
		}
	}

	@Override
	public void buildRequest(Request request) {
		request.setMethod(HttpMethod.POST);
		request.setBucket(bucket);
		request.setKey(key);
		request.addQueryParam("restore", "");

		XmlWriter xmlWriter = new XmlWriter();
		xmlWriter.startWithNs("RestoreRequest");
		xmlWriter.start("Days").value(getExpirationInDays()).end();
		xmlWriter.start("JobParameters");
		if (getTier() != null) {
			xmlWriter.start("Tier").value(getTier().getValue()).end();
		}
		xmlWriter.end();
		xmlWriter.end();
		String xml = xmlWriter.toString();

		byte[] content = xml.getBytes(StandardCharsets.UTF_8);
		request.setContent(new ByteArrayInputStream(content));
		request.addHeader(HttpHeaders.ContentMD5, Md5Utils.md5AsBase64(xml.getBytes()));
		request.addHeader(HttpHeaders.ContentLength, String.valueOf(content.length));
		request.addHeader(HttpHeaders.ContentType, "application/xml");
	}

	public int getExpirationInDays() {
		return expirationInDays;
	}

	public void setExpirationInDays(int expirationInDays) {
		this.expirationInDays = expirationInDays;
	}

	public RestoreTier getTier() {
		return tier;
	}

	public void setTier(RestoreTier tier) {
		this.tier = tier;
	}
}
