package com.lacunasoftware.restpki;


import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;


/**
 * This class generates alphanumeric codes that are easy for humans to read. The codes are intended
 * to be used as "document recovery codes" on printer-friendly versions of signed files, with codes
 * that are (1) easy to read (2) easy to type back with low risk of mistaking similar characters
 * such as "O" and "0" and (3) have a relatively high entropy for the size of the code (high number
 * of possible codes relative to the size of the code, allowing the developer to choose a relatively
 * small code size.
 *
 * To improve readability, the codes contain only uppercase letters and do not include characters
 * like "O", "0", "1" and "I", while still maintaining a relatively high entropy per character.
 * There are 32 possible characters, so every character adds 5 bits to the overall entropy (25%
 * better than using hexadecimal) which results in smaller codes for the same intended entropy. For
 * instance, to generate a code with 80 bits of entropy (2^80 possible codes), an hexadecimal code
 * have to be 20 characters, while a code generated by this class needs only 16 characters.
 */
public class AlphaCode {
	private static final SecureRandom rng = new SecureRandom();
	private static final String alphabet = "ABCDEFGHJKLMNPQRSTUVWXYZ23456789";
	private static int defaultSize = 16;

	/**
	 * Generates an alphanumeric code with the default size (see property DefaultSize). The returned
	 * code is unformatted, and is meant to be stored on your database indexing the document or
	 * related entity. To show the code to users, you should format the code with one of the format()
	 * methods.
	 * @return The alphanumeric code with the default size (unformatted).
	 */
	public static String generate() {
		return generate(defaultSize);
	}

	/**
	 * Generates an alphanumeric code with the given size. Each character contributes 5 bits to the
	 * overall entropy of the generated codes, so for instance a code with size 16 has 80 bits of
	 * entropy (2^80 possible codes). The returned code is unformatted, and is meant to be stored on
	 * your database indexing the document or related entity. To show the code to users, you should
	 * format the code with one of the Format() methods.
	 * @param codeSize Size of the code to be generated.
	 * @return The alphanumeric code with the given size (unformatted).
	 */
	public static String generate(int codeSize) {

		// Allocate a byte array large enough to receive the necessary entropy.
		byte[] bytes = new byte[(int)Math.ceil(codeSize * 5 / 8.0)];

		// Generate the entropy with a cryptographic number generator.
		rng.nextBytes(bytes);

		// Convert random bytes into bits.
		BitSet bits = BitSet.valueOf(bytes);

		// Iterate bits 5-by-5 converting into characters in our alphabet.
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < codeSize; i++) {
			int n = (bits.get(i * 5) ? 1 : 0) << 4
				| (bits.get(i * 5 + 1) ? 1 : 0) << 3
				| (bits.get(i * 5 + 2) ? 1 : 0) << 2
				| (bits.get(i * 5 + 3) ? 1 : 0) << 1
				| (bits.get(i * 5 + 4) ? 1 : 0);
			sb.append(alphabet.charAt(n));
		}
		return sb.toString();
	}

	/**
	 * Formats an alphanumeric code in groups of characters separated by hyphens, for instance a
	 * 16-character code is formatted as XXXX-XXXX-XXXX-XXXX. A guess of the number of groups is made
	 * based on the code size. If you want to specify the number of groups, use the overload
	 * format(string, int) instead.
	 * @param code The unformatted alphanumeric code.
	 * @return The alphanumeric code formatted in groups, or the unformatted code if no good guess of
	 * the number of groups can be made based on the code size.
	 */
	public static String format(String code) {

		if (code == null) {
			throw new IllegalArgumentException("code");
		}

		// Forgive mistakes.
		code = code.trim();

		int nGroups;
		switch (code.length()) {

			// XXXX-XXXX
			case 8:
				nGroups = 2;
				break;

			// XXX-XXX-XXX
			case 9:
				nGroups = 3;
				break;

			// XXXXX-XXXXX
			case 10:
				nGroups = 2;
				break;

			// XXXX-XXXX-XXXX
			case 12:
				nGroups = 3;
				break;

			// XXXXX-XXXXX-XXXXX
			case 15:
				nGroups = 3;
				break;

			// XXXX-XXXX-XXXX-XXXX
			case 16:
				nGroups = 4;
				break;

			// XXXX-XXXX-XXXX-XXXX-XXXX
			case 20:
				nGroups = 5;
				break;

			// XXXX-XXXX-XXXX-XXXX-XXXX-XXXX
			case 24:
				nGroups = 6;
				break;

			// XXXXX-XXXXX-XXXXX-XXXXX-XXXXX
			case 25:
				nGroups = 5;
				break;

			// XXXX-XXXX-XXXX-XXXX-XXXX-XXXX-XXXX
			case 28:
				nGroups = 7;
				break;

			// XXXXX-XXXXX-XXXXX-XXXXX-XXXXX-XXXXX
			case 30:
				nGroups = 6;
				break;

			default:
				nGroups = 1;
				break;
		}

		return format(code, nGroups);
	}

	/**
	 * Formats an alphanumeric code in groups, for instance a 16-character code into 4 groups is
	 * formatted as XXXX-XXXX-XXXX-XXXX. The number of groups should be an exact divisor of the code
	 * size, otherwise the code is returned unformatted. For instance, for a code of size 20, you
	 * might pass nGroups = 4 for XXXXX-XXXXX-XXXXX-XXXX or nGroups = 5 for XXXX-XXXX-XXXX-XXXX-XXXX.
	 * @param code The unformatted alphanumeric code.
	 * @param nGroups The number of groups.
	 * @return The alphanumeric code formatted in blocks, or the unformatted code if the code size is
	 * not divisible by the number of groups.
	 */
	public static String format(String code, int nGroups) {

		if (code == null) {
			throw new IllegalArgumentException("code");
		}
		if (nGroups < 1) {
			throw new IllegalArgumentException("The parameter \"nGroups\" must be >= 1");
		}

		// Simply return the code if nGroup == 1 or if division in groups is not exact.
		if (nGroups == 1 || code.length() % nGroups != 0) {
			return code;
		}

		int charsPerGroup = code.length() / nGroups;
		List<String> groups = new ArrayList<>();
		for (int i = 0; i < charsPerGroup; i++) {
			groups.add(code.substring(i * charsPerGroup, (i + 1) * charsPerGroup));
		}
		return Util.join("-", groups);
	}

	/**
	 * Parses a given alphanumeric code, removing hyphens or other punctuations. This method is
	 * intended to be called to parse a code typed by the user, before using the code to perform a
	 * lookup on the database.
	 * @param formattedCode The formatted alphanumeric code, e.g. XXXX-XXXX-XXXX-XXXX.
	 * @return The unformatted alphanumeric code, e.g.: XXXXXXXXXXXXXXXX.
	 */
	public static String parse(String formattedCode) {
		if (formattedCode == null) {
			throw new IllegalArgumentException("formattedCode");
		}
		return formattedCode.replaceAll("[^A-Za-z0-9]", "");
	}

	/**
	 * Gets the default size for codes (used by the method generate()). Each character contributes
	 * 5 bits to the overall entropy of the generated codes. This value is initially set to 16, which
	 * means 80 bits of entropy (2^80 possible codes).
	 * @return the default size for codes (used by the method generate()).
	 */
	public static int getDefaultSize() {
		return defaultSize;
	}

	/**
	 * Sets the default size for codes (used by the method generate()). Each character contributes
	 * 5 bits to the overall entropy of the generated codes. This value is initially set to 16, which
	 * means 80 bits of entropy (2^80 possible codes).
	 * @param defaultSize the default size for codes (used by the method generate()).
	 */
	public static void setDefaultSize(int defaultSize) {
		AlphaCode.defaultSize = defaultSize;
	}
}
