/*
 * Decompiled with CFR 0.152.
 */
package com.lancedb.lance.spark;

import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import org.apache.spark.sql.Dataset;
import org.apache.spark.sql.Row;
import org.apache.spark.sql.SparkSession;
import org.apache.spark.sql.connector.catalog.TableCatalog;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

public abstract class SparkLanceNamespaceTestBase {
    protected SparkSession spark;
    protected TableCatalog catalog;
    protected String catalogName = "lance_ns";
    @TempDir
    protected Path tempDir;

    @BeforeEach
    void setup() throws IOException {
        this.spark = SparkSession.builder().appName("lance-namespace-test").master("local").config("spark.sql.catalog." + this.catalogName, "com.lancedb.lance.spark.LanceNamespaceSparkCatalog").config("spark.sql.catalog." + this.catalogName + ".impl", this.getNsImpl()).getOrCreate();
        Map<String, String> additionalConfigs = this.getAdditionalNsConfigs();
        for (Map.Entry<String, String> entry : additionalConfigs.entrySet()) {
            this.spark.conf().set("spark.sql.catalog." + this.catalogName + "." + entry.getKey(), entry.getValue());
        }
        this.catalog = (TableCatalog)this.spark.sessionState().catalogManager().catalog(this.catalogName);
    }

    @AfterEach
    void tearDown() throws IOException {
        if (this.spark != null) {
            this.spark.stop();
        }
    }

    protected abstract String getNsImpl();

    protected Map<String, String> getAdditionalNsConfigs() {
        return new HashMap<String, String>();
    }

    protected boolean supportsNamespace() {
        return false;
    }

    protected String generateTableName(String baseName) {
        return baseName + "_" + UUID.randomUUID().toString().replace("-", "");
    }

    @Test
    public void testCreateAndDescribeTable() throws Exception {
        String tableName = this.generateTableName("test_table");
        this.spark.sql("CREATE TABLE " + this.catalogName + ".default." + tableName + " (id BIGINT NOT NULL, name STRING)");
        Dataset describeResult = this.spark.sql("DESCRIBE TABLE " + this.catalogName + ".default." + tableName);
        List columns = describeResult.collectAsList();
        Assertions.assertEquals((int)2, (int)columns.size());
        Row idColumn = (Row)columns.get(0);
        Assertions.assertEquals((Object)"id", (Object)idColumn.getString(0));
        Assertions.assertEquals((Object)"bigint", (Object)idColumn.getString(1));
        Row nameColumn = (Row)columns.get(1);
        Assertions.assertEquals((Object)"name", (Object)nameColumn.getString(0));
        Assertions.assertEquals((Object)"string", (Object)nameColumn.getString(1));
    }

    @Test
    public void testListTables() throws Exception {
        String tableName1 = this.generateTableName("list_test_1");
        String tableName2 = this.generateTableName("list_test_2");
        this.spark.sql("CREATE TABLE " + this.catalogName + ".default." + tableName1 + " (id BIGINT NOT NULL)");
        this.spark.sql("CREATE TABLE " + this.catalogName + ".default." + tableName2 + " (id BIGINT NOT NULL)");
        Dataset tablesResult = this.spark.sql("SHOW TABLES IN " + this.catalogName + ".default");
        List tables = tablesResult.collectAsList();
        Assertions.assertTrue((tables.size() >= 2 ? 1 : 0) != 0);
        boolean foundTable1 = false;
        boolean foundTable2 = false;
        for (Row row : tables) {
            String tableName = row.getString(1);
            if (tableName1.equals(tableName)) {
                foundTable1 = true;
            }
            if (!tableName2.equals(tableName)) continue;
            foundTable2 = true;
        }
        Assertions.assertTrue((boolean)foundTable1);
        Assertions.assertTrue((boolean)foundTable2);
    }

    @Test
    public void testDropTable() throws Exception {
        String tableName = this.generateTableName("drop_test");
        this.spark.sql("CREATE TABLE " + this.catalogName + ".default." + tableName + " (id BIGINT NOT NULL)");
        Dataset result = this.spark.sql("SELECT COUNT(*) FROM " + this.catalogName + ".default." + tableName);
        Assertions.assertNotNull((Object)result);
        Assertions.assertEquals((long)0L, (long)((Row)result.collectAsList().get(0)).getLong(0));
        this.spark.sql("DROP TABLE " + this.catalogName + ".default." + tableName);
        Assertions.assertThrows(Exception.class, () -> this.spark.sql("SELECT COUNT(*) FROM " + this.catalogName + ".default." + tableName).collectAsList());
    }

    @Test
    public void testLoadSparkTable() throws Exception {
        String existingTableName = this.generateTableName("existing_table");
        this.spark.sql("CREATE TABLE " + this.catalogName + ".default." + existingTableName + " (id BIGINT NOT NULL, name STRING)");
        this.spark.sql("INSERT INTO " + this.catalogName + ".default." + existingTableName + " VALUES (1, 'test')");
        Dataset table = this.spark.table(this.catalogName + ".default." + existingTableName);
        Assertions.assertNotNull((Object)table);
        Assertions.assertEquals((long)1L, (long)table.count());
        String nonExistentTableName = this.generateTableName("non_existent");
        Assertions.assertThrows(Exception.class, () -> this.spark.table(this.catalogName + ".default." + nonExistentTableName));
    }

    @Test
    public void testSparkSqlSelect() throws Exception {
        String tableName = this.generateTableName("sql_test_table");
        this.spark.sql("CREATE TABLE " + this.catalogName + ".default." + tableName + " (id INT NOT NULL, name STRING, value DOUBLE)");
        this.spark.sql("INSERT INTO " + this.catalogName + ".default." + tableName + " VALUES (1, 'Alice', 100.0), (2, 'Bob', 200.0), (3, 'Charlie', 300.0)");
        Dataset result = this.spark.sql("SELECT * FROM " + this.catalogName + ".default." + tableName);
        Assertions.assertEquals((long)3L, (long)result.count());
        Dataset filtered = this.spark.sql("SELECT * FROM " + this.catalogName + ".default." + tableName + " WHERE id > 1");
        Assertions.assertEquals((long)2L, (long)filtered.count());
        Dataset aggregated = this.spark.sql("SELECT COUNT(*) as cnt FROM " + this.catalogName + ".default." + tableName);
        Assertions.assertEquals((long)3L, (long)((Row)aggregated.collectAsList().get(0)).getLong(0));
        Dataset projected = this.spark.sql("SELECT name, value FROM " + this.catalogName + ".default." + tableName + " WHERE id = 2");
        Row row = (Row)projected.collectAsList().get(0);
        Assertions.assertEquals((Object)"Bob", (Object)row.getString(0));
        Assertions.assertEquals((double)200.0, (double)row.getDouble(1), (double)0.001);
    }

    @Test
    public void testSparkSqlJoin() throws Exception {
        String tableName1 = this.generateTableName("join_table_1");
        String tableName2 = this.generateTableName("join_table_2");
        this.spark.sql("CREATE TABLE " + this.catalogName + ".default." + tableName1 + " (id INT NOT NULL, name STRING)");
        this.spark.sql("INSERT INTO " + this.catalogName + ".default." + tableName1 + " VALUES (1, 'Alice'), (2, 'Bob'), (3, 'Charlie')");
        this.spark.sql("CREATE TABLE " + this.catalogName + ".default." + tableName2 + " (id INT NOT NULL, score INT)");
        this.spark.sql("INSERT INTO " + this.catalogName + ".default." + tableName2 + " VALUES (1, 95), (2, 87), (3, 92)");
        Dataset joined = this.spark.sql("SELECT t1.name, t2.score FROM " + this.catalogName + ".default." + tableName1 + " t1 JOIN " + this.catalogName + ".default." + tableName2 + " t2 ON t1.id = t2.id");
        Assertions.assertEquals((long)3L, (long)joined.count());
        List results = joined.orderBy("name", new String[0]).collectAsList();
        Assertions.assertEquals((Object)"Alice", (Object)((Row)results.get(0)).getString(0));
        Assertions.assertEquals((int)95, (int)((Row)results.get(0)).getInt(1));
        Assertions.assertEquals((Object)"Bob", (Object)((Row)results.get(1)).getString(0));
        Assertions.assertEquals((int)87, (int)((Row)results.get(1)).getInt(1));
        Assertions.assertEquals((Object)"Charlie", (Object)((Row)results.get(2)).getString(0));
        Assertions.assertEquals((int)92, (int)((Row)results.get(2)).getInt(1));
    }

    @Test
    public void testCreateAndDropNamespace() throws Exception {
        if (!this.supportsNamespace()) {
            return;
        }
        String namespaceName = "test_ns_" + UUID.randomUUID().toString().replace("-", "");
        this.spark.sql("CREATE NAMESPACE " + this.catalogName + "." + namespaceName);
        Dataset namespaces = this.spark.sql("SHOW NAMESPACES IN " + this.catalogName);
        List nsList = namespaces.collectAsList();
        boolean found = false;
        for (Row row : nsList) {
            if (!namespaceName.equals(row.getString(0))) continue;
            found = true;
            break;
        }
        Assertions.assertTrue((boolean)found);
        this.spark.sql("DROP NAMESPACE " + this.catalogName + "." + namespaceName);
        namespaces = this.spark.sql("SHOW NAMESPACES IN " + this.catalogName);
        nsList = namespaces.collectAsList();
        found = false;
        for (Row row : nsList) {
            if (!namespaceName.equals(row.getString(0))) continue;
            found = true;
            break;
        }
        Assertions.assertFalse((boolean)found);
    }

    @Test
    public void testListNamespaces() throws Exception {
        if (!this.supportsNamespace()) {
            return;
        }
        String namespace1 = "list_ns_1_" + UUID.randomUUID().toString().replace("-", "");
        String namespace2 = "list_ns_2_" + UUID.randomUUID().toString().replace("-", "");
        this.spark.sql("CREATE NAMESPACE " + this.catalogName + "." + namespace1);
        this.spark.sql("CREATE NAMESPACE " + this.catalogName + "." + namespace2);
        Dataset namespaces = this.spark.sql("SHOW NAMESPACES IN " + this.catalogName);
        List nsList = namespaces.collectAsList();
        boolean foundNs1 = false;
        boolean foundNs2 = false;
        for (Row row : nsList) {
            String ns = row.getString(0);
            if (namespace1.equals(ns)) {
                foundNs1 = true;
            }
            if (!namespace2.equals(ns)) continue;
            foundNs2 = true;
        }
        Assertions.assertTrue((boolean)foundNs1);
        Assertions.assertTrue((boolean)foundNs2);
    }

    @Test
    public void testNamespaceMetadata() throws Exception {
        if (!this.supportsNamespace()) {
            return;
        }
        String namespaceName = "metadata_ns_" + UUID.randomUUID().toString().replace("-", "");
        this.spark.sql("CREATE NAMESPACE " + this.catalogName + "." + namespaceName + " WITH DBPROPERTIES ('key1'='value1', 'key2'='value2')");
        Dataset properties = this.spark.sql("DESCRIBE NAMESPACE EXTENDED " + this.catalogName + "." + namespaceName);
        List propList = properties.collectAsList();
        Assertions.assertNotNull((Object)propList);
        Assertions.assertTrue((propList.size() > 0 ? 1 : 0) != 0);
    }

    @Test
    public void testNamespaceWithTables() throws Exception {
        if (!this.supportsNamespace()) {
            return;
        }
        String namespaceName = "tables_ns_" + UUID.randomUUID().toString().replace("-", "");
        String tableName = this.generateTableName("ns_table");
        this.spark.sql("CREATE NAMESPACE " + this.catalogName + "." + namespaceName);
        this.spark.sql("CREATE TABLE " + this.catalogName + "." + namespaceName + "." + tableName + " (id BIGINT NOT NULL, name STRING)");
        this.spark.sql("INSERT INTO " + this.catalogName + "." + namespaceName + "." + tableName + " VALUES (1, 'test')");
        Dataset result = this.spark.sql("SELECT * FROM " + this.catalogName + "." + namespaceName + "." + tableName);
        Assertions.assertEquals((long)1L, (long)result.count());
        Dataset tables = this.spark.sql("SHOW TABLES IN " + this.catalogName + "." + namespaceName);
        List tableList = tables.collectAsList();
        Assertions.assertEquals((int)1, (int)tableList.size());
        Assertions.assertEquals((Object)tableName, (Object)((Row)tableList.get(0)).getString(1));
    }

    @Test
    public void testCascadeDropNamespace() throws Exception {
        if (!this.supportsNamespace()) {
            return;
        }
        String namespaceName = "cascade_ns_" + UUID.randomUUID().toString().replace("-", "");
        String tableName = this.generateTableName("cascade_table");
        this.spark.sql("CREATE NAMESPACE " + this.catalogName + "." + namespaceName);
        this.spark.sql("CREATE TABLE " + this.catalogName + "." + namespaceName + "." + tableName + " (id BIGINT NOT NULL)");
        Assertions.assertThrows(Exception.class, () -> this.spark.sql("DROP NAMESPACE " + this.catalogName + "." + namespaceName));
        this.spark.sql("DROP NAMESPACE " + this.catalogName + "." + namespaceName + " CASCADE");
        Dataset namespaces = this.spark.sql("SHOW NAMESPACES IN " + this.catalogName);
        List nsList = namespaces.collectAsList();
        boolean found = false;
        for (Row row : nsList) {
            if (!namespaceName.equals(row.getString(0))) continue;
            found = true;
            break;
        }
        Assertions.assertFalse((boolean)found);
    }

    @Test
    public void testTwoPartIdentifier() throws Exception {
        String tableName = this.generateTableName("two_part_test");
        this.spark.sql("SET spark.sql.defaultCatalog=" + this.catalogName);
        this.spark.sql("CREATE TABLE default." + tableName + " (id BIGINT NOT NULL, name STRING)");
        Dataset tables = this.spark.sql("SHOW TABLES IN default");
        boolean found = tables.collectAsList().stream().anyMatch(row -> tableName.equals(row.getString(1)));
        Assertions.assertTrue((boolean)found);
        Dataset description = this.spark.sql("DESCRIBE TABLE default." + tableName);
        Assertions.assertEquals((long)2L, (long)description.count());
        this.spark.sql("INSERT INTO default." + tableName + " VALUES (1, 'test')");
        Dataset result = this.spark.sql("SELECT * FROM default." + tableName);
        Assertions.assertEquals((long)1L, (long)result.count());
        Row row2 = (Row)result.collectAsList().get(0);
        Assertions.assertEquals((long)1L, (long)row2.getLong(0));
        Assertions.assertEquals((Object)"test", (Object)row2.getString(1));
    }

    @Test
    public void testOnePartIdentifier() throws Exception {
        String tableName = this.generateTableName("one_part_test");
        this.spark.sql("SET spark.sql.defaultCatalog=" + this.catalogName);
        this.spark.sql("USE default");
        this.spark.sql("CREATE TABLE " + tableName + " (id BIGINT NOT NULL, value DOUBLE)");
        Dataset tables = this.spark.sql("SHOW TABLES");
        boolean found = tables.collectAsList().stream().anyMatch(row -> tableName.equals(row.getString(1)));
        Assertions.assertTrue((boolean)found);
        Dataset description = this.spark.sql("DESCRIBE TABLE " + tableName);
        Assertions.assertEquals((long)2L, (long)description.count());
        this.spark.sql("INSERT INTO " + tableName + " VALUES (42, 3.14)");
        Dataset result = this.spark.sql("SELECT * FROM " + tableName);
        Assertions.assertEquals((long)1L, (long)result.count());
        Row row2 = (Row)result.collectAsList().get(0);
        Assertions.assertEquals((long)42L, (long)row2.getLong(0));
        Assertions.assertEquals((double)3.14, (double)row2.getDouble(1), (double)0.001);
    }
}

