/*
 * Copyright (C) 2015 HaiYang Li
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package com.landawn.abacus.type;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.lang.reflect.Constructor;
import java.lang.reflect.Modifier;
import java.sql.Blob;
import java.sql.SQLException;

import com.landawn.abacus.exception.UncheckedSQLException;
import com.landawn.abacus.parser.SerializationConfig;
import com.landawn.abacus.util.CharacterWriter;
import com.landawn.abacus.util.ClassUtil;
import com.landawn.abacus.util.IOUtil;
import com.landawn.abacus.util.N;

/**
 * The InputStream must be encoded by base64.
 *
 * @author Haiyang Li
 * @since 0.8
 */
public class InputStreamType extends AbstractType<InputStream> {

    public static final String INPUT_STREAM = InputStream.class.getSimpleName();

    private final Class<InputStream> typeClass;

    private final Constructor<?> bytesConstructor;

    private final Constructor<?> streamConstructor;

    InputStreamType() {
        this(INPUT_STREAM);
    }

    InputStreamType(String typeName) {
        super(typeName);

        this.typeClass = InputStream.class;

        this.bytesConstructor = null;
        this.streamConstructor = null;
    }

    InputStreamType(Class<InputStream> cls) {
        super(ClassUtil.getSimpleClassName(cls));

        this.typeClass = cls;

        if (Modifier.isAbstract(cls.getModifiers())) {
            this.bytesConstructor = null;
            this.streamConstructor = null;
        } else {
            this.bytesConstructor = ClassUtil.getDeclaredConstructor(cls, byte[].class);
            this.streamConstructor = ClassUtil.getDeclaredConstructor(cls, InputStream.class);
        }
    }

    @Override
    public Class<InputStream> clazz() {
        return typeClass;
    }

    /**
     * Checks if is input stream.
     *
     * @return true, if is input stream
     */
    @Override
    public boolean isInputStream() {
        return true;
    }

    /**
     *
     * @param x
     * @return
     */
    @Override
    public String stringOf(InputStream x) {
        return x == null ? null : N.base64Encode(IOUtil.readAllBytes(x));
    }

    /**
     *
     * @param str
     * @return
     */
    @Override
    public InputStream valueOf(String str) {
        if (str == null) {
            return null;
        }

        if (bytesConstructor != null) {
            return (InputStream) ClassUtil.invokeConstructor(bytesConstructor, N.base64Decode(str));
        } else if (streamConstructor != null) {
            return (InputStream) ClassUtil.invokeConstructor(streamConstructor, new ByteArrayInputStream(N.base64Decode(str)));
        } else {
            return new ByteArrayInputStream(N.base64Decode(str));
        }
    }

    /**
     *
     * @param obj
     * @return
     */
    @Override
    public InputStream valueOf(final Object obj) {
        if (obj == null) {
            return null;
        } else if (obj instanceof Blob) {
            final Blob blob = (Blob) obj;

            try {
                return blob.getBinaryStream();
            } catch (SQLException e) {
                throw new UncheckedSQLException(e);
            }
        } else {
            return valueOf(N.typeOf(obj.getClass()).stringOf(obj));
        }
    }

    /**
     *
     * @param writer
     * @param t
     * @throws IOException Signals that an I/O exception has occurred.
     */
    @Override
    public void write(Writer writer, InputStream t) throws IOException {
        if (t == null) {
            writer.write(NULL_CHAR_ARRAY);
        } else {
            writer.write(N.base64Encode(IOUtil.readAllBytes(t)));
        }
    }

    /**
     *
     * @param writer
     * @param t
     * @param config
     * @throws IOException Signals that an I/O exception has occurred.
     */
    @Override
    public void writeCharacter(CharacterWriter writer, InputStream t, SerializationConfig<?> config) throws IOException {
        if (t == null) {
            writer.write(NULL_CHAR_ARRAY);
        } else {
            if ((config == null) || (config.getStringQuotation() == 0)) {
                writer.write(N.base64Encode(IOUtil.readAllBytes(t)));
            } else {
                writer.write(config.getStringQuotation());
                writer.write(N.base64Encode(IOUtil.readAllBytes(t)));
                writer.write(config.getStringQuotation());
            }
        }
    }
}
