/*
 * Copyright (C) 2015 HaiYang Li
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package com.landawn.abacus.type;

import java.io.IOException;
import java.io.Writer;
import java.util.Collection;

import com.landawn.abacus.parser.SerializationConfig;
import com.landawn.abacus.util.CharacterWriter;

/**
 *
 * @author Haiyang Li
 * @param <T>
 * @since 0.8
 * @see com.landawn.abacus.util.TypeReference
 * @see com.landawn.abacus.util.TypeReference.TypeToken
 */
public interface Type<T> {

    /**
     *
     * @return
     */
    String name();

    String declaringName();

    String xmlName();

    Class<T> clazz();

    /**
     * Checks if is primitive type.
     *
     * @return true, if is primitive type
     */
    boolean isPrimitiveType();

    /**
     * Checks if is primitive wrapper.
     *
     * @return true, if is primitive wrapper
     */
    boolean isPrimitiveWrapper();

    /**
     * Checks if is primitive list.
     *
     * @return true, if is primitive list
     */
    boolean isPrimitiveList();

    /**
     * Checks if is boolean.
     *
     * @return true, if is boolean
     */
    boolean isBoolean();

    /**
     * Checks if is number.
     *
     * @return true, if is number
     */
    boolean isNumber();

    /**
     * Checks if is string.
     *
     * @return true, if is string
     */
    boolean isString();

    boolean isCharSequence();

    /**
     * Checks if is date.
     *
     * @return true, if is date
     */
    boolean isDate();

    /**
     * Checks if is calendar.
     *
     * @return true, if is calendar
     */
    boolean isCalendar();

    /**
     * Checks if is joda date time.
     *
     * @return true, if is joda date time
     */
    boolean isJodaDateTime();

    /**
     * Checks if is primitive array.
     *
     * @return true, if is primitive array
     */
    boolean isPrimitiveArray();

    /**
     * Checks if is primitive byte array.
     *
     * @return true, if is primitive byte array
     */
    boolean isPrimitiveByteArray();

    /**
     * Checks if is object array.
     *
     * @return true, if is object array
     */
    boolean isObjectArray();

    /**
     * Checks if is array.
     *
     * @return true, if is array
     */
    boolean isArray();

    /**
     * Checks if is list.
     *
     * @return true, if is list
     */
    boolean isList();

    /**
     * Checks if is sets the.
     *
     * @return true, if is sets the
     */
    boolean isSet();

    /**
     * Checks if is collection.
     *
     * @return true, if is collection
     */
    boolean isCollection();

    /**
     * Checks if is map.
     *
     * @return true, if is map
     */
    boolean isMap();

    /**
     * Checks if is entity.
     *
     * @return true, if is entity
     */
    boolean isEntity();

    /**
     * Checks if is map entity.
     *
     * @return true, if is map entity
     */
    boolean isMapEntity();

    /**
     * Checks if is entity id.
     *
     * @return true, if is entity id
     */
    boolean isEntityId();

    /**
     * Checks if is data set.
     *
     * @return true, if is data set
     */
    boolean isDataSet();

    /**
     * Checks if is input stream.
     *
     * @return true, if is input stream
     */
    boolean isInputStream();

    /**
     * Checks if is reader.
     *
     * @return true, if is reader
     */
    boolean isReader();

    /**
     * Checks if is byte buffer.
     *
     * @return true, if is byte buffer
     */
    boolean isByteBuffer();

    /**
     * Checks if is generic type.
     *
     * @return true, if is generic type
     */
    boolean isGenericType();

    /**
     * Checks if is immutable.
     *
     * @return true, if is immutable
     */
    boolean isImmutable();

    /**
     * Checks if is comparable.
     *
     * @return true, if is comparable
     */
    boolean isComparable();

    /**
     * Returns {@code true} if the value of this type can be serialized to json/xml/... String directly. The primitive
     * type/array/wrapper, date, calendar ... belong to this category. Object Array/Collection/Map/Entity are not.
     *
     * @return true, if is serializable
     */
    boolean isSerializable();

    /**
     * Checks if is optional or nullable.
     *
     * @return true, if is optional or nullable
     */
    boolean isOptionalOrNullable();

    /**
     * Gets the serialization type.
     *
     * @return
     */
    SerializationType getSerializationType();

    /**
     * Gets the element type.
     *
     * @return
     */
    Type<?> getElementType();

    /**
     * Gets the parameter types.
     *
     * @return
     */
    Type<?>[] getParameterTypes();

    T defaultValue();

    /**
     *
     * @param value
     * @return
     */
    boolean isDefaultValue(T value);

    /**
     *
     * @param x
     * @param y
     * @return
     */
    int compare(T x, T y);

    /**
     *
     * @param x
     * @return
     */
    String stringOf(T x);

    /**
     *
     * @param str
     * @return
     */
    T valueOf(String str);

    /**
     *
     * @param obj
     * @return
     */
    T valueOf(Object obj);

    /**
     *
     * @param cbuf
     * @param offset
     * @param len
     * @return
     */
    T valueOf(char[] cbuf, int offset, int len);

    /**
     *
     * @param writer
     * @param x
     * @throws IOException Signals that an I/O exception has occurred.
     */
    void write(Writer writer, T x) throws IOException;

    /**
     *
     * @param writer
     * @param x
     * @param config
     * @throws IOException Signals that an I/O exception has occurred.
     */
    void writeCharacter(CharacterWriter writer, T x, SerializationConfig<?> config) throws IOException;

    /**
     * Collection 2 array.
     *
     * @param c
     * @return
     */
    T collection2Array(Collection<?> c);

    /**
     * Array 2 collection.
     *
     * @param <E>
     * @param collClass
     * @param x
     * @return
     */
    <E> Collection<E> array2Collection(Class<?> collClass, T x);

    /**
     * Array 2 collection.
     *
     * @param <E>
     * @param resultCollection
     * @param x
     * @return
     */
    <E> Collection<E> array2Collection(Collection<E> resultCollection, T x);

    /**
     *
     * @param x
     * @return
     */
    int hashCode(T x);

    /**
     * Deep hash code.
     *
     * @param x
     * @return
     */
    int deepHashCode(T x);

    /**
     *
     * @param x
     * @param y
     * @return true, if successful
     */
    boolean equals(T x, T y);

    /**
     *
     * @param x
     * @param y
     * @return true, if successful
     */
    boolean deepEquals(T x, T y);

    /**
     *
     * @param x
     * @return
     */
    String toString(T x);

    /**
     * Deep to string.
     *
     * @param x
     * @return
     */
    String deepToString(T x);

    /**
     * The Enum SerializationType.
     */
    enum SerializationType {

        /** The serializable. */
        SERIALIZABLE,
        /** The entity. */
        ENTITY,
        /** The map. */
        MAP,
        /** The array. */
        ARRAY,
        /** The collection. */
        COLLECTION,
        /** The map entity. */
        MAP_ENTITY,
        /** The entity id. */
        ENTITY_ID,
        /** The data set. */
        DATA_SET,
        /** The unknown. */
        UNKNOWN;
    }
}
