/*
 * Copyright (C) 2016 HaiYang Li
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package com.landawn.abacus.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 *
 * @author Haiyang Li
 * @param <L>
 * @param <R>
 * @since 0.8
 * @see N#difference(Collection, Collection)
 * @see N#differentSet(Collection, Collection)
 * @see N#symmetricDifference(Collection, Collection)
 * @see N#symmetricDifferentSet(Collection, Collection)
 * @see N#excludeAll(Collection, Collection)
 * @see N#excludeAllToSet(Collection, Collection)
 * @see N#removeAll(Collection, Collection)
 * @see N#intersection(Collection, Collection)
 * @see N#commonSet(Collection, Collection)
 */
public class Difference<L, R> {

    final L common;

    final L leftOnly;

    final R rightOnly;

    Difference(L common, L leftOnly, R rightOnly) {
        this.common = common;
        this.leftOnly = leftOnly;
        this.rightOnly = rightOnly;
    }

    /**
     *
     * @param <T1>
     * @param <T2>
     * @param <L>
     * @param <R>
     * @param a
     * @param b
     * @return
     * @see N#difference(Collection, Collection)
     * @see N#differentSet(Collection, Collection)
     * @see N#symmetricDifference(Collection, Collection)
     * @see N#symmetricDifferentSet(Collection, Collection)
     * @see N#excludeAll(Collection, Collection)
     * @see N#excludeAllToSet(Collection, Collection)
     * @see N#removeAll(Collection, Collection)
     * @see N#intersection(Collection, Collection)
     * @see N#commonSet(Collection, Collection)
     */
    public static <T1, T2, L extends List<T1>, R extends List<T2>> Difference<L, R> of(T1[] a, T2[] b) {
        return of(Arrays.asList(a), Arrays.asList(b));
    }

    /**
     *
     * @param <T1>
     * @param <T2>
     * @param <L>
     * @param <R>
     * @param a
     * @param b
     * @return
     * @see N#difference(Collection, Collection)
     * @see N#differentSet(Collection, Collection)
     * @see N#symmetricDifference(Collection, Collection)
     * @see N#symmetricDifferentSet(Collection, Collection)
     * @see N#excludeAll(Collection, Collection)
     * @see N#excludeAllToSet(Collection, Collection)
     * @see N#removeAll(Collection, Collection)
     * @see N#intersection(Collection, Collection)
     * @see N#commonSet(Collection, Collection)
     */
    public static <T1, T2, L extends List<T1>, R extends List<T2>> Difference<L, R> of(Collection<? extends T1> a, Collection<? extends T2> b) {
        List<T1> common = new ArrayList<>();
        List<T1> leftOnly = new ArrayList<>();
        List<T2> rightOnly = new ArrayList<>();

        if (N.isNullOrEmpty(a)) {
            if (N.isNullOrEmpty(b)) {
                // Do nothing. All empty.
            } else {
                rightOnly.addAll(b);
            }
        } else if (N.isNullOrEmpty(b)) {
            leftOnly.addAll(a);
        } else {
            final Multiset<T2> bOccurrences = Multiset.from(b);

            for (T1 e : a) {
                if (bOccurrences.getAndRemove(e) > 0) {
                    common.add(e);
                } else {
                    leftOnly.add(e);
                }
            }

            for (T2 e : b) {
                if (bOccurrences.getAndRemove(e) > 0) {
                    rightOnly.add(e);
                }

                if (bOccurrences.isEmpty()) {
                    break;
                }
            }
        }

        return new Difference<>((L) common, (L) leftOnly, (R) rightOnly);
    }

    /**
     *
     * @return
     */
    public L inCommon() {
        return common;
    }

    /**
     * On left only.
     *
     * @return
     */
    public L onLeftOnly() {
        return leftOnly;
    }

    /**
     * On right only.
     *
     * @return
     */
    public R onRightOnly() {
        return rightOnly;
    }

    /**
     *
     * @return
     */
    @SuppressWarnings("rawtypes")
    public boolean areEqual() {
        return (leftOnly instanceof Map && (((Map) leftOnly).isEmpty() && ((Map) rightOnly).isEmpty()))
                || (leftOnly instanceof Collection && (((Collection) leftOnly).isEmpty() && ((Collection) rightOnly).isEmpty()));
    }

    @Override
    public String toString() {
        return "{inCommon=" + common + ", onLeftOnly=" + leftOnly + ", onRightOnly=" + rightOnly + "}";
    }

    /**
     * The Class MapDifference.
     *
     * @param <L>
     * @param <R>
     * @param <D>
     * 
     * @see N#difference(Collection, Collection)
     * @see N#differentSet(Collection, Collection)
     * @see N#symmetricDifference(Collection, Collection)
     * @see N#symmetricDifferentSet(Collection, Collection)
     * @see N#excludeAll(Collection, Collection)
     * @see N#excludeAllToSet(Collection, Collection)
     * @see N#removeAll(Collection, Collection)
     * @see N#intersection(Collection, Collection)
     * @see N#commonSet(Collection, Collection)
     */
    public static final class MapDifference<L, R, D> extends Difference<L, R> {

        /** The diff values. */
        private final D diffValues;

        /**
         * Instantiates a new map difference.
         *
         * @param common
         * @param leftOnly
         * @param rightOnly
         * @param diff
         */
        MapDifference(L common, L leftOnly, R rightOnly, D diff) {
            super(common, leftOnly, rightOnly);
            this.diffValues = diff;
        }

        /**
         *
         * @param <K1>
         * @param <V1>
         * @param <K2>
         * @param <V2>
         * @param <L>
         * @param <R>
         * @param <D>
         * @param map1
         * @param map2
         * @return
         */
        @SuppressWarnings("unlikely-arg-type")
        public static <CK, K1 extends CK, V1, K2 extends CK, V2> MapDifference<Map<K1, V1>, Map<K2, V2>, Map<CK, Pair<V1, V2>>> of(
                final Map<? extends K1, ? extends V1> map1, final Map<? extends K2, ? extends V2> map2) {
            final Map<K1, V1> common = new LinkedHashMap<>();
            final Map<K1, V1> leftOnly = new LinkedHashMap<>();
            final Map<K2, V2> rightOnly = new LinkedHashMap<>();
            final Map<CK, Pair<V1, V2>> diff = new LinkedHashMap<>();

            if (N.isNullOrEmpty(map1)) {
                if (N.isNullOrEmpty(map2)) {
                    // Do nothing. All empty.
                } else {
                    rightOnly.putAll(map2);
                }
            } else if (N.isNullOrEmpty(map2)) {
                leftOnly.putAll(map1);
            } else {
                Object key1 = null;
                V2 val2 = null;
                for (Entry<K1, V1> entry1 : ((Map<K1, V1>) map1).entrySet()) {
                    key1 = entry1.getKey();
                    val2 = map2.get(key1);

                    if (val2 == null) {
                        if (map2.containsKey(key1)) {
                            if (entry1.getValue() == null) {
                                common.put(entry1.getKey(), entry1.getValue());
                            } else {
                                diff.put(entry1.getKey(), Pair.of(entry1.getValue(), val2));
                            }
                        } else {
                            leftOnly.put(entry1.getKey(), entry1.getValue());
                        }
                    } else if (N.equals(entry1.getValue(), val2)) {
                        common.put(entry1.getKey(), entry1.getValue());
                    } else {
                        diff.put(entry1.getKey(), Pair.of(entry1.getValue(), val2));
                    }
                }

                for (Entry<K2, V2> entry2 : ((Map<K2, V2>) map2).entrySet()) {
                    if (common.containsKey(entry2.getKey()) || diff.containsKey(entry2.getKey())) {
                        continue;
                    }

                    rightOnly.put(entry2.getKey(), entry2.getValue());
                }
            }

            return new MapDifference<>(common, leftOnly, rightOnly, diff);
        }

        /**
         *
         * @param entity1
         * @param entity2
         * @return
         */
        public static MapDifference<Map<String, Object>, Map<String, Object>, Map<String, Pair<Object, Object>>> of(Object entity1, Object entity2) {
            if (ClassUtil.isEntity(entity1.getClass()) == false || ClassUtil.isEntity(entity2.getClass()) == false) {
                throw new IllegalArgumentException(
                        entity1.getClass().getCanonicalName() + " or " + entity2.getClass().getCanonicalName() + " is not an entity class");
            }

            return of(Maps.entity2Map(entity1), Maps.entity2Map(entity2));
        }

        /**
         * Returns the entries that appear in both maps, but with different values.
         *
         * @return
         */
        public D withDifferentValues() {
            return diffValues;
        }

        /**
         *
         * @return
         */
        @Override
        @SuppressWarnings("rawtypes")
        public boolean areEqual() {
            return super.areEqual() && ((Map) diffValues).isEmpty();
        }

        /**
         *
         * @return
         */
        @Override
        public String toString() {
            return "{inCommon=" + common + ", onLeftOnly=" + leftOnly + ", onRightOnly=" + rightOnly + ", withDifferentValues=" + diffValues + "}";
        }
    }
}
