/*
 * Copyright (c) 2018, Haiyang Li.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.landawn.abacus.util;

import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.RandomAccess;

import com.landawn.abacus.util.u.Nullable;
import com.landawn.abacus.util.u.OptionalByte;
import com.landawn.abacus.util.u.OptionalChar;
import com.landawn.abacus.util.u.OptionalDouble;
import com.landawn.abacus.util.u.OptionalFloat;
import com.landawn.abacus.util.u.OptionalInt;
import com.landawn.abacus.util.u.OptionalLong;
import com.landawn.abacus.util.u.OptionalShort;
import com.landawn.abacus.util.function.Function;

/**
 * <p>
 * Note: This class includes codes copied from Apache Commons Lang, Google Guava and other open source projects under the Apache License 2.0.
 * The methods copied from other libraries/frameworks/projects may be modified in this class.
 * </p>
 *  
 * <p>
 * This is a utility class for iterable data structures, including {@code Collection/Array/Iterator}.
 * </p>
 * 
 * <p>
 * The methods in this class should only read the input {@code Collection/Array/Iterator} parameters, not modify them.
 * </p>
 * 
 * @see com.landawn.abacus.util.N
 * @see com.landawn.abacus.util.Iterators
 * @see com.landawn.abacus.util.Maps
 * @see com.landawn.abacus.util.StringUtil
 */
public final class Iterables {

    private Iterables() {
        // Utility class.
    }

    /**
     * Returns {@code OptionalChar.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalChar min(final char... a) {
        return a == null || a.length == 0 ? OptionalChar.empty() : OptionalChar.of(N.min(a));
    }

    /**
     * Returns {@code OptionalByte.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalByte min(final byte... a) {
        return a == null || a.length == 0 ? OptionalByte.empty() : OptionalByte.of(N.min(a));
    }

    /**
     * Returns {@code OptionalShort.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalShort min(final short... a) {
        return a == null || a.length == 0 ? OptionalShort.empty() : OptionalShort.of(N.min(a));
    }

    /**
     * Returns {@code OptionalInt.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalInt min(final int... a) {
        return a == null || a.length == 0 ? OptionalInt.empty() : OptionalInt.of(N.min(a));
    }

    /**
     * Returns {@code OptionalLong.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalLong min(final long... a) {
        return a == null || a.length == 0 ? OptionalLong.empty() : OptionalLong.of(N.min(a));
    }

    /**
     * Returns {@code OptionalFloat.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalFloat min(final float... a) {
        return a == null || a.length == 0 ? OptionalFloat.empty() : OptionalFloat.of(N.min(a));
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalDouble min(final double... a) {
        return a == null || a.length == 0 ? OptionalDouble.empty() : OptionalDouble.of(N.min(a));
    }

    /**
     * Returns {@code OptionalChar.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalChar max(final char... a) {
        return a == null || a.length == 0 ? OptionalChar.empty() : OptionalChar.of(N.max(a));
    }

    /**
     * Returns {@code OptionalByte.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalByte max(final byte... a) {
        return a == null || a.length == 0 ? OptionalByte.empty() : OptionalByte.of(N.max(a));
    }

    /**
     * Returns {@code OptionalShort.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalShort max(final short... a) {
        return a == null || a.length == 0 ? OptionalShort.empty() : OptionalShort.of(N.max(a));
    }

    /**
     * Returns {@code OptionalInt.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalInt max(final int... a) {
        return a == null || a.length == 0 ? OptionalInt.empty() : OptionalInt.of(N.max(a));
    }

    /**
     * Returns {@code OptionalLong.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalLong max(final long... a) {
        return a == null || a.length == 0 ? OptionalLong.empty() : OptionalLong.of(N.max(a));
    }

    /**
     * Returns {@code OptionalFloat.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalFloat max(final float... a) {
        return a == null || a.length == 0 ? OptionalFloat.empty() : OptionalFloat.of(N.max(a));
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    @SafeVarargs
    public static OptionalDouble max(final double... a) {
        return a == null || a.length == 0 ? OptionalDouble.empty() : OptionalDouble.of(N.max(a));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    public static <T extends Comparable<? super T>> Nullable<T> min(final T[] a) {
        return N.isNullOrEmpty(a) ? Nullable.<T> empty() : Nullable.of(N.min(a));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @param cmp
     * @return
     */
    public static <T> Nullable<T> min(final T[] a, final Comparator<? super T> cmp) {
        return N.isNullOrEmpty(a) ? Nullable.<T> empty() : Nullable.of(N.min(a, cmp));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param c
     * @return
     */
    public static <T extends Comparable<? super T>> Nullable<T> min(final Collection<? extends T> c) {
        return N.isNullOrEmpty(c) ? Nullable.<T> empty() : Nullable.of(N.min(c));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param c
     * @param cmp
     * @return
     */
    public static <T> Nullable<T> min(final Collection<? extends T> c, final Comparator<? super T> cmp) {
        return N.isNullOrEmpty(c) ? Nullable.<T> empty() : Nullable.of(N.min(c, cmp));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param iter
     * @return
     */
    public static <T extends Comparable<? super T>> Nullable<T> min(final Iterator<? extends T> iter) {
        return min(iter, N.NULL_MAX_COMPARATOR);
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param iter
     * @param cmp
     * @return
     */
    public static <T> Nullable<T> min(final Iterator<? extends T> iter, Comparator<? super T> cmp) {
        cmp = cmp == null ? N.NULL_MAX_COMPARATOR : cmp;

        if (iter == null || iter.hasNext() == false) {
            return Nullable.<T> empty();
        }

        T candidate = null;
        T next = null;

        do {
            next = iter.next();

            if (next == null && cmp == N.NULL_MIN_COMPARATOR) {
                return Nullable.of(next);
            } else if (cmp.compare(next, candidate) < 0) {
                candidate = next;
            }
        } while (iter.hasNext());

        return Nullable.of(candidate);
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @param cmp
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static <T> Nullable<T> minBy(final T[] a, final Function<? super T, ? extends Comparable> keyMapper) {
        return min(a, Fn.comparingBy(keyMapper));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param c
     * @param cmp
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static <T> Nullable<T> minBy(final Collection<? extends T> c, final Function<? super T, ? extends Comparable> keyMapper) {
        return min(c, Fn.comparingBy(keyMapper));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param iter
     * @param cmp
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static <T> Nullable<T> minBy(final Iterator<? extends T> iter, final Function<? super T, ? extends Comparable> keyMapper) {
        return min(iter, Fn.comparingBy(keyMapper));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    public static <T extends Comparable<? super T>> Nullable<T> max(final T[] a) {
        return N.isNullOrEmpty(a) ? Nullable.<T> empty() : Nullable.of(N.max(a));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @param cmp
     * @return
     */
    public static <T> Nullable<T> max(final T[] a, final Comparator<? super T> cmp) {
        return N.isNullOrEmpty(a) ? Nullable.<T> empty() : Nullable.of(N.max(a, cmp));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param c 
     * @return
     */
    public static <T extends Comparable<? super T>> Nullable<T> max(final Collection<? extends T> c) {
        return N.isNullOrEmpty(c) ? Nullable.<T> empty() : Nullable.of(N.max(c));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param c 
     * @return
     */
    public static <T> Nullable<T> max(final Collection<? extends T> c, final Comparator<? super T> cmp) {
        return N.isNullOrEmpty(c) ? Nullable.<T> empty() : Nullable.of(N.max(c, cmp));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param iter 
     * @return
     */
    public static <T extends Comparable<? super T>> Nullable<T> max(final Iterator<? extends T> iter) {
        return max(iter, N.NULL_MIN_COMPARATOR);
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param iter
     * @param cmp
     * @return
     */
    public static <T> Nullable<T> max(final Iterator<? extends T> iter, Comparator<? super T> cmp) {
        cmp = cmp == null ? N.NULL_MIN_COMPARATOR : cmp;

        if (iter == null || iter.hasNext() == false) {
            return Nullable.<T> empty();
        }

        T candidate = null;
        T next = null;

        do {
            next = iter.next();

            if (next == null && cmp == N.NULL_MAX_COMPARATOR) {
                return Nullable.of(next);
            } else if (cmp.compare(next, candidate) > 0) {
                candidate = next;
            }
        } while (iter.hasNext());

        return Nullable.of(candidate);
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @param cmp
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static <T> Nullable<T> maxBy(final T[] a, final Function<? super T, ? extends Comparable> keyMapper) {
        return max(a, Fn.comparingBy(keyMapper));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param c
     * @param cmp
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static <T> Nullable<T> maxBy(final Collection<? extends T> c, final Function<? super T, ? extends Comparable> keyMapper) {
        return max(c, Fn.comparingBy(keyMapper));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param iter
     * @param cmp
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static <T> Nullable<T> maxBy(final Iterator<? extends T> iter, final Function<? super T, ? extends Comparable> keyMapper) {
        return max(iter, Fn.comparingBy(keyMapper));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @return
     */
    public static <T extends Comparable<? super T>> Nullable<T> median(final T[] a) {
        return N.isNullOrEmpty(a) ? Nullable.<T> empty() : Nullable.of(N.median(a));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param c 
     * @return
     */
    public static <T extends Comparable<? super T>> Nullable<T> median(final Collection<? extends T> c) {
        return N.isNullOrEmpty(c) ? Nullable.<T> empty() : Nullable.of(N.median(c));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @param cmp
     * @return
     */
    public static <T> Nullable<T> median(final T[] a, final Comparator<? super T> cmp) {
        return N.isNullOrEmpty(a) ? Nullable.<T> empty() : Nullable.of(N.median(a, cmp));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param c
     * @param cmp
     * @return
     */
    public static <T> Nullable<T> median(final Collection<? extends T> c, final Comparator<? super T> cmp) {
        return N.isNullOrEmpty(c) ? Nullable.<T> empty() : Nullable.of(N.median(c, cmp));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param a
     * @param keyMapper
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static <T> Nullable<T> medianBy(final T[] a, final Function<? super T, ? extends Comparable> keyMapper) {
        return median(a, Fn.comparingBy(keyMapper));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty.
     * 
     * @param c
     * @param keyMapper
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static <T> Nullable<T> medianBy(final Collection<? extends T> c, final Function<? super T, ? extends Comparable> keyMapper) {
        return median(c, Fn.comparingBy(keyMapper));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or its length/size is less than {@code k}.
     * 
     * @param c
     * @param k 
     * @return
     */
    public static <T extends Comparable<? super T>> Nullable<T> kthLargest(final Collection<? extends T> c, final int k) {
        return N.isNullOrEmpty(c) || c.size() < k ? Nullable.<T> empty() : Nullable.of(N.kthLargest(c, k));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or its length/size is less than {@code k}.
     * 
     * @param a
     * @param k 
     * @return
     */
    public static <T extends Comparable<? super T>> Nullable<T> kthLargest(final T[] a, final int k) {
        return N.isNullOrEmpty(a) || a.length < k ? Nullable.<T> empty() : Nullable.of(N.kthLargest(a, k));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or its length/size is less than {@code k}.
     * 
     * @param c
     * @param k
     * @param cmp
     * @return
     */
    public static <T> Nullable<T> kthLargest(final Collection<? extends T> c, final int k, final Comparator<? super T> cmp) {
        return N.isNullOrEmpty(c) || c.size() < k ? Nullable.<T> empty() : Nullable.of(N.kthLargest(c, k, cmp));
    }

    /**
     * Returns {@code Nullable.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or its length/size is less than {@code k}.
     * 
     * @param a
     * @param k
     * @param cmp
     * @return
     */
    public static <T> Nullable<T> kthLargest(final T[] a, final int k, final Comparator<? super T> cmp) {
        return N.isNullOrEmpty(a) || a.length < k ? Nullable.<T> empty() : Nullable.of(N.kthLargest(a, k, cmp));
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param a
     * @return
     */
    public static <T extends Number> OptionalDouble averageInt(final T[] a) {
        return averageInt(a, Fn.numToInt());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param a
     * @param fromIndex
     * @param toIndex
     * @return
     */
    public static <T extends Number> OptionalDouble averageInt(final T[] a, final int fromIndex, final int toIndex) {
        return averageInt(a, fromIndex, toIndex, Fn.numToInt());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param a
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageInt(final T[] a, final Throwables.ToIntFunction<? super T, E> func) throws E {
        if (N.isNullOrEmpty(a)) {
            return OptionalDouble.empty();
        }

        return averageInt(a, 0, a.length, func);
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param a
     * @param fromIndex
     * @param toIndex
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageInt(final T[] a, final int fromIndex, final int toIndex,
            final Throwables.ToIntFunction<? super T, E> func) throws E {
        N.checkFromToIndex(fromIndex, toIndex, N.len(a));

        if (fromIndex == toIndex) {
            return OptionalDouble.empty();
        }

        return OptionalDouble.of(((double) N.sumInt(a, fromIndex, toIndex, func)) / (toIndex - fromIndex));
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param c
     * @return
     */
    public static <T extends Number> OptionalDouble averageInt(final Collection<? extends T> c) {
        return averageInt(c, Fn.numToInt());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param c
     * @param fromIndex
     * @param toIndex
     * @return
     */
    public static <T extends Number> OptionalDouble averageInt(final Collection<? extends T> c, final int fromIndex, final int toIndex) {
        return averageInt(c, fromIndex, toIndex, Fn.numToInt());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param c
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageInt(final Collection<? extends T> c, final Throwables.ToIntFunction<? super T, E> func)
            throws E {
        if (N.isNullOrEmpty(c)) {
            return OptionalDouble.empty();
        }

        return OptionalDouble.of(((double) N.sumInt(c, func)) / c.size());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param c
     * @param fromIndex
     * @param toIndex
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageInt(final Collection<? extends T> c, final int fromIndex, final int toIndex,
            final Throwables.ToIntFunction<? super T, E> func) throws E {
        N.checkFromToIndex(fromIndex, toIndex, N.size(c));

        if (fromIndex == toIndex) {
            return OptionalDouble.empty();
        }

        return OptionalDouble.of(((double) N.sumInt(c, fromIndex, toIndex, func)) / (toIndex - fromIndex));
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param a
     * @return
     */
    public static <T extends Number> OptionalDouble averageLong(final T[] a) {
        return averageLong(a, Fn.numToLong());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param a
     * @param fromIndex
     * @param toIndex
     * @return
     */
    public static <T extends Number> OptionalDouble averageLong(final T[] a, final int fromIndex, final int toIndex) {
        return averageLong(a, fromIndex, toIndex, Fn.numToLong());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param a
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageLong(final T[] a, final Throwables.ToLongFunction<? super T, E> func) throws E {
        if (N.isNullOrEmpty(a)) {
            return OptionalDouble.empty();
        }

        return averageLong(a, 0, a.length, func);
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param a
     * @param fromIndex
     * @param toIndex
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageLong(final T[] a, final int fromIndex, final int toIndex,
            final Throwables.ToLongFunction<? super T, E> func) throws E {
        N.checkFromToIndex(fromIndex, toIndex, N.len(a));

        if (fromIndex == toIndex) {
            return OptionalDouble.empty();
        }

        return OptionalDouble.of(((double) N.sumLong(a, fromIndex, toIndex, func)) / (toIndex - fromIndex));
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param c
     * @return
     */
    public static <T extends Number> OptionalDouble averageLong(final Collection<? extends T> c) {
        return averageLong(c, Fn.numToLong());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param c
     * @param fromIndex
     * @param toIndex
     * @return
     */
    public static <T extends Number> OptionalDouble averageLong(final Collection<? extends T> c, final int fromIndex, final int toIndex) {
        return averageLong(c, fromIndex, toIndex, Fn.numToLong());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param c
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageLong(final Collection<? extends T> c, final Throwables.ToLongFunction<? super T, E> func)
            throws E {
        if (N.isNullOrEmpty(c)) {
            return OptionalDouble.empty();
        }

        return OptionalDouble.of(((double) N.sumLong(c, func)) / c.size());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param c
     * @param fromIndex
     * @param toIndex
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageLong(final Collection<? extends T> c, final int fromIndex, final int toIndex,
            final Throwables.ToLongFunction<? super T, E> func) throws E {
        N.checkFromToIndex(fromIndex, toIndex, N.size(c));

        if (fromIndex == toIndex) {
            return OptionalDouble.empty();
        }

        return OptionalDouble.of(((double) N.sumLong(c, fromIndex, toIndex, func)) / (toIndex - fromIndex));
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param a
     * @return
     */
    public static <T extends Number> OptionalDouble averageDouble(final T[] a) {
        return averageDouble(a, Fn.numToDouble());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param a
     * @param fromIndex
     * @param toIndex
     * @return
     */
    public static <T extends Number> OptionalDouble averageDouble(final T[] a, final int fromIndex, final int toIndex) {
        return averageDouble(a, fromIndex, toIndex, Fn.numToDouble());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param a
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageDouble(final T[] a, final Throwables.ToDoubleFunction<? super T, E> func) throws E {
        if (N.isNullOrEmpty(a)) {
            return OptionalDouble.empty();
        }

        return averageDouble(a, 0, a.length, func);
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param a
     * @param fromIndex
     * @param toIndex
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageDouble(final T[] a, final int fromIndex, final int toIndex,
            final Throwables.ToDoubleFunction<? super T, E> func) throws E {
        N.checkFromToIndex(fromIndex, toIndex, N.len(a));

        if (fromIndex == toIndex) {
            return OptionalDouble.empty();
        }

        final KahanSummation summation = new KahanSummation();

        for (int i = fromIndex; i < toIndex; i++) {
            summation.add(func.applyAsDouble(a[i]));
        }

        return summation.average();
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param c
     * @return
     */
    public static <T extends Number> OptionalDouble averageDouble(final Collection<? extends T> c) {
        return averageDouble(c, Fn.numToDouble());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param c
     * @param fromIndex
     * @param toIndex
     * @return
     */
    public static <T extends Number> OptionalDouble averageDouble(final Collection<? extends T> c, final int fromIndex, final int toIndex) {
        return averageDouble(c, fromIndex, toIndex, Fn.numToDouble());
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param c
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageDouble(final Collection<? extends T> c, final Throwables.ToDoubleFunction<? super T, E> func)
            throws E {
        if (N.isNullOrEmpty(c)) {
            return OptionalDouble.empty();
        }

        final KahanSummation summation = new KahanSummation();

        for (T e : c) {
            summation.add(func.applyAsDouble(e));
        }

        return summation.average();
    }

    /**
     * Returns {@code OptionalDouble.empty()} if the specified {@code Array/Collection} is {@code null} or empty, or {@code fromIndex == toIndex}.
     *
     * @param <T>
     * @param <E>
     * @param c
     * @param fromIndex
     * @param toIndex
     * @param func
     * @return
     * @throws E the e
     */
    public static <T, E extends Exception> OptionalDouble averageDouble(final Collection<? extends T> c, final int fromIndex, final int toIndex,
            final Throwables.ToDoubleFunction<? super T, E> func) throws E {
        N.checkFromToIndex(fromIndex, toIndex, N.size(c));

        if (fromIndex == toIndex) {
            return OptionalDouble.empty();
        }

        final KahanSummation summation = new KahanSummation();

        if (c instanceof List && c instanceof RandomAccess) {
            final List<T> list = (List<T>) c;

            for (int i = fromIndex; i < toIndex; i++) {
                summation.add(func.applyAsDouble(list.get(i)));
            }
        } else {
            int idx = 0;

            for (T e : c) {
                if (idx++ < fromIndex) {
                    continue;
                }

                summation.add(func.applyAsDouble(e));

                if (idx >= toIndex) {
                    break;
                }
            }
        }

        return summation.average();
    }

    public static OptionalInt indexOf(final Object[] a, final Object objToFind) {
        return Index.of(a, objToFind);
    }

    public static OptionalInt indexOf(final Collection<?> c, final Object objToFind) {
        return Index.of(c, objToFind);
    }

    public static OptionalInt lastIndexOf(final Object[] a, final Object objToFind) {
        return Index.last(a, objToFind);
    }

    public static OptionalInt lastIndexOf(final Collection<?> c, final Object objToFind) {
        return Index.last(c, objToFind);
    }

    @SuppressWarnings("rawtypes")
    public static <T> boolean padLeft(final List<T> list, final int minLen, final T objToAdd) {
        N.checkArgNotNegative(minLen, "minLen");

        final int size = N.size(list);

        if (size < minLen) {
            final int elementCountToAdd = minLen - size;
            final Object[] a = new Object[elementCountToAdd];

            if (objToAdd != null) {
                N.fill(a, objToAdd);

                list.addAll(0, (List) Arrays.asList(a));
            }

            return true;
        }

        return false;
    }

    @SuppressWarnings("rawtypes")
    public static <T> boolean padRight(final Collection<T> c, final int minLen, final T objToAdd) {
        N.checkArgNotNegative(minLen, "minLen");

        final int size = N.size(c);

        if (size < minLen) {
            final int elementCountToAdd = minLen - size;
            final Object[] a = new Object[elementCountToAdd];

            if (objToAdd != null) {
                N.fill(a, objToAdd);

                c.addAll((Collection) Arrays.asList(a));
            }

            return true;
        }

        return false;
    }

    /**
     * The Class Slice.
     *
     * @param <T> the generic type
     */
    static final class Slice<T> extends ImmutableCollection<T> {

        /** The from index. */
        private final int fromIndex;

        /** The to index. */
        private final int toIndex;

        /**
         * Instantiates a new sub collection.
         *
         * @param a the a
         * @param fromIndex the from index
         * @param toIndex the to index
         */
        Slice(final T[] a, final int fromIndex, final int toIndex) {
            this(Array.asList(a), fromIndex, toIndex);
        }

        /**
         * Instantiates a new sub collection.
         *
         * @param c the c
         * @param fromIndex the from index
         * @param toIndex the to index
         */
        Slice(final Collection<? extends T> c, final int fromIndex, final int toIndex) {
            super(c);
            this.fromIndex = fromIndex;
            this.toIndex = toIndex;
        }

        /**
         * Contains.
         *
         * @param o the o
         * @return
         */
        @Override
        public boolean contains(Object o) {
            final Iterator<T> iter = this.iterator();

            while (iter.hasNext()) {
                if (N.equals(iter.next(), o)) {
                    return true;
                }
            }

            return false;
        }

        /**
         * Contains all.
         *
         * @param c the c
         * @return
         */
        @Override
        public boolean containsAll(final Collection<?> c) {
            for (Object e : c) {
                if (contains(e) == false) {
                    return false;
                }
            }

            return true;
        }

        /**
         * Checks if is empty.
         *
         * @return true, if is empty
         */
        @Override
        public boolean isEmpty() {
            return size() == 0;
        }

        /**
         * Size.
         *
         * @return the int
         */
        @Override
        public int size() {
            return toIndex - fromIndex;
        }

        /**
         * Iterator.
         *
         * @return the iterator
         */
        @Override
        public ObjIterator<T> iterator() {
            if (coll == null || fromIndex == toIndex) {
                return ObjIterator.<T> empty();
            }

            final Iterator<T> iter = coll.iterator();

            if (fromIndex == 0 && toIndex == coll.size()) {
                return ObjIterator.of(iter);
            } else if (fromIndex == 0) {
                return Iterators.limit(iter, toIndex - fromIndex);
            } else if (toIndex == coll.size()) {
                return Iterators.skip(iter, fromIndex);
            } else {
                return Iterators.skipAndLimit(iter, fromIndex, toIndex - fromIndex);
            }
        }

        /**
         * To array.
         *
         * @return the object[]
         */
        @Override
        public Object[] toArray() {
            final Iterator<T> iter = this.iterator();
            final Object[] a = new Object[size()];

            for (int i = 0, len = a.length; i < len; i++) {
                a[i] = iter.next();
            }

            return a;
        }

        /**
         * To array.
         *
         * @param <A> the generic type
         * @param a the a
         * @return the a[]
         */
        @Override
        public <A> A[] toArray(A[] a) {
            if (a.length < size()) {
                a = N.copyOf(a, size());
            }

            final Iterator<T> iter = this.iterator();

            for (int i = 0, len = a.length; i < len; i++) {
                a[i] = (A) iter.next();
            }

            return a;
        }
    }
}
