// Code generated by lark suite oapi sdk gen
/*
 * MIT License
 *
 * Copyright (c) 2022 Lark Technologies Pte. Ltd.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice, shall be included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.lark.oapi.event;

import com.lark.oapi.core.Constants;
import com.lark.oapi.core.IHandler;
import com.lark.oapi.core.exception.DecryptException;
import com.lark.oapi.core.exception.EventTypeAlreadyHasHandlerException;
import com.lark.oapi.core.exception.IncorrectChallengeException;
import com.lark.oapi.core.exception.IncorrectSignatureException;
import com.lark.oapi.core.request.EventReq;
import com.lark.oapi.core.response.EventResp;
import com.lark.oapi.core.utils.Decryptor;
import com.lark.oapi.core.utils.Jsons;
import com.lark.oapi.core.utils.Strings;
import com.lark.oapi.event.exception.HandlerNotFoundException;
import com.lark.oapi.event.model.BaseEvent;
import com.lark.oapi.event.model.BaseEventV2;
import com.lark.oapi.event.model.Fuzzy;
import com.lark.oapi.service.acs.v1.AcsService;
import com.lark.oapi.service.application.v6.ApplicationService;
import com.lark.oapi.service.approval.v4.ApprovalService;
import com.lark.oapi.service.calendar.v4.CalendarService;
import com.lark.oapi.service.contact.v3.ContactService;
import com.lark.oapi.service.drive.v1.DriveService;
import com.lark.oapi.service.helpdesk.v1.HelpdeskService;
import com.lark.oapi.service.im.v1.ImService;
import com.lark.oapi.service.meeting_room.v1.MeetingRoomService;
import com.lark.oapi.service.task.v1.TaskService;
import com.lark.oapi.service.vc.v1.VcService;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.codec.binary.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EventDispatcher implements IHandler {

  private static final Logger log = LoggerFactory.getLogger(EventDispatcher.class);
  private Map<String, IEventHandler> eventType2EventHandler = new HashMap<>();
  private String verificationToken;
  private String encryptKey;

  public EventDispatcher(Builder builder) {
    this.verificationToken = builder.verificationToken;
    this.encryptKey = builder.encryptKey;
    this.eventType2EventHandler = builder.eventType2EventHandler;
    eventType2EventHandler.put("app_ticket", new AppTicketEventHandler());
  }

  public static Builder newBuilder(String verificationToken, String encryptKey) {
    return new Builder(verificationToken, encryptKey);
  }

  public String getVerificationToken() {
    return verificationToken;
  }

  public String getEncryptKey() {
    return encryptKey;
  }

  private String parseReq(EventReq eventReq) throws UnsupportedEncodingException {
    log.info("event req,header:{},body:{}", Jsons.LONG_TO_STR.toJson(eventReq.getHeaders()),
        new String(eventReq.getBody(), StandardCharsets.UTF_8));
    if (!Strings.isEmpty(encryptKey)) {
      Fuzzy fuzzy = Jsons.DEFAULT.fromJson(new String(eventReq.getBody(), StandardCharsets.UTF_8),
          Fuzzy.class);
      if (fuzzy == null || Strings.isEmpty(fuzzy.getEncrypt())) {
        throw new DecryptException("The result of event decryption failed");
      }
      return fuzzy.getEncrypt().trim();
    }
    return new String(eventReq.getBody(), StandardCharsets.UTF_8).trim();
  }

  private String decryptEvent(String cipherEventJsonStr) {
    if (!Strings.isEmpty(encryptKey)) {
      // 非线程安全，所以每次要new
      String plainEventJsonStr = new Decryptor(encryptKey).decrypt(cipherEventJsonStr);
      log.debug("plain Event: {}", plainEventJsonStr);
      return plainEventJsonStr.trim();
    }
    return cipherEventJsonStr;
  }

  private boolean verifySign(EventReq eventReq) throws NoSuchAlgorithmException {
    if (Strings.isEmpty(encryptKey)) {
      return true;
    }

    String cipherEventJsonStr = new String(eventReq.getBody(), StandardCharsets.UTF_8);
    String timestamp, nonce, sourceSign, targetSign;
    timestamp = eventReq.getHeaderFirstValue(Constants.X_LARK_REQUEST_TIMESTAMP);
    nonce = eventReq.getHeaderFirstValue(Constants.X_LARK_REQUEST_NONCE);
    sourceSign = eventReq.getHeaderFirstValue(Constants.X_LARK_SIGNATURE);
    targetSign = calculateSignature(timestamp, nonce, encryptKey, cipherEventJsonStr);
    return targetSign.equals(sourceSign);
  }

  protected String calculateSignature(String timestamp, String nonce, String encryptKey,
      String bodyString) throws NoSuchAlgorithmException {
    StringBuilder content = new StringBuilder();
    content.append(timestamp).append(nonce).append(encryptKey).append(bodyString);
    MessageDigest alg = MessageDigest.getInstance("SHA-256");
    String sign = Hex.encodeHexString(alg.digest(content.toString().getBytes()));
    return sign;
  }

  private EventResp doHandle(String plainEventJsonStr, String eventType, String reqType,
      String challenge, String token, EventReq req) throws Exception {
    EventResp resp = new EventResp();
    resp.setStatusCode(200);
    resp.setContentType(Constants.APPLICATION_JSON);

    // 使用challenge进行鉴权
    if (Constants.URL_VERIFICATION.equals(reqType)) {
      if (!verificationToken.equals(token)) {
        throw new IncorrectChallengeException();
      }

      resp.setBody(String.format(EventResp.CHALLENGE_RESPONSE_FORMAT,
          challenge).getBytes(StandardCharsets.UTF_8));
      return resp;
    }

    // 查找处理器，进行处理
    IEventHandler handler = eventType2EventHandler.get(eventType);
    if (handler == null) {
      throw new HandlerNotFoundException(eventType);
    }

    // 装配参数
    Object eventMsg = handler.getEvent();
    if (handler instanceof CustomEventHandler) {
      eventMsg = req;
    } else {
      eventMsg = Jsons.DEFAULT.fromJson(plainEventJsonStr, eventMsg.getClass());
    }

    if (eventMsg instanceof BaseEventV2) {
      ((BaseEventV2) eventMsg).setEventReq(req);
    } else if (eventMsg instanceof BaseEvent) {
      ((BaseEvent) eventMsg).setEventReq(req);
    }

    // 执行处理器
    handler.handle(eventMsg);
    resp.setBody(
        String.format(EventResp.RESPONSE_FORMAT, "success").getBytes(StandardCharsets.UTF_8));
    return resp;
  }

  public EventResp handle(EventReq eventReq) throws Throwable {
    EventResp eventResp = new EventResp();
    eventResp.setStatusCode(200);
    eventResp.setContentType(Constants.CONTENT_TYPE);
    try {
      // 解析请求，如果需要的话
      String cipherEventJsonStr = parseReq(eventReq);

      // 解密请求，如果需要的话
      String plainEventJsonStr = decryptEvent(cipherEventJsonStr);

      // 解析关键字段
      Fuzzy fuzzy = Jsons.DEFAULT.fromJson(plainEventJsonStr, Fuzzy.class);
      if (Strings.isNotEmpty(fuzzy.getEncrypt())) {
        throw new IllegalArgumentException("process encrypted msg event, need config encryptKey");
      }
      String token = fuzzy.getToken();
      String eventType = "";
      if (fuzzy.getEvent() != null) {
        eventType = fuzzy.getEvent().getType();
      }
      if (fuzzy.getHeader() != null) {
        token = fuzzy.getHeader().getToken();
        eventType = fuzzy.getHeader().getEventType();
      }
      String challenge = fuzzy.getChallenge();
      String reqType = fuzzy.getType();

      // 验签逻辑
      if (!Constants.URL_VERIFICATION.equals(reqType)) {
        if (!verifySign(eventReq)) {
          throw new IncorrectSignatureException();
        }
      }

      // 处理逻辑
      return doHandle(plainEventJsonStr, eventType, reqType, challenge, token, eventReq);
    } catch (Throwable e) {
      log.error("handle event failed,httpPath:{},requestId:{},err:"
          , eventReq.getHttpPath(), eventReq.getRequestID(), e);
      if (e instanceof HandlerNotFoundException) {
        eventResp.setBody(String.format(EventResp.RESPONSE_FORMAT,
            e.getMessage()).getBytes(StandardCharsets.UTF_8));
        return eventResp;
      }

      eventResp.setStatusCode(500);
      eventResp.setBody(String.format(EventResp.RESPONSE_FORMAT,
          e.getMessage()).getBytes(StandardCharsets.UTF_8));
    }

    return eventResp;
  }

  public static class Builder {

    private Map<String, IEventHandler> eventType2EventHandler = new HashMap<>();
    private String verificationToken;
    private String encryptKey;

    public Builder(String verificationToken, String encryptKey) {
      this.verificationToken = verificationToken;
      this.encryptKey = encryptKey;
    }

    public EventDispatcher build() {
      return new EventDispatcher(this);
    }

    public Builder onP2AccessRecordCreatedV1(AcsService.P2AccessRecordCreatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("acs.access_record.created_v1")) {
        throw new EventTypeAlreadyHasHandlerException("acs.access_record.created_v1");
      }
      eventType2EventHandler.put("acs.access_record.created_v1", handler);
      return this;
    }

    public Builder onP2UserUpdatedV1(AcsService.P2UserUpdatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("acs.user.updated_v1")) {
        throw new EventTypeAlreadyHasHandlerException("acs.user.updated_v1");
      }
      eventType2EventHandler.put("acs.user.updated_v1", handler);
      return this;
    }


    public Builder onP2ApplicationCreatedV6(
        ApplicationService.P2ApplicationCreatedV6Handler handler) {
      if (eventType2EventHandler.containsKey("application.application.created_v6")) {
        throw new EventTypeAlreadyHasHandlerException("application.application.created_v6");
      }
      eventType2EventHandler.put("application.application.created_v6", handler);
      return this;
    }

    public Builder onP2ApplicationAppVersionAuditV6(
        ApplicationService.P2ApplicationAppVersionAuditV6Handler handler) {
      if (eventType2EventHandler.containsKey("application.application.app_version.audit_v6")) {
        throw new EventTypeAlreadyHasHandlerException(
            "application.application.app_version.audit_v6");
      }
      eventType2EventHandler.put("application.application.app_version.audit_v6", handler);
      return this;
    }

    public Builder onP2ApplicationAppVersionPublishApplyV6(
        ApplicationService.P2ApplicationAppVersionPublishApplyV6Handler handler) {
      if (eventType2EventHandler.containsKey(
          "application.application.app_version.publish_apply_v6")) {
        throw new EventTypeAlreadyHasHandlerException(
            "application.application.app_version.publish_apply_v6");
      }
      eventType2EventHandler.put("application.application.app_version.publish_apply_v6", handler);
      return this;
    }

    public Builder onP2ApplicationAppVersionPublishRevokeV6(
        ApplicationService.P2ApplicationAppVersionPublishRevokeV6Handler handler) {
      if (eventType2EventHandler.containsKey(
          "application.application.app_version.publish_revoke_v6")) {
        throw new EventTypeAlreadyHasHandlerException(
            "application.application.app_version.publish_revoke_v6");
      }
      eventType2EventHandler.put("application.application.app_version.publish_revoke_v6", handler);
      return this;
    }

    public Builder onP2ApplicationFeedbackCreatedV6(
        ApplicationService.P2ApplicationFeedbackCreatedV6Handler handler) {
      if (eventType2EventHandler.containsKey("application.application.feedback.created_v6")) {
        throw new EventTypeAlreadyHasHandlerException(
            "application.application.feedback.created_v6");
      }
      eventType2EventHandler.put("application.application.feedback.created_v6", handler);
      return this;
    }

    public Builder onP2ApplicationFeedbackUpdatedV6(
        ApplicationService.P2ApplicationFeedbackUpdatedV6Handler handler) {
      if (eventType2EventHandler.containsKey("application.application.feedback.updated_v6")) {
        throw new EventTypeAlreadyHasHandlerException(
            "application.application.feedback.updated_v6");
      }
      eventType2EventHandler.put("application.application.feedback.updated_v6", handler);
      return this;
    }

    public Builder onP2ApplicationVisibilityAddedV6(
        ApplicationService.P2ApplicationVisibilityAddedV6Handler handler) {
      if (eventType2EventHandler.containsKey("application.application.visibility.added_v6")) {
        throw new EventTypeAlreadyHasHandlerException(
            "application.application.visibility.added_v6");
      }
      eventType2EventHandler.put("application.application.visibility.added_v6", handler);
      return this;
    }

    public Builder onP2ApprovalUpdatedV4(ApprovalService.P2ApprovalUpdatedV4Handler handler) {
      if (eventType2EventHandler.containsKey("approval.approval.updated_v4")) {
        throw new EventTypeAlreadyHasHandlerException("approval.approval.updated_v4");
      }
      eventType2EventHandler.put("approval.approval.updated_v4", handler);
      return this;
    }


    public Builder onP2CalendarChangedV4(CalendarService.P2CalendarChangedV4Handler handler) {
      if (eventType2EventHandler.containsKey("calendar.calendar.changed_v4")) {
        throw new EventTypeAlreadyHasHandlerException("calendar.calendar.changed_v4");
      }
      eventType2EventHandler.put("calendar.calendar.changed_v4", handler);
      return this;
    }

    public Builder onP2CalendarAclCreatedV4(CalendarService.P2CalendarAclCreatedV4Handler handler) {
      if (eventType2EventHandler.containsKey("calendar.calendar.acl.created_v4")) {
        throw new EventTypeAlreadyHasHandlerException("calendar.calendar.acl.created_v4");
      }
      eventType2EventHandler.put("calendar.calendar.acl.created_v4", handler);
      return this;
    }

    public Builder onP2CalendarAclDeletedV4(CalendarService.P2CalendarAclDeletedV4Handler handler) {
      if (eventType2EventHandler.containsKey("calendar.calendar.acl.deleted_v4")) {
        throw new EventTypeAlreadyHasHandlerException("calendar.calendar.acl.deleted_v4");
      }
      eventType2EventHandler.put("calendar.calendar.acl.deleted_v4", handler);
      return this;
    }

    public Builder onP2CalendarEventChangedV4(
        CalendarService.P2CalendarEventChangedV4Handler handler) {
      if (eventType2EventHandler.containsKey("calendar.calendar.event.changed_v4")) {
        throw new EventTypeAlreadyHasHandlerException("calendar.calendar.event.changed_v4");
      }
      eventType2EventHandler.put("calendar.calendar.event.changed_v4", handler);
      return this;
    }

    public Builder onP2CustomAttrEventUpdatedV3(
        ContactService.P2CustomAttrEventUpdatedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.custom_attr_event.updated_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.custom_attr_event.updated_v3");
      }
      eventType2EventHandler.put("contact.custom_attr_event.updated_v3", handler);
      return this;
    }

    public Builder onP2DepartmentCreatedV3(ContactService.P2DepartmentCreatedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.department.created_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.department.created_v3");
      }
      eventType2EventHandler.put("contact.department.created_v3", handler);
      return this;
    }

    public Builder onP2DepartmentDeletedV3(ContactService.P2DepartmentDeletedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.department.deleted_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.department.deleted_v3");
      }
      eventType2EventHandler.put("contact.department.deleted_v3", handler);
      return this;
    }

    public Builder onP2DepartmentUpdatedV3(ContactService.P2DepartmentUpdatedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.department.updated_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.department.updated_v3");
      }
      eventType2EventHandler.put("contact.department.updated_v3", handler);
      return this;
    }

    public Builder onP2EmployeeTypeEnumActivedV3(
        ContactService.P2EmployeeTypeEnumActivedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.employee_type_enum.actived_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.employee_type_enum.actived_v3");
      }
      eventType2EventHandler.put("contact.employee_type_enum.actived_v3", handler);
      return this;
    }

    public Builder onP2EmployeeTypeEnumCreatedV3(
        ContactService.P2EmployeeTypeEnumCreatedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.employee_type_enum.created_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.employee_type_enum.created_v3");
      }
      eventType2EventHandler.put("contact.employee_type_enum.created_v3", handler);
      return this;
    }

    public Builder onP2EmployeeTypeEnumDeactivatedV3(
        ContactService.P2EmployeeTypeEnumDeactivatedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.employee_type_enum.deactivated_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.employee_type_enum.deactivated_v3");
      }
      eventType2EventHandler.put("contact.employee_type_enum.deactivated_v3", handler);
      return this;
    }

    public Builder onP2EmployeeTypeEnumDeletedV3(
        ContactService.P2EmployeeTypeEnumDeletedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.employee_type_enum.deleted_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.employee_type_enum.deleted_v3");
      }
      eventType2EventHandler.put("contact.employee_type_enum.deleted_v3", handler);
      return this;
    }

    public Builder onP2EmployeeTypeEnumUpdatedV3(
        ContactService.P2EmployeeTypeEnumUpdatedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.employee_type_enum.updated_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.employee_type_enum.updated_v3");
      }
      eventType2EventHandler.put("contact.employee_type_enum.updated_v3", handler);
      return this;
    }

    public Builder onP2ScopeUpdatedV3(ContactService.P2ScopeUpdatedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.scope.updated_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.scope.updated_v3");
      }
      eventType2EventHandler.put("contact.scope.updated_v3", handler);
      return this;
    }

    public Builder onP2UserCreatedV3(ContactService.P2UserCreatedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.user.created_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.user.created_v3");
      }
      eventType2EventHandler.put("contact.user.created_v3", handler);
      return this;
    }

    public Builder onP2UserDeletedV3(ContactService.P2UserDeletedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.user.deleted_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.user.deleted_v3");
      }
      eventType2EventHandler.put("contact.user.deleted_v3", handler);
      return this;
    }

    public Builder onP2UserUpdatedV3(ContactService.P2UserUpdatedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact.user.updated_v3")) {
        throw new EventTypeAlreadyHasHandlerException("contact.user.updated_v3");
      }
      eventType2EventHandler.put("contact.user.updated_v3", handler);
      return this;
    }


    public Builder onP2FileDeletedV1(DriveService.P2FileDeletedV1Handler handler) {
      if (eventType2EventHandler.containsKey("drive.file.deleted_v1")) {
        throw new EventTypeAlreadyHasHandlerException("drive.file.deleted_v1");
      }
      eventType2EventHandler.put("drive.file.deleted_v1", handler);
      return this;
    }

    public Builder onP2FileEditV1(DriveService.P2FileEditV1Handler handler) {
      if (eventType2EventHandler.containsKey("drive.file.edit_v1")) {
        throw new EventTypeAlreadyHasHandlerException("drive.file.edit_v1");
      }
      eventType2EventHandler.put("drive.file.edit_v1", handler);
      return this;
    }

    public Builder onP2FilePermissionMemberAddedV1(
        DriveService.P2FilePermissionMemberAddedV1Handler handler) {
      if (eventType2EventHandler.containsKey("drive.file.permission_member_added_v1")) {
        throw new EventTypeAlreadyHasHandlerException("drive.file.permission_member_added_v1");
      }
      eventType2EventHandler.put("drive.file.permission_member_added_v1", handler);
      return this;
    }

    public Builder onP2FilePermissionMemberRemovedV1(
        DriveService.P2FilePermissionMemberRemovedV1Handler handler) {
      if (eventType2EventHandler.containsKey("drive.file.permission_member_removed_v1")) {
        throw new EventTypeAlreadyHasHandlerException("drive.file.permission_member_removed_v1");
      }
      eventType2EventHandler.put("drive.file.permission_member_removed_v1", handler);
      return this;
    }

    public Builder onP2FileReadV1(DriveService.P2FileReadV1Handler handler) {
      if (eventType2EventHandler.containsKey("drive.file.read_v1")) {
        throw new EventTypeAlreadyHasHandlerException("drive.file.read_v1");
      }
      eventType2EventHandler.put("drive.file.read_v1", handler);
      return this;
    }

    public Builder onP2FileTitleUpdatedV1(DriveService.P2FileTitleUpdatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("drive.file.title_updated_v1")) {
        throw new EventTypeAlreadyHasHandlerException("drive.file.title_updated_v1");
      }
      eventType2EventHandler.put("drive.file.title_updated_v1", handler);
      return this;
    }

    public Builder onP2FileTrashedV1(DriveService.P2FileTrashedV1Handler handler) {
      if (eventType2EventHandler.containsKey("drive.file.trashed_v1")) {
        throw new EventTypeAlreadyHasHandlerException("drive.file.trashed_v1");
      }
      eventType2EventHandler.put("drive.file.trashed_v1", handler);
      return this;
    }


    public Builder onP2NotificationApproveV1(
        HelpdeskService.P2NotificationApproveV1Handler handler) {
      if (eventType2EventHandler.containsKey("helpdesk.notification.approve_v1")) {
        throw new EventTypeAlreadyHasHandlerException("helpdesk.notification.approve_v1");
      }
      eventType2EventHandler.put("helpdesk.notification.approve_v1", handler);
      return this;
    }

    public Builder onP2TicketCreatedV1(HelpdeskService.P2TicketCreatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("helpdesk.ticket.created_v1")) {
        throw new EventTypeAlreadyHasHandlerException("helpdesk.ticket.created_v1");
      }
      eventType2EventHandler.put("helpdesk.ticket.created_v1", handler);
      return this;
    }

    public Builder onP2TicketUpdatedV1(HelpdeskService.P2TicketUpdatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("helpdesk.ticket.updated_v1")) {
        throw new EventTypeAlreadyHasHandlerException("helpdesk.ticket.updated_v1");
      }
      eventType2EventHandler.put("helpdesk.ticket.updated_v1", handler);
      return this;
    }


    public Builder onP2ChatDisbandedV1(ImService.P2ChatDisbandedV1Handler handler) {
      if (eventType2EventHandler.containsKey("im.chat.disbanded_v1")) {
        throw new EventTypeAlreadyHasHandlerException("im.chat.disbanded_v1");
      }
      eventType2EventHandler.put("im.chat.disbanded_v1", handler);
      return this;
    }

    public Builder onP2ChatUpdatedV1(ImService.P2ChatUpdatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("im.chat.updated_v1")) {
        throw new EventTypeAlreadyHasHandlerException("im.chat.updated_v1");
      }
      eventType2EventHandler.put("im.chat.updated_v1", handler);
      return this;
    }

    public Builder onP2ChatMemberBotAddedV1(ImService.P2ChatMemberBotAddedV1Handler handler) {
      if (eventType2EventHandler.containsKey("im.chat.member.bot.added_v1")) {
        throw new EventTypeAlreadyHasHandlerException("im.chat.member.bot.added_v1");
      }
      eventType2EventHandler.put("im.chat.member.bot.added_v1", handler);
      return this;
    }

    public Builder onP2ChatMemberBotDeletedV1(ImService.P2ChatMemberBotDeletedV1Handler handler) {
      if (eventType2EventHandler.containsKey("im.chat.member.bot.deleted_v1")) {
        throw new EventTypeAlreadyHasHandlerException("im.chat.member.bot.deleted_v1");
      }
      eventType2EventHandler.put("im.chat.member.bot.deleted_v1", handler);
      return this;
    }

    public Builder onP2ChatMemberUserAddedV1(ImService.P2ChatMemberUserAddedV1Handler handler) {
      if (eventType2EventHandler.containsKey("im.chat.member.user.added_v1")) {
        throw new EventTypeAlreadyHasHandlerException("im.chat.member.user.added_v1");
      }
      eventType2EventHandler.put("im.chat.member.user.added_v1", handler);
      return this;
    }

    public Builder onP2ChatMemberUserDeletedV1(ImService.P2ChatMemberUserDeletedV1Handler handler) {
      if (eventType2EventHandler.containsKey("im.chat.member.user.deleted_v1")) {
        throw new EventTypeAlreadyHasHandlerException("im.chat.member.user.deleted_v1");
      }
      eventType2EventHandler.put("im.chat.member.user.deleted_v1", handler);
      return this;
    }

    public Builder onP2ChatMemberUserWithdrawnV1(
        ImService.P2ChatMemberUserWithdrawnV1Handler handler) {
      if (eventType2EventHandler.containsKey("im.chat.member.user.withdrawn_v1")) {
        throw new EventTypeAlreadyHasHandlerException("im.chat.member.user.withdrawn_v1");
      }
      eventType2EventHandler.put("im.chat.member.user.withdrawn_v1", handler);
      return this;
    }

    public Builder onP2MessageReadV1(ImService.P2MessageReadV1Handler handler) {
      if (eventType2EventHandler.containsKey("im.message.message_read_v1")) {
        throw new EventTypeAlreadyHasHandlerException("im.message.message_read_v1");
      }
      eventType2EventHandler.put("im.message.message_read_v1", handler);
      return this;
    }

    public Builder onP2MessageReceiveV1(ImService.P2MessageReceiveV1Handler handler) {
      if (eventType2EventHandler.containsKey("im.message.receive_v1")) {
        throw new EventTypeAlreadyHasHandlerException("im.message.receive_v1");
      }
      eventType2EventHandler.put("im.message.receive_v1", handler);
      return this;
    }

    public Builder onP2MessageReactionCreatedV1(
        ImService.P2MessageReactionCreatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("im.message.reaction.created_v1")) {
        throw new EventTypeAlreadyHasHandlerException("im.message.reaction.created_v1");
      }
      eventType2EventHandler.put("im.message.reaction.created_v1", handler);
      return this;
    }

    public Builder onP2MessageReactionDeletedV1(
        ImService.P2MessageReactionDeletedV1Handler handler) {
      if (eventType2EventHandler.containsKey("im.message.reaction.deleted_v1")) {
        throw new EventTypeAlreadyHasHandlerException("im.message.reaction.deleted_v1");
      }
      eventType2EventHandler.put("im.message.reaction.deleted_v1", handler);
      return this;
    }


    public Builder onP2MeetingRoomCreatedV1(
        MeetingRoomService.P2MeetingRoomCreatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("meeting_room.meeting_room.created_v1")) {
        throw new EventTypeAlreadyHasHandlerException("meeting_room.meeting_room.created_v1");
      }
      eventType2EventHandler.put("meeting_room.meeting_room.created_v1", handler);
      return this;
    }

    public Builder onP2MeetingRoomDeletedV1(
        MeetingRoomService.P2MeetingRoomDeletedV1Handler handler) {
      if (eventType2EventHandler.containsKey("meeting_room.meeting_room.deleted_v1")) {
        throw new EventTypeAlreadyHasHandlerException("meeting_room.meeting_room.deleted_v1");
      }
      eventType2EventHandler.put("meeting_room.meeting_room.deleted_v1", handler);
      return this;
    }

    public Builder onP2MeetingRoomStatusChangedV1(
        MeetingRoomService.P2MeetingRoomStatusChangedV1Handler handler) {
      if (eventType2EventHandler.containsKey("meeting_room.meeting_room.status_changed_v1")) {
        throw new EventTypeAlreadyHasHandlerException(
            "meeting_room.meeting_room.status_changed_v1");
      }
      eventType2EventHandler.put("meeting_room.meeting_room.status_changed_v1", handler);
      return this;
    }

    public Builder onP2MeetingRoomUpdatedV1(
        MeetingRoomService.P2MeetingRoomUpdatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("meeting_room.meeting_room.updated_v1")) {
        throw new EventTypeAlreadyHasHandlerException("meeting_room.meeting_room.updated_v1");
      }
      eventType2EventHandler.put("meeting_room.meeting_room.updated_v1", handler);
      return this;
    }


    public Builder onP2TaskUpdateTenantV1(TaskService.P2TaskUpdateTenantV1Handler handler) {
      if (eventType2EventHandler.containsKey("task.task.update_tenant_v1")) {
        throw new EventTypeAlreadyHasHandlerException("task.task.update_tenant_v1");
      }
      eventType2EventHandler.put("task.task.update_tenant_v1", handler);
      return this;
    }

    public Builder onP2TaskUpdatedV1(TaskService.P2TaskUpdatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("task.task.updated_v1")) {
        throw new EventTypeAlreadyHasHandlerException("task.task.updated_v1");
      }
      eventType2EventHandler.put("task.task.updated_v1", handler);
      return this;
    }

    public Builder onP2TaskCommentUpdatedV1(TaskService.P2TaskCommentUpdatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("task.task.comment.updated_v1")) {
        throw new EventTypeAlreadyHasHandlerException("task.task.comment.updated_v1");
      }
      eventType2EventHandler.put("task.task.comment.updated_v1", handler);
      return this;
    }


    public Builder onP2MeetingJoinMeetingV1(VcService.P2MeetingJoinMeetingV1Handler handler) {
      if (eventType2EventHandler.containsKey("vc.meeting.join_meeting_v1")) {
        throw new EventTypeAlreadyHasHandlerException("vc.meeting.join_meeting_v1");
      }
      eventType2EventHandler.put("vc.meeting.join_meeting_v1", handler);
      return this;
    }

    public Builder onP2MeetingLeaveMeetingV1(VcService.P2MeetingLeaveMeetingV1Handler handler) {
      if (eventType2EventHandler.containsKey("vc.meeting.leave_meeting_v1")) {
        throw new EventTypeAlreadyHasHandlerException("vc.meeting.leave_meeting_v1");
      }
      eventType2EventHandler.put("vc.meeting.leave_meeting_v1", handler);
      return this;
    }

    public Builder onP2MeetingEndedV1(VcService.P2MeetingEndedV1Handler handler) {
      if (eventType2EventHandler.containsKey("vc.meeting.meeting_ended_v1")) {
        throw new EventTypeAlreadyHasHandlerException("vc.meeting.meeting_ended_v1");
      }
      eventType2EventHandler.put("vc.meeting.meeting_ended_v1", handler);
      return this;
    }

    public Builder onP2MeetingStartedV1(VcService.P2MeetingStartedV1Handler handler) {
      if (eventType2EventHandler.containsKey("vc.meeting.meeting_started_v1")) {
        throw new EventTypeAlreadyHasHandlerException("vc.meeting.meeting_started_v1");
      }
      eventType2EventHandler.put("vc.meeting.meeting_started_v1", handler);
      return this;
    }

    public Builder onP2MeetingRecordingEndedV1(VcService.P2MeetingRecordingEndedV1Handler handler) {
      if (eventType2EventHandler.containsKey("vc.meeting.recording_ended_v1")) {
        throw new EventTypeAlreadyHasHandlerException("vc.meeting.recording_ended_v1");
      }
      eventType2EventHandler.put("vc.meeting.recording_ended_v1", handler);
      return this;
    }

    public Builder onP2MeetingRecordingReadyV1(VcService.P2MeetingRecordingReadyV1Handler handler) {
      if (eventType2EventHandler.containsKey("vc.meeting.recording_ready_v1")) {
        throw new EventTypeAlreadyHasHandlerException("vc.meeting.recording_ready_v1");
      }
      eventType2EventHandler.put("vc.meeting.recording_ready_v1", handler);
      return this;
    }

    public Builder onP2MeetingRecordingStartedV1(
        VcService.P2MeetingRecordingStartedV1Handler handler) {
      if (eventType2EventHandler.containsKey("vc.meeting.recording_started_v1")) {
        throw new EventTypeAlreadyHasHandlerException("vc.meeting.recording_started_v1");
      }
      eventType2EventHandler.put("vc.meeting.recording_started_v1", handler);
      return this;
    }

    public Builder onP2MeetingShareEndedV1(VcService.P2MeetingShareEndedV1Handler handler) {
      if (eventType2EventHandler.containsKey("vc.meeting.share_ended_v1")) {
        throw new EventTypeAlreadyHasHandlerException("vc.meeting.share_ended_v1");
      }
      eventType2EventHandler.put("vc.meeting.share_ended_v1", handler);
      return this;
    }

    public Builder onP2MeetingShareStartedV1(VcService.P2MeetingShareStartedV1Handler handler) {
      if (eventType2EventHandler.containsKey("vc.meeting.share_started_v1")) {
        throw new EventTypeAlreadyHasHandlerException("vc.meeting.share_started_v1");
      }
      eventType2EventHandler.put("vc.meeting.share_started_v1", handler);
      return this;
    }


    public Builder onCustomizedEvent(String eventType, CustomEventHandler handler) {
      if (eventType2EventHandler.containsKey(eventType)) {
        throw new EventTypeAlreadyHasHandlerException(eventType);
      }
      eventType2EventHandler.put(eventType, handler);
      return this;
    }

    public Builder onP1MessageReadV1(ImService.P1MessageReadV1Handler handler) {
      if (eventType2EventHandler.containsKey("message_read")) {
        throw new EventTypeAlreadyHasHandlerException("message_read");
      }
      eventType2EventHandler.put("message_read", handler);
      return this;
    }

    public Builder onP1MessageReceivedV1(ImService.P1MessageReceivedV1Handler handler) {
      if (eventType2EventHandler.containsKey("message")) {
        throw new EventTypeAlreadyHasHandlerException("message");
      }
      eventType2EventHandler.put("message", handler);
      return this;
    }

    public Builder onP1UserChangedV3(ContactService.P1UserChangedV3Handler handler) {
      if (eventType2EventHandler.containsKey("user_add")) {
        throw new EventTypeAlreadyHasHandlerException("user_add");
      }
      eventType2EventHandler.put("user_add", handler);

      if (eventType2EventHandler.containsKey("user_leave")) {
        throw new EventTypeAlreadyHasHandlerException("user_leave");
      }
      eventType2EventHandler.put("user_leave", handler);

      if (eventType2EventHandler.containsKey("user_update")) {
        throw new EventTypeAlreadyHasHandlerException("user_update");
      }
      eventType2EventHandler.put("user_update", handler);
      return this;
    }

    public Builder onP1UserStatusChangedV3(ContactService.P1UserStatusChangedV3Handler handler) {
      if (eventType2EventHandler.containsKey("user_status_change")) {
        throw new EventTypeAlreadyHasHandlerException("user_status_change");
      }
      eventType2EventHandler.put("user_status_change", handler);
      return this;
    }

    public Builder onP1ContactScopeChangedV3(
        ContactService.P1ContactScopeChangedV3Handler handler) {
      if (eventType2EventHandler.containsKey("contact_scope_change")) {
        throw new EventTypeAlreadyHasHandlerException("contact_scope_change");
      }
      eventType2EventHandler.put("contact_scope_change", handler);
      return this;
    }

    public Builder onP1DepartmentChangedV3(ContactService.P1DepartmentChangedV3Handler handler) {
      if (eventType2EventHandler.containsKey("dept_add")) {
        throw new EventTypeAlreadyHasHandlerException("dept_add");
      }
      eventType2EventHandler.put("dept_add", handler);

      if (eventType2EventHandler.containsKey("dept_update")) {
        throw new EventTypeAlreadyHasHandlerException("dept_update");
      }
      eventType2EventHandler.put("dept_update", handler);

      if (eventType2EventHandler.containsKey("dept_delete")) {
        throw new EventTypeAlreadyHasHandlerException("dept_delete");
      }
      eventType2EventHandler.put("dept_delete", handler);
      return this;
    }

    public Builder onP1P2PChatCreatedV1(ImService.P1P2PChatCreatedV1Handler handler) {
      if (eventType2EventHandler.containsKey("p2p_chat_create")) {
        throw new EventTypeAlreadyHasHandlerException("p2p_chat_create");
      }
      eventType2EventHandler.put("p2p_chat_create", handler);
      return this;
    }

    public Builder onP1ThirdPartyMeetingRoomChangedV1(
        MeetingRoomService.P1ThirdPartyMeetingRoomChangedV1Handler handler) {
      if (eventType2EventHandler.containsKey("third_party_meeting_room_event_created")) {
        throw new EventTypeAlreadyHasHandlerException("third_party_meeting_room_event_created");
      }
      eventType2EventHandler.put("third_party_meeting_room_event_created", handler);

      if (eventType2EventHandler.containsKey("third_party_meeting_room_event_updated")) {
        throw new EventTypeAlreadyHasHandlerException("third_party_meeting_room_event_updated");
      }
      eventType2EventHandler.put("third_party_meeting_room_event_updated", handler);

      if (eventType2EventHandler.containsKey("third_party_meeting_room_event_deleted")) {
        throw new EventTypeAlreadyHasHandlerException("third_party_meeting_room_event_deleted");
      }
      eventType2EventHandler.put("third_party_meeting_room_event_deleted", handler);

      return this;
    }


    public Builder onP1LeaveApprovalV4(ApprovalService.P1LeaveApprovalV4Handler handler) {
      if (eventType2EventHandler.containsKey("leave_approvalV2")) {
        throw new EventTypeAlreadyHasHandlerException("leave_approvalV2");
      }
      eventType2EventHandler.put("leave_approvalV2", handler);
      return this;
    }

    public Builder onP1WorkApprovalV4(ApprovalService.P1WorkApprovalV4Handler handler) {
      if (eventType2EventHandler.containsKey("work_approval")) {
        throw new EventTypeAlreadyHasHandlerException("work_approval");
      }
      eventType2EventHandler.put("work_approval", handler);
      return this;
    }

    public Builder onP1ShiftApprovalV4(ApprovalService.P1ShiftApprovalV4Handler handler) {
      if (eventType2EventHandler.containsKey("shift_approval")) {
        throw new EventTypeAlreadyHasHandlerException("shift_approval");
      }
      eventType2EventHandler.put("shift_approval", handler);
      return this;
    }

    public Builder onP1RemedyApprovalV4(ApprovalService.P1RemedyApprovalV4Handler handler) {
      if (eventType2EventHandler.containsKey("remedy_approval")) {
        throw new EventTypeAlreadyHasHandlerException("remedy_approval");
      }
      eventType2EventHandler.put("remedy_approval", handler);
      return this;
    }

    public Builder onP1TripApprovalV4(ApprovalService.P1TripApprovalV4Handler handler) {
      if (eventType2EventHandler.containsKey("trip_approval")) {
        throw new EventTypeAlreadyHasHandlerException("trip_approval");
      }
      eventType2EventHandler.put("trip_approval", handler);
      return this;
    }

    public Builder onP1OutApprovalV4(ApprovalService.P1OutApprovalV4Handler handler) {
      if (eventType2EventHandler.containsKey("out_approval")) {
        throw new EventTypeAlreadyHasHandlerException("out_approval");
      }
      eventType2EventHandler.put("out_approval", handler);
      return this;
    }

    public Builder onP1AppOpenV6(ApplicationService.P1AppOpenV6Handler handler) {
      if (eventType2EventHandler.containsKey("app_open")) {
        throw new EventTypeAlreadyHasHandlerException("app_open");
      }
      eventType2EventHandler.put("app_open", handler);
      return this;
    }

    public Builder onP1AppStatusChangedV6(ApplicationService.P1AppStatusChangedV6Handler handler) {
      if (eventType2EventHandler.containsKey("app_status_change")) {
        throw new EventTypeAlreadyHasHandlerException("app_status_change");
      }
      eventType2EventHandler.put("app_status_change", handler);
      return this;
    }

    public Builder onP1OrderPaidV6(ApplicationService.P1OrderPaidV6Handler handler) {
      if (eventType2EventHandler.containsKey("order_paid")) {
        throw new EventTypeAlreadyHasHandlerException("order_paid");
      }
      eventType2EventHandler.put("order_paid", handler);
      return this;
    }

    public Builder onP1AppUninstalledV6(ApplicationService.P1AppUninstalledV6Handler handler) {
      if (eventType2EventHandler.containsKey("app_uninstalled")) {
        throw new EventTypeAlreadyHasHandlerException("app_uninstalled");
      }
      eventType2EventHandler.put("app_uninstalled", handler);
      return this;
    }

    // 当 ISV 想要自己管理 token 和 appTicket 时，需要注册该处理器来获取 appTicket。这时 SDK内 将不在管理token
    public Builder onAppTicketEvent(CustomAppTicketEventHandler handler) {
      eventType2EventHandler.put("app_ticket", handler);
      return this;
    }

  }

}


