package com.laserfiche.repository.api.clients;

import com.laserfiche.repository.api.clients.impl.model.Attribute;
import com.laserfiche.repository.api.clients.impl.model.ODataValueContextOfListOfAttribute;

import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

public interface AttributesClient {

    /**
     * - Returns the attribute associated with the key. Alternatively, return the attribute associated with the key within &quot;Everyone&quot; group.
     * - Optional query parameters: everyone (bool, default false). When true, the server only searches for the attribute value with the given key upon the authenticated users attributes. If false, only the authenticated users attributes will be queried.
     *
     * @param repoId       The requested repository ID.
     * @param attributeKey The requested attribute key.
     * @param everyone     Boolean value that indicates whether to return attributes associated with everyone or the currently authenticated user.
     * @return CompletableFuture&lt;Attribute&gt; The return value
     */
    CompletableFuture<Attribute> getTrusteeAttributeValueByKey(String repoId, String attributeKey, Boolean everyone);

    /**
     * - Returns the attribute key value pairs associated with the authenticated user. Alternatively, return only the attribute key value pairs that are associated with the &quot;Everyone&quot; group.
     * - Attribute keys can be used with subsequent calls to get specific attribute values.
     * - Default page size: 100. Allowed OData query options: Select, Count, OrderBy, Skip, Top, SkipToken, Prefer. Optional query parameters: everyone (bool, default false). When true, this route does not return the attributes that are tied to the currently authenticated user, but rather the attributes assigned to the &quot;Everyone&quot; group. Note when this is true, the response does not include both the &quot;Everyone&quot; groups attribute and the currently authenticated user, but only the &quot;Everyone&quot; groups.
     *
     * @param repoId   The requested repository ID.
     * @param everyone Boolean value that indicates whether to return attributes key value pairs associated with everyone or the currently authenticated user.
     * @param prefer   An optional OData header. Can be used to set the maximum page size using odata.maxpagesize.
     * @param select   Limits the properties returned in the result.
     * @param orderby  Specifies the order in which items are returned. The maximum number of expressions is 5.
     * @param top      Limits the number of items returned from a collection.
     * @param skip     Excludes the specified number of items of the queried collection from the result.
     * @param count    Indicates whether the total count of items within a collection are returned in the result.
     * @return CompletableFuture&lt;ODataValueContextOfListOfAttribute&gt; The return value
     */
    CompletableFuture<ODataValueContextOfListOfAttribute> getTrusteeAttributeKeyValuePairs(String repoId,
            Boolean everyone, String prefer, String select, String orderby, Integer top, Integer skip, Boolean count);

    /**
     * Returns the next subset of the requested collection, using a nextlink url.
     *
     * @param nextLink    A url that allows retrieving the next subset of the requested collection.
     * @param maxPageSize Optionally specify the maximum number of items to retrieve.
     * @return CompletableFuture&lt;ODataValueContextOfListOfAttribute&gt; The return value
     */
    CompletableFuture<ODataValueContextOfListOfAttribute> getTrusteeAttributeKeyValuePairsNextLink(String nextLink,
            Integer maxPageSize);

    /**
     * Provides the functionality to iteratively (i.e. through paging) call &lt;b&gt;getTrusteeAttributeKeyValuePairs&lt;/b&gt;, and apply a function on the response of each iteration.
     *
     * @param callback    A delegate that will be called each time new data is retrieved. Returns false to stop receiving more data; returns true to be called again if there's more data.
     * @param maxPageSize Optionally specify the maximum number of items to retrieve.
     * @param repoId      The requested repository ID.
     * @param everyone    Boolean value that indicates whether to return attributes key value pairs associated with everyone or the currently authenticated user.
     * @param prefer      An optional OData header. Can be used to set the maximum page size using odata.maxpagesize.
     * @param select      Limits the properties returned in the result.
     * @param orderby     Specifies the order in which items are returned. The maximum number of expressions is 5.
     * @param top         Limits the number of items returned from a collection.
     * @param skip        Excludes the specified number of items of the queried collection from the result.
     * @param count       Indicates whether the total count of items within a collection are returned in the result.
     * @return CompletableFuture&lt;Void&gt; The return value
     */
    CompletableFuture<Void> getTrusteeAttributeKeyValuePairsForEach(
            Function<CompletableFuture<ODataValueContextOfListOfAttribute>, CompletableFuture<Boolean>> callback,
            Integer maxPageSize, String repoId, Boolean everyone, String prefer, String select, String orderby,
            Integer top, Integer skip, Boolean count);
}
