package com.laserfiche.repository.api.clients.impl;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.laserfiche.repository.api.clients.impl.deserialization.OffsetDateTimeDeserializer;
import com.laserfiche.repository.api.clients.impl.model.ProblemDetails;
import kong.unirest.Header;
import kong.unirest.Headers;
import kong.unirest.UnirestInstance;
import org.threeten.bp.OffsetDateTime;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class ApiClient {

    protected String baseUrl;

    protected UnirestInstance httpClient;

    protected ObjectMapper objectMapper;

    public ApiClient(String baseUrl, UnirestInstance httpClient) {
        this.baseUrl = baseUrl;
        this.httpClient = httpClient;
        SimpleModule module = new SimpleModule();
        module.addDeserializer(OffsetDateTime.class, new OffsetDateTimeDeserializer());
        this.objectMapper = JsonMapper
                .builder()
                .addModule(module)
                .disable(MapperFeature.CAN_OVERRIDE_ACCESS_MODIFIERS)
                .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
                .enable(MapperFeature.ACCEPT_CASE_INSENSITIVE_ENUMS)
                .build();
    }

    protected String mergeMaxSizeIntoPrefer(Integer maxSize, String prefer) {
        if (maxSize == null)
            return prefer;
        else
            return prefer == null ? String.format("maxpagesize=%d", maxSize) : String.format("%s; maxpagesize=%d",
                    prefer, maxSize);
    }

    protected Map<String, Object> getNonNullParameters(String[] parameterNames, Object[] parameters) {
        if (parameterNames == null || parameters == null) {
            throw new IllegalArgumentException("Input cannot be null.");
        }
        if (parameterNames.length != parameters.length) {
            throw new IllegalArgumentException("The array for parameter name and value should have the same length.");
        }
        Map<String, Object> paramKeyValuePairs = new HashMap<>();
        for (int i = 0; i < parameters.length; i++) {
            if (parameters[i] != null) {
                List<Object> values = new ArrayList<>();
                if (parameters[i] instanceof Object[]) {
                    Object[] objects = (Object[]) parameters[i];
                    for (Object object : objects) {
                        values.add(object);
                    }
                } else {
                    values.add(parameters[i]);
                }
                if (values.size() == 1) {
                    paramKeyValuePairs.put(parameterNames[i], values.get(0));
                } else {
                    paramKeyValuePairs.put(parameterNames[i], values);
                }
            }
        }
        return paramKeyValuePairs;
    }

    protected String toJson(Object object) {
        String json = null;
        try {
            json = objectMapper.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            System.err.println(e);
        }
        return json;
    }

    protected Map<String, String> getHeadersMap(Headers headers) {
        return headers
                .all()
                .stream()
                .collect(Collectors.toMap(Header::getName, Header::getValue));
    }

    protected ProblemDetails deserializeToProblemDetails(String jsonString) throws JsonProcessingException {
        ProblemDetails problemDetails = objectMapper.readValue(jsonString, ProblemDetails.class);
        if (problemDetails.get("title") != null)
            problemDetails.setTitle(problemDetails
                    .get("title")
                    .toString());
        if (problemDetails.get("type") != null)
            problemDetails.setType(problemDetails
                    .get("type")
                    .toString());
        if (problemDetails.get("instance") != null)
            problemDetails.setInstance(problemDetails
                    .get("instance")
                    .toString());
        if (problemDetails.get("detail") != null)
            problemDetails.setDetail(problemDetails
                    .get("detail")
                    .toString());
        problemDetails.setStatus(Integer.parseInt(problemDetails
                .get("status")
                .toString()));
        problemDetails.setExtensions((Map<String, Object>) problemDetails.get("extensions"));
        return problemDetails;
    }

    protected String decideErrorMessage(ProblemDetails problemDetails, String genericErrorMessage) {
        return (problemDetails != null && problemDetails.getTitle() != null && problemDetails
                .getTitle()
                .trim()
                .length() > 0) ? problemDetails.getTitle() : genericErrorMessage;
    }
}
