package com.launchdarkly.sdk.android;

import com.launchdarkly.logging.LDLogger;
import com.launchdarkly.sdk.ContextKind;
import com.launchdarkly.sdk.android.subsystems.PersistentDataStore;

import java.util.ArrayList;
import java.util.List;

abstract class Migration {
    // This logic updates or removes persistent data generated by earlier versions of the SDK for
    // use in the current version.
    //
    // Most SDK releases do not change the persistent data format. For those that do, the first
    // time the SDK is run with that version, after doing any necessary migration it will store a
    // key such as "v4.0.0" under the namespace that keeps track of migrations, so that any
    // subsequent migration knows to translate from that version of the schema.
    //
    // It's not always possible to migrate older data, in which case we will simply delete it to
    // avoid accumulating obsolete data during updates. See comments below regarding the migration
    // issues for each schema version.
    //
    // The reason that this class operates on the PersistentDataStore abstraction, rather than
    // directly on SharedPreferences, is that if some custom implementation of PersistentDataStore
    // is being used in the SDK instead of the default SharedPreferences implementation, it is
    // still desirable to migrate SDK data if we change the schema in the future.

    static final String SHARED_PREFS_BASE_KEY = "LaunchDarkly-";
    static final String MIGRATIONS_NAMESPACE = SHARED_PREFS_BASE_KEY + "migrations";
    static final String CURRENT_SCHEMA_ID = "v4.0.0";

    private Migration() {}

    static void migrateWhenNeeded(PersistentDataStore store, LDLogger logger) {
        if (store.getValue(MIGRATIONS_NAMESPACE, CURRENT_SCHEMA_ID) != null) {
            return;
        }

        updateFromPre_v4_0_0(store, logger);

        store.setValue(MIGRATIONS_NAMESPACE, CURRENT_SCHEMA_ID, CURRENT_SCHEMA_ID);
    }

    static void updateFromPre_v4_0_0(PersistentDataStore store, LDLogger logger) {
        // Prior to the v4.0.0 release, all supported versions have been using the v2.7.0 schema. We
        // are not able to migrate stored flag data from v2.7.0, because the storage namespaces are
        // based on one-way hashes of the full JSON representation of a user, whereas now we are just
        // hashing the key; there's no way to reconstruct just the key value from the existing hash. So
        // we will simply delete any stored flag data from the old schema.
        //
        // We are able to migrate the generated anonymous user key, if present, which we persist as
        // a generated key for the context kind "user".

        List<String> namespacesToDelete = new ArrayList<>();
        String generatedAnonUserKey = null;
        for (String namespace: store.getAllNamespaces()) {
            if (namespace.startsWith(SHARED_PREFS_BASE_KEY)) {
                namespacesToDelete.add(namespace);
                if (namespace.equals(SHARED_PREFS_BASE_KEY + "id")) {
                    generatedAnonUserKey = store.getValue(namespace, "instanceId");
                }
            }
        }
        for (String namespace: namespacesToDelete) {
            store.clear(namespace, true);
        }
        if (generatedAnonUserKey != null) {
            PersistentDataStoreWrapper w = new PersistentDataStoreWrapper(store, logger);
            w.setGeneratedContextKey(ContextKind.DEFAULT, generatedAnonUserKey);
        }

        if (namespacesToDelete.size() != 0) {
            logger.debug("initialized v4.0.0 store schema and removed earlier SDK data");
        }
    }
}
