/*
 * Decompiled with CFR 0.152.
 */
package com.launchdarkly.client;

import com.google.common.annotations.VisibleForTesting;
import com.google.gson.JsonElement;
import com.launchdarkly.client.CustomEvent;
import com.launchdarkly.client.EventProcessor;
import com.launchdarkly.client.FeatureRep;
import com.launchdarkly.client.FeatureRequestEvent;
import com.launchdarkly.client.FeatureRequestor;
import com.launchdarkly.client.IdentifyEvent;
import com.launchdarkly.client.LDConfig;
import com.launchdarkly.client.LDUser;
import com.launchdarkly.client.NewRelicReflector;
import com.launchdarkly.client.PollingProcessor;
import com.launchdarkly.client.StreamProcessor;
import com.launchdarkly.client.UpdateProcessor;
import java.io.Closeable;
import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import org.apache.http.annotation.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class LDClient
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(LDClient.class);
    private final LDConfig config;
    private final FeatureRequestor requestor;
    private final EventProcessor eventProcessor;
    private UpdateProcessor updateProcessor;
    protected static final String CLIENT_VERSION = LDClient.getClientVersion();

    public LDClient(String apiKey) {
        this(apiKey, LDConfig.DEFAULT);
    }

    public LDClient(String apiKey, LDConfig config) {
        this.config = config;
        this.requestor = this.createFeatureRequestor(apiKey, config);
        this.eventProcessor = this.createEventProcessor(apiKey, config);
        if (config.offline) {
            logger.info("Starting LaunchDarkly client in offline mode");
            return;
        }
        if (config.useLdd) {
            logger.info("Starting LaunchDarkly in LDD mode. Skipping direct feature retrieval.");
            return;
        }
        if (config.stream) {
            logger.info("Enabling streaming API");
            this.updateProcessor = this.createStreamProcessor(apiKey, config, this.requestor);
        } else {
            logger.info("Disabling streaming API");
            this.updateProcessor = this.createPollingProcessor(config);
        }
        Future<Void> startFuture = this.updateProcessor.start();
        if (config.startWaitMillis > 0L) {
            logger.info("Waiting up to " + config.startWaitMillis + " milliseconds for LaunchDarkly client to start...");
            try {
                startFuture.get(config.startWaitMillis, TimeUnit.MILLISECONDS);
            }
            catch (TimeoutException e) {
                logger.error("Timeout encountered waiting for LaunchDarkly client initialization");
            }
            catch (Exception e) {
                logger.error("Exception encountered waiting for LaunchDarkly client initialization", (Throwable)e);
            }
        }
    }

    public boolean initialized() {
        return this.isOffline() || this.config.useLdd || this.updateProcessor.initialized();
    }

    @VisibleForTesting
    protected FeatureRequestor createFeatureRequestor(String apiKey, LDConfig config) {
        return new FeatureRequestor(apiKey, config);
    }

    @VisibleForTesting
    protected EventProcessor createEventProcessor(String apiKey, LDConfig config) {
        return new EventProcessor(apiKey, config);
    }

    @VisibleForTesting
    protected StreamProcessor createStreamProcessor(String apiKey, LDConfig config, FeatureRequestor requestor) {
        return new StreamProcessor(apiKey, config, requestor);
    }

    @VisibleForTesting
    protected PollingProcessor createPollingProcessor(LDConfig config) {
        return new PollingProcessor(config, this.requestor);
    }

    public void track(String eventName, LDUser user, JsonElement data) {
        if (this.isOffline()) {
            return;
        }
        boolean processed = this.eventProcessor.sendEvent(new CustomEvent(eventName, user, data));
        if (!processed) {
            logger.warn("Exceeded event queue capacity. Increase capacity to avoid dropping events.");
        }
    }

    public void track(String eventName, LDUser user) {
        if (this.isOffline()) {
            return;
        }
        this.track(eventName, user, null);
    }

    public void identify(LDUser user) {
        if (this.isOffline()) {
            return;
        }
        boolean processed = this.eventProcessor.sendEvent(new IdentifyEvent(user));
        if (!processed) {
            logger.warn("Exceeded event queue capacity. Increase capacity to avoid dropping events.");
        }
    }

    private void sendFlagRequestEvent(String featureKey, LDUser user, boolean value, boolean defaultValue) {
        if (this.isOffline()) {
            return;
        }
        boolean processed = this.eventProcessor.sendEvent(new FeatureRequestEvent<Boolean>(featureKey, user, value, defaultValue));
        if (!processed) {
            logger.warn("Exceeded event queue capacity. Increase capacity to avoid dropping events.");
        }
        NewRelicReflector.annotateTransaction(featureKey, String.valueOf(value));
    }

    public boolean getFlag(String featureKey, LDUser user, boolean defaultValue) {
        return this.toggle(featureKey, user, defaultValue);
    }

    public Map<String, Boolean> allFlags(LDUser user) {
        if (this.isOffline()) {
            return null;
        }
        if (!this.initialized()) {
            return null;
        }
        Map<String, FeatureRep<?>> flags = this.config.featureStore.all();
        HashMap<String, Boolean> result = new HashMap<String, Boolean>();
        for (String key : flags.keySet()) {
            result.put(key, this.evaluate(key, user, null));
        }
        return result;
    }

    public boolean toggle(String featureKey, LDUser user, boolean defaultValue) {
        if (this.isOffline()) {
            return defaultValue;
        }
        boolean value = this.evaluate(featureKey, user, defaultValue);
        this.sendFlagRequestEvent(featureKey, user, value, defaultValue);
        return value;
    }

    private Boolean evaluate(String featureKey, LDUser user, Boolean defaultValue) {
        if (!this.initialized()) {
            return defaultValue;
        }
        try {
            FeatureRep<?> result = this.config.featureStore.get(featureKey);
            if (result != null) {
                FeatureRep pollingResult;
                if (this.config.stream && this.config.debugStreaming && !result.equals(pollingResult = this.requestor.makeRequest(featureKey, true))) {
                    logger.warn("Mismatch between streaming and polling feature! Streaming: {} Polling: {}", result, pollingResult);
                }
            } else {
                logger.warn("Unknown feature flag " + featureKey + "; returning default value: ");
                return defaultValue;
            }
            Boolean val = (Boolean)result.evaluate(user);
            if (val == null) {
                return defaultValue;
            }
            return val;
        }
        catch (Exception e) {
            logger.error("Encountered exception in LaunchDarkly client", (Throwable)e);
            return defaultValue;
        }
    }

    @Override
    public void close() throws IOException {
        this.eventProcessor.close();
        if (this.updateProcessor != null) {
            this.updateProcessor.close();
        }
    }

    public void flush() {
        this.eventProcessor.flush();
    }

    public boolean isOffline() {
        return this.config.offline;
    }

    private static String getClientVersion() {
        Class<LDConfig> clazz = LDConfig.class;
        String className = clazz.getSimpleName() + ".class";
        String classPath = clazz.getResource(className).toString();
        if (!classPath.startsWith("jar")) {
            return "Unknown";
        }
        String manifestPath = classPath.substring(0, classPath.lastIndexOf("!") + 1) + "/META-INF/MANIFEST.MF";
        Manifest manifest = null;
        try {
            manifest = new Manifest(new URL(manifestPath).openStream());
            Attributes attr = manifest.getMainAttributes();
            String value = attr.getValue("Implementation-Version");
            return value;
        }
        catch (IOException e) {
            logger.warn("Unable to determine LaunchDarkly client library version", (Throwable)e);
            return "Unknown";
        }
    }
}

