/*
 * Decompiled with CFR 0.152.
 */
package com.launchdarkly.client;

import com.google.gson.JsonElement;
import com.google.gson.JsonPrimitive;
import com.launchdarkly.client.CustomEvent;
import com.launchdarkly.client.EvaluationException;
import com.launchdarkly.client.Event;
import com.launchdarkly.client.EventProcessor;
import com.launchdarkly.client.FeatureFlag;
import com.launchdarkly.client.FeatureRequestEvent;
import com.launchdarkly.client.FeatureRequestor;
import com.launchdarkly.client.IdentifyEvent;
import com.launchdarkly.client.LDClientInterface;
import com.launchdarkly.client.LDConfig;
import com.launchdarkly.client.LDUser;
import com.launchdarkly.client.NewRelicReflector;
import com.launchdarkly.client.PollingProcessor;
import com.launchdarkly.client.StreamProcessor;
import com.launchdarkly.client.UpdateProcessor;
import com.launchdarkly.client.VariationType;
import com.launchdarkly.shaded.com.google.common.annotations.VisibleForTesting;
import com.launchdarkly.shaded.org.apache.commons.codec.binary.Hex;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LDClient
implements LDClientInterface {
    private static final Logger logger = LoggerFactory.getLogger(LDClient.class);
    private static final String HMAC_ALGORITHM = "HmacSHA256";
    static final String CLIENT_VERSION = LDClient.getClientVersion();
    private final LDConfig config;
    private final String sdkKey;
    private final FeatureRequestor requestor;
    private final EventProcessor eventProcessor;
    private UpdateProcessor updateProcessor;
    private final AtomicBoolean eventCapacityExceeded = new AtomicBoolean(false);

    public LDClient(String sdkKey) {
        this(sdkKey, LDConfig.DEFAULT);
    }

    public LDClient(String sdkKey, LDConfig config) {
        this.config = config;
        this.sdkKey = sdkKey;
        this.requestor = this.createFeatureRequestor(sdkKey, config);
        this.eventProcessor = this.createEventProcessor(sdkKey, config);
        if (config.offline) {
            logger.info("Starting LaunchDarkly client in offline mode");
            return;
        }
        if (config.useLdd) {
            logger.info("Starting LaunchDarkly in LDD mode. Skipping direct feature retrieval.");
            return;
        }
        if (config.stream) {
            logger.info("Enabling streaming API");
            this.updateProcessor = this.createStreamProcessor(sdkKey, config, this.requestor);
        } else {
            logger.info("Disabling streaming API");
            logger.warn("You should only disable the streaming API if instructed to do so by LaunchDarkly support");
            this.updateProcessor = this.createPollingProcessor(config);
        }
        Future<Void> startFuture = this.updateProcessor.start();
        if (config.startWaitMillis > 0L) {
            logger.info("Waiting up to " + config.startWaitMillis + " milliseconds for LaunchDarkly client to start...");
            try {
                startFuture.get(config.startWaitMillis, TimeUnit.MILLISECONDS);
            }
            catch (TimeoutException e) {
                logger.error("Timeout encountered waiting for LaunchDarkly client initialization");
            }
            catch (Exception e) {
                logger.error("Exception encountered waiting for LaunchDarkly client initialization", (Throwable)e);
            }
        }
    }

    @Override
    public boolean initialized() {
        return this.isOffline() || this.config.useLdd || this.updateProcessor.initialized();
    }

    @VisibleForTesting
    protected FeatureRequestor createFeatureRequestor(String sdkKey, LDConfig config) {
        return new FeatureRequestor(sdkKey, config);
    }

    @VisibleForTesting
    protected EventProcessor createEventProcessor(String sdkKey, LDConfig config) {
        return new EventProcessor(sdkKey, config);
    }

    @VisibleForTesting
    protected StreamProcessor createStreamProcessor(String sdkKey, LDConfig config, FeatureRequestor requestor) {
        return new StreamProcessor(sdkKey, config, requestor);
    }

    @VisibleForTesting
    protected PollingProcessor createPollingProcessor(LDConfig config) {
        return new PollingProcessor(config, this.requestor);
    }

    @Override
    public void track(String eventName, LDUser user, JsonElement data) {
        if (this.isOffline()) {
            return;
        }
        if (user == null || user.getKey() == null) {
            logger.warn("Track called with null user or null user key!");
        }
        this.sendEvent(new CustomEvent(eventName, user, data));
    }

    @Override
    public void track(String eventName, LDUser user) {
        if (this.isOffline()) {
            return;
        }
        this.track(eventName, user, null);
    }

    @Override
    public void identify(LDUser user) {
        if (user == null || user.getKey() == null) {
            logger.warn("Identify called with null user or null user key!");
        }
        this.sendEvent(new IdentifyEvent(user));
    }

    private void sendFlagRequestEvent(String featureKey, LDUser user, JsonElement value, JsonElement defaultValue, Integer version) {
        if (this.sendEvent(new FeatureRequestEvent(featureKey, user, value, defaultValue, version, null))) {
            NewRelicReflector.annotateTransaction(featureKey, String.valueOf(value));
        }
    }

    private boolean sendEvent(Event event) {
        if (this.isOffline() || !this.config.sendEvents) {
            return false;
        }
        boolean processed = this.eventProcessor.sendEvent(event);
        if (processed) {
            this.eventCapacityExceeded.compareAndSet(true, false);
        } else if (this.eventCapacityExceeded.compareAndSet(false, true)) {
            logger.warn("Exceeded event queue capacity. Increase capacity to avoid dropping events.");
        }
        return true;
    }

    @Override
    public Map<String, JsonElement> allFlags(LDUser user) {
        if (this.isOffline()) {
            logger.debug("allFlags() was called when client is in offline mode.");
        }
        if (!this.initialized()) {
            if (this.config.featureStore.initialized()) {
                logger.warn("allFlags() was called before client initialized; using last known values from feature store");
            } else {
                logger.warn("allFlags() was called before client initialized; feature store unavailable, returning null");
                return null;
            }
        }
        if (user == null || user.getKey() == null) {
            logger.warn("allFlags() was called with null user or null user key! returning null");
            return null;
        }
        Map<String, FeatureFlag> flags = this.config.featureStore.all();
        HashMap<String, JsonElement> result = new HashMap<String, JsonElement>();
        for (Map.Entry<String, FeatureFlag> entry : flags.entrySet()) {
            try {
                JsonElement evalResult = entry.getValue().evaluate(user, this.config.featureStore).getValue();
                result.put(entry.getKey(), evalResult);
            }
            catch (EvaluationException e) {
                logger.error("Exception caught when evaluating all flags:", (Throwable)e);
            }
        }
        return result;
    }

    @Override
    public boolean boolVariation(String featureKey, LDUser user, boolean defaultValue) {
        JsonElement value = this.evaluate(featureKey, user, (JsonElement)new JsonPrimitive(Boolean.valueOf(defaultValue)), VariationType.Boolean);
        return value.getAsJsonPrimitive().getAsBoolean();
    }

    @Override
    @Deprecated
    public boolean toggle(String featureKey, LDUser user, boolean defaultValue) {
        logger.warn("Deprecated method: Toggle() called. Use boolVariation() instead.");
        return this.boolVariation(featureKey, user, defaultValue);
    }

    @Override
    public Integer intVariation(String featureKey, LDUser user, int defaultValue) {
        JsonElement value = this.evaluate(featureKey, user, (JsonElement)new JsonPrimitive((Number)defaultValue), VariationType.Integer);
        return value.getAsJsonPrimitive().getAsInt();
    }

    @Override
    public Double doubleVariation(String featureKey, LDUser user, Double defaultValue) {
        JsonElement value = this.evaluate(featureKey, user, (JsonElement)new JsonPrimitive((Number)defaultValue), VariationType.Double);
        return value.getAsJsonPrimitive().getAsDouble();
    }

    @Override
    public String stringVariation(String featureKey, LDUser user, String defaultValue) {
        JsonElement value = this.evaluate(featureKey, user, (JsonElement)new JsonPrimitive(defaultValue), VariationType.String);
        return value.getAsJsonPrimitive().getAsString();
    }

    @Override
    public JsonElement jsonVariation(String featureKey, LDUser user, JsonElement defaultValue) {
        JsonElement value = this.evaluate(featureKey, user, defaultValue, VariationType.Json);
        return value;
    }

    @Override
    public boolean isFlagKnown(String featureKey) {
        if (!this.initialized()) {
            if (this.config.featureStore.initialized()) {
                logger.warn("isFlagKnown called before client initialized for feature flag " + featureKey + "; using last known values from feature store");
            } else {
                logger.warn("isFlagKnown called before client initialized for feature flag " + featureKey + "; feature store unavailable, returning false");
                return false;
            }
        }
        try {
            if (this.config.featureStore.get(featureKey) != null) {
                return true;
            }
        }
        catch (Exception e) {
            logger.error("Encountered exception in LaunchDarkly client", (Throwable)e);
        }
        return false;
    }

    private JsonElement evaluate(String featureKey, LDUser user, JsonElement defaultValue, VariationType expectedType) {
        if (user == null || user.getKey() == null) {
            logger.warn("Null user or null user key when evaluating flag: " + featureKey + "; returning default value");
            this.sendFlagRequestEvent(featureKey, user, defaultValue, defaultValue, null);
            return defaultValue;
        }
        if (user.getKeyAsString().isEmpty()) {
            logger.warn("User key is blank. Flag evaluation will proceed, but the user will not be stored in LaunchDarkly");
        }
        if (!this.initialized()) {
            if (this.config.featureStore.initialized()) {
                logger.warn("Evaluation called before client initialized for feature flag " + featureKey + "; using last known values from feature store");
            } else {
                logger.warn("Evaluation called before client initialized for feature flag " + featureKey + "; feature store unavailable, returning default value");
                this.sendFlagRequestEvent(featureKey, user, defaultValue, defaultValue, null);
                return defaultValue;
            }
        }
        try {
            FeatureFlag featureFlag = this.config.featureStore.get(featureKey);
            if (featureFlag == null) {
                logger.warn("Unknown feature flag " + featureKey + "; returning default value");
                this.sendFlagRequestEvent(featureKey, user, defaultValue, defaultValue, null);
                return defaultValue;
            }
            FeatureFlag.EvalResult evalResult = featureFlag.evaluate(user, this.config.featureStore);
            for (FeatureRequestEvent event : evalResult.getPrerequisiteEvents()) {
                this.sendEvent(event);
            }
            if (evalResult.getValue() != null) {
                expectedType.assertResultType(evalResult.getValue());
                this.sendFlagRequestEvent(featureKey, user, evalResult.getValue(), defaultValue, featureFlag.getVersion());
                return evalResult.getValue();
            }
        }
        catch (Exception e) {
            logger.error("Encountered exception in LaunchDarkly client", (Throwable)e);
        }
        this.sendFlagRequestEvent(featureKey, user, defaultValue, defaultValue, null);
        return defaultValue;
    }

    @Override
    public void close() throws IOException {
        logger.info("Closing LaunchDarkly Client");
        this.eventProcessor.close();
        if (this.updateProcessor != null) {
            this.updateProcessor.close();
        }
        if (this.config.httpClient != null) {
            if (this.config.httpClient.dispatcher() != null && this.config.httpClient.dispatcher().executorService() != null) {
                this.config.httpClient.dispatcher().cancelAll();
                this.config.httpClient.dispatcher().executorService().shutdownNow();
            }
            if (this.config.httpClient.connectionPool() != null) {
                this.config.httpClient.connectionPool().evictAll();
            }
            if (this.config.httpClient.cache() != null) {
                this.config.httpClient.cache().close();
            }
        }
    }

    @Override
    public void flush() {
        this.eventProcessor.flush();
    }

    @Override
    public boolean isOffline() {
        return this.config.offline;
    }

    @Override
    public String secureModeHash(LDUser user) {
        if (user == null || user.getKey() == null) {
            return null;
        }
        try {
            Mac mac = Mac.getInstance(HMAC_ALGORITHM);
            mac.init(new SecretKeySpec(this.sdkKey.getBytes(), HMAC_ALGORITHM));
            return Hex.encodeHexString(mac.doFinal(user.getKeyAsString().getBytes("UTF8")));
        }
        catch (UnsupportedEncodingException | InvalidKeyException | NoSuchAlgorithmException e) {
            logger.error("Could not generate secure mode hash", (Throwable)e);
            return null;
        }
    }

    private static String getClientVersion() {
        Class<LDConfig> clazz = LDConfig.class;
        String className = clazz.getSimpleName() + ".class";
        String classPath = clazz.getResource(className).toString();
        if (!classPath.startsWith("jar")) {
            return "Unknown";
        }
        String manifestPath = classPath.substring(0, classPath.lastIndexOf("!") + 1) + "/META-INF/MANIFEST.MF";
        Manifest manifest = null;
        try {
            manifest = new Manifest(new URL(manifestPath).openStream());
            Attributes attr = manifest.getMainAttributes();
            String value = attr.getValue("Implementation-Version");
            return value;
        }
        catch (IOException e) {
            logger.warn("Unable to determine LaunchDarkly client library version", (Throwable)e);
            return "Unknown";
        }
    }
}

