/*
 * Decompiled with CFR 0.152.
 */
package com.launchdarkly.client;

import com.google.gson.JsonElement;
import com.google.gson.JsonPrimitive;
import com.launchdarkly.client.Components;
import com.launchdarkly.client.EvaluationException;
import com.launchdarkly.client.Event;
import com.launchdarkly.client.EventFactory;
import com.launchdarkly.client.EventProcessor;
import com.launchdarkly.client.EventProcessorFactory;
import com.launchdarkly.client.FeatureFlag;
import com.launchdarkly.client.FeatureStore;
import com.launchdarkly.client.FeatureStoreFactory;
import com.launchdarkly.client.LDClientInterface;
import com.launchdarkly.client.LDConfig;
import com.launchdarkly.client.LDUser;
import com.launchdarkly.client.NewRelicReflector;
import com.launchdarkly.client.UpdateProcessor;
import com.launchdarkly.client.UpdateProcessorFactory;
import com.launchdarkly.client.VariationType;
import com.launchdarkly.client.VersionedDataKind;
import com.launchdarkly.shaded.org.apache.commons.codec.binary.Hex;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class LDClient
implements LDClientInterface {
    private static final Logger logger = LoggerFactory.getLogger(LDClient.class);
    private static final String HMAC_ALGORITHM = "HmacSHA256";
    static final String CLIENT_VERSION = LDClient.getClientVersion();
    private final LDConfig config;
    private final String sdkKey;
    final EventProcessor eventProcessor;
    final UpdateProcessor updateProcessor;
    final FeatureStore featureStore;
    final boolean shouldCloseFeatureStore;
    private final EventFactory eventFactory = EventFactory.DEFAULT;

    public LDClient(String sdkKey) {
        this(sdkKey, LDConfig.DEFAULT);
    }

    public LDClient(String sdkKey, LDConfig config) {
        this.config = config;
        this.sdkKey = sdkKey;
        if (config.deprecatedFeatureStore != null) {
            this.featureStore = config.deprecatedFeatureStore;
            this.shouldCloseFeatureStore = false;
        } else {
            FeatureStoreFactory factory = config.featureStoreFactory == null ? Components.inMemoryFeatureStore() : config.featureStoreFactory;
            this.featureStore = factory.createFeatureStore();
            this.shouldCloseFeatureStore = true;
        }
        EventProcessorFactory epFactory = config.eventProcessorFactory == null ? Components.defaultEventProcessor() : config.eventProcessorFactory;
        this.eventProcessor = epFactory.createEventProcessor(sdkKey, config);
        UpdateProcessorFactory upFactory = config.updateProcessorFactory == null ? Components.defaultUpdateProcessor() : config.updateProcessorFactory;
        this.updateProcessor = upFactory.createUpdateProcessor(sdkKey, config, this.featureStore);
        Future<Void> startFuture = this.updateProcessor.start();
        if (config.startWaitMillis > 0L) {
            if (!config.offline && !config.useLdd) {
                logger.info("Waiting up to " + config.startWaitMillis + " milliseconds for LaunchDarkly client to start...");
            }
            try {
                startFuture.get(config.startWaitMillis, TimeUnit.MILLISECONDS);
            }
            catch (TimeoutException e) {
                logger.error("Timeout encountered waiting for LaunchDarkly client initialization");
            }
            catch (Exception e) {
                logger.error("Exception encountered waiting for LaunchDarkly client initialization", (Throwable)e);
            }
        }
    }

    @Override
    public boolean initialized() {
        return this.updateProcessor.initialized();
    }

    @Override
    public void track(String eventName, LDUser user, JsonElement data) {
        if (this.isOffline()) {
            return;
        }
        if (user == null || user.getKey() == null) {
            logger.warn("Track called with null user or null user key!");
        }
        this.eventProcessor.sendEvent(this.eventFactory.newCustomEvent(eventName, user, data));
    }

    @Override
    public void track(String eventName, LDUser user) {
        if (this.isOffline()) {
            return;
        }
        this.track(eventName, user, null);
    }

    @Override
    public void identify(LDUser user) {
        if (user == null || user.getKey() == null) {
            logger.warn("Identify called with null user or null user key!");
        }
        this.eventProcessor.sendEvent(this.eventFactory.newIdentifyEvent(user));
    }

    private void sendFlagRequestEvent(Event.FeatureRequest event) {
        this.eventProcessor.sendEvent(event);
        NewRelicReflector.annotateTransaction(event.key, String.valueOf(event.value));
    }

    @Override
    public Map<String, JsonElement> allFlags(LDUser user) {
        if (this.isOffline()) {
            logger.debug("allFlags() was called when client is in offline mode.");
        }
        if (!this.initialized()) {
            if (this.featureStore.initialized()) {
                logger.warn("allFlags() was called before client initialized; using last known values from feature store");
            } else {
                logger.warn("allFlags() was called before client initialized; feature store unavailable, returning null");
                return null;
            }
        }
        if (user == null || user.getKey() == null) {
            logger.warn("allFlags() was called with null user or null user key! returning null");
            return null;
        }
        Map<String, FeatureFlag> flags = this.featureStore.all(VersionedDataKind.FEATURES);
        HashMap<String, JsonElement> result = new HashMap<String, JsonElement>();
        for (Map.Entry<String, FeatureFlag> entry : flags.entrySet()) {
            try {
                JsonElement evalResult = entry.getValue().evaluate(user, this.featureStore, this.eventFactory).getResult().getValue();
                result.put(entry.getKey(), evalResult);
            }
            catch (EvaluationException e) {
                logger.error("Exception caught when evaluating all flags:", (Throwable)e);
            }
        }
        return result;
    }

    @Override
    public boolean boolVariation(String featureKey, LDUser user, boolean defaultValue) {
        JsonElement value = this.evaluate(featureKey, user, (JsonElement)new JsonPrimitive(Boolean.valueOf(defaultValue)), VariationType.Boolean);
        return value.getAsJsonPrimitive().getAsBoolean();
    }

    @Override
    public Integer intVariation(String featureKey, LDUser user, int defaultValue) {
        JsonElement value = this.evaluate(featureKey, user, (JsonElement)new JsonPrimitive((Number)defaultValue), VariationType.Integer);
        return value.getAsJsonPrimitive().getAsInt();
    }

    @Override
    public Double doubleVariation(String featureKey, LDUser user, Double defaultValue) {
        JsonElement value = this.evaluate(featureKey, user, (JsonElement)new JsonPrimitive((Number)defaultValue), VariationType.Double);
        return value.getAsJsonPrimitive().getAsDouble();
    }

    @Override
    public String stringVariation(String featureKey, LDUser user, String defaultValue) {
        JsonElement value = this.evaluate(featureKey, user, (JsonElement)new JsonPrimitive(defaultValue), VariationType.String);
        return value.getAsJsonPrimitive().getAsString();
    }

    @Override
    public JsonElement jsonVariation(String featureKey, LDUser user, JsonElement defaultValue) {
        JsonElement value = this.evaluate(featureKey, user, defaultValue, VariationType.Json);
        return value;
    }

    @Override
    public boolean isFlagKnown(String featureKey) {
        if (!this.initialized()) {
            if (this.featureStore.initialized()) {
                logger.warn("isFlagKnown called before client initialized for feature flag " + featureKey + "; using last known values from feature store");
            } else {
                logger.warn("isFlagKnown called before client initialized for feature flag " + featureKey + "; feature store unavailable, returning false");
                return false;
            }
        }
        try {
            if (this.featureStore.get(VersionedDataKind.FEATURES, featureKey) != null) {
                return true;
            }
        }
        catch (Exception e) {
            logger.error("Encountered exception in LaunchDarkly client", (Throwable)e);
        }
        return false;
    }

    private JsonElement evaluate(String featureKey, LDUser user, JsonElement defaultValue, VariationType expectedType) {
        if (!this.initialized()) {
            if (this.featureStore.initialized()) {
                logger.warn("Evaluation called before client initialized for feature flag " + featureKey + "; using last known values from feature store");
            } else {
                logger.warn("Evaluation called before client initialized for feature flag " + featureKey + "; feature store unavailable, returning default value");
                this.sendFlagRequestEvent(this.eventFactory.newUnknownFeatureRequestEvent(featureKey, user, defaultValue));
                return defaultValue;
            }
        }
        try {
            FeatureFlag featureFlag = this.featureStore.get(VersionedDataKind.FEATURES, featureKey);
            if (featureFlag == null) {
                logger.info("Unknown feature flag " + featureKey + "; returning default value");
                this.sendFlagRequestEvent(this.eventFactory.newUnknownFeatureRequestEvent(featureKey, user, defaultValue));
                return defaultValue;
            }
            if (user == null || user.getKey() == null) {
                logger.warn("Null user or null user key when evaluating flag: " + featureKey + "; returning default value");
                this.sendFlagRequestEvent(this.eventFactory.newDefaultFeatureRequestEvent(featureFlag, user, defaultValue));
                return defaultValue;
            }
            if (user.getKeyAsString().isEmpty()) {
                logger.warn("User key is blank. Flag evaluation will proceed, but the user will not be stored in LaunchDarkly");
            }
            FeatureFlag.EvalResult evalResult = featureFlag.evaluate(user, this.featureStore, this.eventFactory);
            for (Event.FeatureRequest event : evalResult.getPrerequisiteEvents()) {
                this.eventProcessor.sendEvent(event);
            }
            if (evalResult.getResult() != null && evalResult.getResult().getValue() != null) {
                expectedType.assertResultType(evalResult.getResult().getValue());
                this.sendFlagRequestEvent(this.eventFactory.newFeatureRequestEvent(featureFlag, user, evalResult.getResult(), defaultValue));
                return evalResult.getResult().getValue();
            }
            this.sendFlagRequestEvent(this.eventFactory.newDefaultFeatureRequestEvent(featureFlag, user, defaultValue));
            return defaultValue;
        }
        catch (Exception e) {
            logger.error("Encountered exception in LaunchDarkly client", (Throwable)e);
            this.sendFlagRequestEvent(this.eventFactory.newUnknownFeatureRequestEvent(featureKey, user, defaultValue));
            return defaultValue;
        }
    }

    @Override
    public void close() throws IOException {
        logger.info("Closing LaunchDarkly Client");
        if (this.shouldCloseFeatureStore) {
            this.featureStore.close();
        }
        this.eventProcessor.close();
        this.updateProcessor.close();
        if (this.config.httpClient != null) {
            if (this.config.httpClient.dispatcher() != null && this.config.httpClient.dispatcher().executorService() != null) {
                this.config.httpClient.dispatcher().cancelAll();
                this.config.httpClient.dispatcher().executorService().shutdownNow();
            }
            if (this.config.httpClient.connectionPool() != null) {
                this.config.httpClient.connectionPool().evictAll();
            }
            if (this.config.httpClient.cache() != null) {
                this.config.httpClient.cache().close();
            }
        }
    }

    @Override
    public void flush() {
        this.eventProcessor.flush();
    }

    @Override
    public boolean isOffline() {
        return this.config.offline;
    }

    @Override
    public String secureModeHash(LDUser user) {
        if (user == null || user.getKey() == null) {
            return null;
        }
        try {
            Mac mac = Mac.getInstance(HMAC_ALGORITHM);
            mac.init(new SecretKeySpec(this.sdkKey.getBytes(), HMAC_ALGORITHM));
            return Hex.encodeHexString(mac.doFinal(user.getKeyAsString().getBytes("UTF8")));
        }
        catch (UnsupportedEncodingException | InvalidKeyException | NoSuchAlgorithmException e) {
            logger.error("Could not generate secure mode hash", (Throwable)e);
            return null;
        }
    }

    @Override
    public String version() {
        return CLIENT_VERSION;
    }

    private static String getClientVersion() {
        Class<LDConfig> clazz = LDConfig.class;
        String className = clazz.getSimpleName() + ".class";
        String classPath = clazz.getResource(className).toString();
        if (!classPath.startsWith("jar")) {
            return "Unknown";
        }
        String manifestPath = classPath.substring(0, classPath.lastIndexOf("!") + 1) + "/META-INF/MANIFEST.MF";
        Manifest manifest = null;
        try {
            manifest = new Manifest(new URL(manifestPath).openStream());
            Attributes attr = manifest.getMainAttributes();
            String value = attr.getValue("Implementation-Version");
            return value;
        }
        catch (IOException e) {
            logger.warn("Unable to determine LaunchDarkly client library version", (Throwable)e);
            return "Unknown";
        }
    }
}

