package com.launchdarkly.client;

import com.google.gson.JsonElement;
import com.launchdarkly.client.value.LDValue;

/**
 * Base class for all analytics events that are generated by the client. Also defines all of its own subclasses.
 */
public class Event {
  final long creationDate;
  final LDUser user;

  /**
   * Base event constructor.
   * @param creationDate the timetamp in milliseconds
   * @param user the user associated with the event
   */
  public Event(long creationDate, LDUser user) {
    this.creationDate = creationDate;
    this.user = user;
  }
  
  /**
   * A custom event created with {@link LDClientInterface#track(String, LDUser)} or one of its overloads.
   */
  public static final class Custom extends Event {
    final String key;
    final LDValue data;
    final Double metricValue;

    /**
     * Constructs a custom event.
     * @param timestamp the timestamp in milliseconds
     * @param key the event key
     * @param user the user associated with the event
     * @param data custom data if any (null is the same as {@link LDValue#ofNull()})
     * @param metricValue custom metric value if any
     * @since 4.8.0
     */
    public Custom(long timestamp, String key, LDUser user, LDValue data, Double metricValue) {
      super(timestamp, user);
      this.key = key;
      this.data = data == null ? LDValue.ofNull() : data;
      this.metricValue = metricValue;
    }

    /**
     * Deprecated constructor.
     * @param timestamp the timestamp in milliseconds
     * @param key the event key
     * @param user the user associated with the event
     * @param data custom data if any (null is the same as {@link LDValue#ofNull()})
     * @deprecated
     */
    @Deprecated
    public Custom(long timestamp, String key, LDUser user, JsonElement data) {
      this(timestamp, key, user, LDValue.unsafeFromJsonElement(data), null);
    }
  }

  /**
   * An event created with {@link LDClientInterface#identify(LDUser)}.
   */
  public static final class Identify extends Event {
    /**
     * Constructs an identify event.
     * @param timestamp the timestamp in milliseconds
     * @param user the user associated with the event
     */
    public Identify(long timestamp, LDUser user) {
      super(timestamp, user);
    }
  }

  /**
   * An event created internally by the SDK to hold user data that may be referenced by multiple events.
   */
  public static final class Index extends Event {
    /**
     * Constructs an index event.
     * @param timestamp the timestamp in milliseconds
     * @param user the user associated with the event
     */
    public Index(long timestamp, LDUser user) {
      super(timestamp, user);
    }
  }
  
  /**
   * An event generated by a feature flag evaluation.
   */
  public static final class FeatureRequest extends Event {
    final String key;
    final Integer variation;
    final LDValue value;
    final LDValue defaultVal;
    final Integer version;
    final String prereqOf;
    final boolean trackEvents;
    final Long debugEventsUntilDate;
    final EvaluationReason reason;
    final boolean debug;

    /**
     * Constructs a feature request event.
     * @param timestamp the timestamp in milliseconds
     * @param key the flag key
     * @param user the user associated with the event
     * @param version the flag version, or null if the flag was not found
     * @param variation the result variation, or null if there was an error
     * @param value the result value
     * @param defaultVal the default value passed by the application
     * @param reason the evaluation reason, if it is to be included in the event
     * @param prereqOf if this flag was evaluated as a prerequisite, this is the key of the flag that referenced it
     * @param trackEvents true if full event tracking is turned on for this flag
     * @param debugEventsUntilDate if non-null, the time until which event debugging should be enabled
     * @param debug true if this is a debugging event
     * @since 4.8.0
     */
    public FeatureRequest(long timestamp, String key, LDUser user, Integer version, Integer variation, LDValue value,
        LDValue defaultVal, EvaluationReason reason, String prereqOf, boolean trackEvents, Long debugEventsUntilDate, boolean debug) {
      super(timestamp, user);
      this.key = key;
      this.version = version;
      this.variation = variation;
      this.value = value;
      this.defaultVal = defaultVal;
      this.prereqOf = prereqOf;
      this.trackEvents = trackEvents;
      this.debugEventsUntilDate = debugEventsUntilDate;
      this.reason = reason;
      this.debug = debug;
    }
    
    /**
     * Deprecated constructor.
     * @param timestamp the timestamp in milliseconds
     * @param key the flag key
     * @param user the user associated with the event
     * @param version the flag version, or null if the flag was not found
     * @param variation the result variation, or null if there was an error
     * @param value the result value
     * @param defaultVal the default value passed by the application
     * @param prereqOf if this flag was evaluated as a prerequisite, this is the key of the flag that referenced it
     * @param trackEvents true if full event tracking is turned on for this flag
     * @param debugEventsUntilDate if non-null, the time until which event debugging should be enabled
     * @param debug true if this is a debugging event
     * @deprecated
     */
    @Deprecated
    public FeatureRequest(long timestamp, String key, LDUser user, Integer version, Integer variation, JsonElement value,
        JsonElement defaultVal, String prereqOf, boolean trackEvents, Long debugEventsUntilDate, boolean debug) {
      this(timestamp, key, user, version, variation, LDValue.unsafeFromJsonElement(value), LDValue.unsafeFromJsonElement(defaultVal),
          null, prereqOf, trackEvents, debugEventsUntilDate, debug);
    }

    /**
     * Deprecated constructor.
     * @param timestamp the timestamp in milliseconds
     * @param key the flag key
     * @param user the user associated with the event
     * @param version the flag version, or null if the flag was not found
     * @param variation the result variation, or null if there was an error
     * @param value the result value
     * @param defaultVal the default value passed by the application
     * @param reason the evaluation reason, if it is to be included in the event
     * @param prereqOf if this flag was evaluated as a prerequisite, this is the key of the flag that referenced it
     * @param trackEvents true if full event tracking is turned on for this flag
     * @param debugEventsUntilDate if non-null, the time until which event debugging should be enabled
     * @param debug true if this is a debugging event
     * @deprecated
     */
    @Deprecated
    public FeatureRequest(long timestamp, String key, LDUser user, Integer version, Integer variation, JsonElement value,
        JsonElement defaultVal, String prereqOf, boolean trackEvents, Long debugEventsUntilDate, EvaluationReason reason, boolean debug) {
      this(timestamp, key, user, version, variation, LDValue.unsafeFromJsonElement(value), LDValue.unsafeFromJsonElement(defaultVal),
          reason, prereqOf, trackEvents, debugEventsUntilDate, debug);
    }
  }

}